/**
 * @file    YTM32B1MD1_flash.ld
 * @version V1.3.2
 *
 * @brief   Linker file for the GNU C Compiler
 *
 * @addtogroup LINKER_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : YTM32B1MD with 64 KB SRAM
 *   Peripheral           : 
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V1.3.2
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd. 
 *   All Rights Reserved.
==================================================================================================*/
/*
 * System Memory Map:
 * 0x00000000 0x0007FFFF 512KB PFlash
 * 0x1FFF8000 0x1FFFFFFF 32KB  SRAM_L
 * 0x20000000 0x20007FFF 32KB  SRAM_U
 */

/*                             SRAM Memory Map
 *  ------------------------------------------------------------------
 *  |         Address         |  Size   |         Description        |
 *  ------------------------------------------------------------------
 *  | 0x1FFF8000 - 0x1FFF83F7 | 1KB-8B  | Interrupt Vector Table     |
 *  | 0x1FFF83F8 - 0x1FFF83FF | 8B      | WdgTst Data                |
 *  | 0x1FFF8400 - 0x200077FF | 61KB    | (MCAL) Data                |
 *  | 0x20007800 - 0x20007BFF | 1KB     | RamTst Data                |
 *  | 0x20007C00 - 0x20007FFF | 1KB     | Stack                      |
 */

/* Entry Point */
ENTRY(Reset_Handler)

__HeapSize  = DEFINED(HEAP_SIZE)  ? HEAP_SIZE  : 0;
__StackSize = DEFINED(STACK_SIZE) ? STACK_SIZE : 1K;
__WdgTstSize = DEFINED(SAFLIB_WDGTST_EN) ? 8 : 0;
__RamTstSize = DEFINED(SAFLIB_RAMTST_EN) ? __StackSize : 0;

/* If symbol __flash_vector_table__=1 is defined at link time
 * the interrupt vector will not be copied to RAM.
 * Warning: Using the interrupt vector from Flash will not allow
 * INT_SYS_InstallHandler because the section is Read Only.
 */
M_VECTOR_RAM_SIZE = DEFINED(__flash_vector_table__) ? 0x0 : 1K;

/* Specify the memory areas */
MEMORY
{
    /* Flash */
    m_interrupts    (RX) : ORIGIN = 0x00000000, LENGTH = 1K
    m_text          (RX) : ORIGIN = 0x00000400, LENGTH = 511K

    /* SRAM_L */
    m_custom        (RW) : ORIGIN = 0x1FFF8000, LENGTH = 0K
    /* SRAM_U */
    m_vec           (RW) : ORIGIN = 0x1FFF8000, LENGTH = 1K - __WdgTstSize
    m_wdgtst        (RW) : ORIGIN = 0x1FFF83F8, LENGTH = __WdgTstSize
    m_data          (RW) : ORIGIN = 0x1FFF8400, LENGTH = 62K - __RamTstSize
    m_ramtst        (RW) : ORIGIN = 0x20008000 - __StackSize - __RamTstSize, LENGTH = __RamTstSize
    m_stack         (RW) : ORIGIN = 0x20008000 - __StackSize, LENGTH = __StackSize
}

/* Define output sections */
SECTIONS
{
    /* The startup code goes first into internal flash */
    .interrupts :
    {
        __VECTOR_ROM = .;
        __VECTOR_TABLE = .;
        __interrupts_start__ = .;
        . = ALIGN(4);
        KEEP(*(.isr_vector))     /* Startup code */
        __interrupts_end__ = .;
        . = ALIGN(4);
    } > m_interrupts

    /* The program code and other data goes into internal flash */
    .text :
    {
        . = ALIGN(4);
        *(.text)                 /* .text sections (code) */
        *(.text*)                /* .text* sections (code) */
        *(.rodata)               /* .rodata sections (constants, strings, etc.) */
        *(.rodata*)              /* .rodata* sections (constants, strings, etc.) */
        *(.init)                 /* section used in crti.o files */
        *(.fini)                 /* section used in crti.o files */
        *(.eh_frame)             /* section used in crtbegin.o files */
        . = ALIGN(4);
        *(.mcal_text)
        . = ALIGN(4);
        *(.mcal_const_cfg)
        . = ALIGN(4);
        *(.mcal_const)
        . = ALIGN(4);
        *(.mcal_const_no_cacheable)
        . = ALIGN(4);
        *(.cortst_code)         /* .cortst_code sections */
        *(.cortst_branch_code0) /* .cortst_branch_code0 sections */
        *(.cortst_branch_code1) /* .cortst_branch_code1 sections */
        *(.cortst_shell_code)   /* .cortst_shell_code sections */
        *(.cortst_vector_table) /* .cortst_vector_table sections */
        *(.cortst_rom_data)     /* .cortst_rom_data sections */        
        . = ALIGN(4);
        __init_table = .;
        KEEP(*(.init_table))  
        . = ALIGN(4);
        __zero_table = .;
        KEEP(*(.zero_table))
        . = ALIGN(4);        
    } > m_text

    __INIT_TABLE = __init_table;
    __ZERO_TABLE = __zero_table;    

    /* Section used by the libgcc.a library for fvp4 */
    .ARM :
    {
        __exidx_start = .;
        *(.ARM.exidx*)
        __exidx_end = .;
    } > m_text

    __etext = .;    /* Define a global symbol at end of code. */
    __DATA_ROM = .; /* Symbol is used by startup for data initialization. */

    .interrupts_ram :
    {
        . = ALIGN(4);
        __VECTOR_RAM__ = .;
        __interrupts_ram_start__ = .; /* Create a global symbol at data start. */
        *(.m_interrupts_ram)          /* This is a user defined section. */
        . += (__interrupts_end__ - __interrupts_start__);
        . = ALIGN(4);
        __interrupts_ram_end__ = .;   /* Define a global symbol at data end. */
    } > m_vec

    __VECTOR_RAM = DEFINED(__flash_vector_table__) ? ORIGIN(m_interrupts) : __VECTOR_RAM__ ;
    __RAM_VECTOR_TABLE_SIZE = DEFINED(__flash_vector_table__) ? 0x0 : (__interrupts_ram_end__ - __interrupts_ram_start__) ;
    __VECTOR_END = __interrupts_end__;

     .wdgtst_ram (NOLOAD):
    {
        __WDGTST_RAM_START = .;
        __wdgtst_start__ = .;
        . = ALIGN(4);
        *(.bss.wdgtst_data)
        . = ALIGN(4);
        __wdgtst_end__ = .;
    } > m_wdgtst

    __WDGTST_RAM_END = ORIGIN(m_wdgtst) + LENGTH(m_wdgtst);

    .data : AT(__DATA_ROM)
    {
        . = ALIGN(4);
        __DATA_RAM = .;
        __RAM_START = .;
        __data_start__ = .;      /* Create a global symbol at data start. */
        *(.data)                 /* .data sections */
        *(.data*)                /* .data* sections */
        . = ALIGN(4);
        *(.mcal_data)  
		. = ALIGN(4);
        *(.mcal_data_no_cacheable)     
        . = ALIGN(4);
        *(.cortst_ram_data)         /* .cortst_ram_data sections */
        *(.cortst_ram_data_target0) /* .cortst_ram_data_target0 sections */
        *(.cortst_ram_data_target1) /* .cortst_ram_data_target1 sections */
        *(.cortst_ram_test_code)    /* .cortst_ram_test_code sections */
        *(.cortst_shell_data)       /* .cortst_shell_data sections */
        . = ALIGN(4);
        __data_end__ = .;        /* Define a global symbol at data end. */
    } > m_data

    __DATA_END = __DATA_ROM + (__data_end__ - __data_start__);
    __CODE_ROM = __DATA_END; /* Symbol is used by code initialization. */

    .code : AT(__CODE_ROM)
    {
        . = ALIGN(4);
        __CODE_RAM = .;
        __code_start__ = .;      /* Create a global symbol at code start. */
        __code_ram_start__ = .;
        *(.ramcode)             /* Custom section for storing code in RAM */
        . = ALIGN(4);
        __code_end__ = .;        /* Define a global symbol at code end. */
        __code_ram_end__ = .;
    } > m_data

    __CODE_END = __CODE_ROM + (__code_end__ - __code_start__);
    __CUSTOM_ROM = __CODE_END;

    /* Custom Section Block that can be used to place data at absolute address. */
    /* Use __attribute__((section (".customSection"))) to place data here. */
    /* Use this section only when MTB (Micro Trace Buffer) is not used, because MTB uses the same RAM area, as described in YTM32B Reference Manual. */
    .customSectionBlock  ORIGIN(m_custom) : AT(__CUSTOM_ROM)
    {
        __customSection_start__ = .;
        KEEP(*(.customSection))  /* Keep section even if not referenced. */
        __customSection_end__ = .;
    } > m_custom

    __CUSTOM_RAM = __customSection_start__;
    __CUSTOM_END = __CUSTOM_ROM + (__customSection_end__ - __customSection_start__);
    __rom_end    = __CUSTOM_END;
    __PREINIT_ROM = __CUSTOM_END;

    .preinit_array : AT(__PREINIT_ROM)
    {     
        __preinit_array_start = .;     
        KEEP (*(.preinit_array))    /* section used in libc.a */     
        __preinit_array_end = .;   
    }> m_data

    __PREINIT_RAM = __preinit_array_start;
    __PREINIT_END = __PREINIT_ROM + (__preinit_array_end - __preinit_array_start);
    __INIT_ROM = __PREINIT_END;

    .init_array : AT(__INIT_ROM)
    {
        __init_array_start = .;     
        KEEP (*(SORT(.init_array.*)))  /* section used in crtbegin.o files */     
        KEEP (*(.init_array))          /* section used in crtbegin.o files */     
        KEEP (*(SORT(.fini_array.*)))  /* section used in crtbegin.o files */     
        KEEP (*(.fini_array))          /* section used in crtbegin.o files */     
        __init_array_end = .;   
    }> m_data

    __INIT_RAM = __init_array_start;
    __INIT_END = __INIT_ROM + (__init_array_end - __init_array_start);

    /* Uninitialized data section. */
    .bss (NOLOAD) :
    {
        /* This is used by the startup in order to initialize the .bss section. */
        . = ALIGN(4);
        __BSS_START = .;
        __bss_start__ = .;
        *(.bss)
        *(.bss*)
        *(COMMON)
        . = ALIGN(16);
        *(.mcal_bss)
		. = ALIGN(16);
        __non_cacheable_bss_start = .;
        *(.mcal_bss_no_cacheable)      
        . = ALIGN(4);
        __non_cacheable_bss_end = .; 
        __bss_end__ = .;
        __BSS_END = .;
    } > m_data

    /* Put heap section after the program data */
    .heap :
    {
        . = ALIGN(8);
        __end__ = .;
        __heap_start__ = .;
        PROVIDE(end = .);
        PROVIDE(_end = .);
        PROVIDE(__end = .);
        __HeapBase = .;
        . += __HeapSize;
        __HeapLimit = .;
        __heap_limit = .;
        __heap_end__ = .;
    } > m_data

     .ramtst_ram (NOLOAD):
    {
        __RAMTST_RAM_START = .;
        __ramtst_start__ = .;
        *(.ramtst_data)
        __ramtst_end__ = .;
    } > m_ramtst

    /* Initializes stack on the end of block */
    __StackTop   = ORIGIN(m_stack) + LENGTH(m_stack);
    __StackLimit = __StackTop - __StackSize;
    PROVIDE(__stack = __StackTop);
    __RAM_END = __StackTop - 1;

    .stack __StackLimit :
    {
        . = ALIGN(8);
        __stack_start__ = .;
        . += __StackSize;
        __stack_end__ = .;
    } > m_stack

    .ARM.attributes 0 : { *(.ARM.attributes) }
    
    /* Memory validation */
    ASSERT(__rom_end <= (ORIGIN(m_text) + LENGTH(m_text)), "Region m_text overflowed!")

    ASSERT(__StackLimit >= __HeapLimit, "region m_data overflowed with stack and heap")
}
