/**
 * @file    YTM32B1HA0_flash.ld
 * @version V1.3.2
 *
 * @brief   Linker file for the GNU C Compiler
 *
 * @addtogroup LINKER_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : YTM32B1HA0 with 256 KB SRAM
 *   Peripheral           : none
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V1.3.2
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd. 
 *   All Rights Reserved.
==================================================================================================*/
/*
 * System Memory Map
 * 0x00000000 0x00007FFF 32KB   ITCM
 * 0x02000000 0x021FFFFF 2MB    PFlash0-1
 * 0x06000000 0x0603FFFF 256KB  DFlash
 * 0x10000000 0x10003FFF 16KB   NVR0
 * 0x10020000 0x10023FFF 16KB   NVR1
 * 0x20000000 0x2001FFFF 128KB  DTCM
 * 0x20020000 0x2005FFFF 256KB  OCRAM0-1
 */

/*
 * Application Memory Map
 * 0x00000000 0x00007FFF 32KB   ITCM      APP performance-sensitive code 
 * 0x02000000 0x020007FF 2KB    PFlash0   MCAL boot vector table
 * 0x02000800 0x02000BFF 1KB    PFlash0   MCAL ISR vector table stored in PFlash
 * 0x02000C00 0x0200BFFF 45KB   PFlash0   MCAL&APP code and data non-cacheable
 * 0x0200C000 0x021FFFFF 2000KB PFlash0-1 MCAL&APP code and data
 * 0x06000000 0x0603FFFF 256KB  DFlash    MCAL&APP FEE (Flash EEPROM Emulation)
 * 0x10000000 0x10003FFF 16KB   NVR0      CUS_NVR, HCU_NVR, USER_NVR, SHA_NVR, BOOT_NVR
 * 0x10020000 0x10023FFF 16KB   NVR1      OTP_NVR
 * 0x20000000 0x200003FF 1KB    DTCM      MCAL ISR Vector Table running in DTCM
 * 0x20000400 0x2001EFFF 123KB  DTCM      APP performance-sensitive data
 * 0x2001F000 0x2001FFFF 4KB    DTCM      MCAL&APP stack
 * 0x20020000 0x20027FFF 32KB   OCRAM0    APP standby code and data
 * 0x20028000 0x2002DFFF 24KB   OCRAM0    MCAL&APP code and data non-cacheable
 * 0x2002E000 0x2005FFFF 200KB  OCRAM0-1  MCAL&APP code and data
 */

/* Entry Point */
ENTRY(Reset_Handler)

__HeapSize  = DEFINED(HEAP_SIZE)  ? HEAP_SIZE  : 0;
__StackSize = DEFINED(STACK_SIZE) ? STACK_SIZE : 4K;

/* Specify the memory areas */
MEMORY
{
    /* ITCM */
    m_itcm      (RW) : ORIGIN = 0x00000000, LENGTH = 32K

    /* PFlash0-1 */
    m_bvt       (RX) : ORIGIN = 0x02000000, LENGTH = 2K
    m_ivt       (RX) : ORIGIN = 0x02000800, LENGTH = 1K
    m_nc_text   (RX) : ORIGIN = 0x02000C00, LENGTH = 45K
    m_text      (RX) : ORIGIN = 0x0200C000, LENGTH = 2000K

    /* DFlash */
    m_dflash    (RW) : ORIGIN = 0x06000000, LENGTH = 256K

    /* NVR0 */
    m_nvr0      (RW) : ORIGIN = 0x10000000, LENGTH = 16K
    /* NVR1 */
    m_nvr1      (RW) : ORIGIN = 0x10020000, LENGTH = 16K   

    /* DTCM */
    m_dtcm_ivt  (RW) : ORIGIN = 0x20000000, LENGTH = 1K
    m_dtcm      (RW) : ORIGIN = 0x20000400, LENGTH = 123K
    m_stack     (RW) : ORIGIN = 0x2001F000, LENGTH = 4K

    /* OCRAM0-1 */
    m_standby   (RW) : ORIGIN = 0x20020000, LENGTH = 32K
    m_nc_data   (RW) : ORIGIN = 0x20028000, LENGTH = 24K    
    m_data      (RW) : ORIGIN = 0x2002E000, LENGTH = 200K
}

/* Define output sections */
SECTIONS
{
    /* The boot vector table goes first into internal flash */
    .bvt :
    {
        . = ALIGN(2048);
        __bvt_start__ = .;
        KEEP(*(.bvt_header))        /* Boot Vector Table */

        . = ALIGN(0x40);
        __secure_boot_config_group_start__ = .;
        KEEP(*(.sb_config_group))   /* secure boot config group */
        __secure_boot_config_group_end__ = .;

        . = ALIGN(0x40);
        __secure_boot_config_section_start__ = .;
        KEEP(*(.sb_config_section)) /* secure boot config section */
        __secure_boot_config_section_end__ = .;

        . = ALIGN(0x10);
        __secure_boot_cmac_start__ = .;
        KEEP(*(.sb_cmac))           /* CMAC result of the secure boot config section */
        __secure_boot_cmac_end__ = .;
        __bvt_end__ = .;
    } > m_bvt
    
    /* The initial ISR vector table stored in internal flash */
    .interrupts :
    {
        . = ALIGN(1024);
        __VECTOR_ROM_START = .;
        __VECTOR_TABLE = .;
        __interrupts_init_start = .;
        KEEP(*(.isr_vector))     /* Startup code */
        . = ALIGN(4);
        __interrupts_init_end = .;
        __VECTOR_ROM_END = .;
    } > m_ivt

    .nc_text :
    {
        . = ALIGN(4);
        __nc_text_start__ = .;
        *(.mcal_const_no_cacheable)
        . = ALIGN(4);
        __nc_text_end__ = .;
    } > m_nc_text

    /* The program code and other data goes into internal flash */
    .text :
    {
        . = ALIGN(4);
        *(.text)                 /* .text sections (code) */
        *(.text*)                /* .text* sections (code) */
        *(.rodata)               /* .rodata sections (constants, strings, etc.) */
        *(.rodata*)              /* .rodata* sections (constants, strings, etc.) */
        *(.init)                 /* section used in crti.o files */
        *(.fini)                 /* section used in crti.o files */
        *(.eh_frame)             /* section used in crtbegin.o files */
        . = ALIGN(4);
        *(.mcal_text)
        . = ALIGN(4);
        *(.mcal_const_cfg)
        . = ALIGN(4);
        *(.mcal_const)
        . = ALIGN(4);
        __INIT_TABLE = .;
        __init_table = .;
        KEEP(*(.init_table))  
        . = ALIGN(4);
        __ZERO_TABLE = .;
        __zero_table = .;
        KEEP(*(.zero_table))
        . = ALIGN(4);       
    } > m_text

    /* Section used by the libgcc.a library for fvp4 */
    .ARM :
    {
        __exidx_start = .;
        *(.ARM.exidx*)
        __exidx_end = .;
    } > m_text

    . = ALIGN(4);
    __text_end = .; /* Symbol is used by startup for data initialization. */

    __ITCM_ROM_START = __text_end;

   .itcm_text : AT(__ITCM_ROM_START)
    {
        . = ALIGN(4);
        __ITCM_RAM_START = .;
        __itcm_start__ = .;
        *(.itcm_text)
        . = ALIGN(4);
        *(.ramcode)
        . = ALIGN(4);
        __itcm_end__ = .;
    } > m_itcm

    __ITCM_START = ORIGIN(m_itcm);
    __ITCM_END = ORIGIN(m_itcm) + LENGTH(m_itcm);
    __ITCM_ROM_END = __ITCM_ROM_START + (__itcm_end__ - __itcm_start__);

    .interrupts_ram :
    {
        . = ALIGN(1024);
        __VECTOR_RAM = .;
        __VECTOR_RAM_START = .;
        __interrupts_ram_start__ = .; /* Create a global symbol at data start. */
        . += (__interrupts_init_end - __interrupts_init_start);
        . = ALIGN(4);
        __interrupts_ram_end__ = .;   /* Define a global symbol at data end. */
    } > m_dtcm_ivt

    .dtcm_bss (NOLOAD) :
    {
        . = ALIGN(4);
        __DTCM_BSS_START = .;
        __dtcm_bss_start__ = .;
        *(.dtcm_bss)
        . = ALIGN(4);
        __dtcm_bss_end__ = .;
        __DTCM_BSS_END = .;
    } > m_dtcm

    __DTCM_ROM_START = __ITCM_ROM_END;

    .dtcm_data : AT(__DTCM_ROM_START)
    {
        . = ALIGN(4);
        __DTCM_RAM_START = .;
        __dtcm_data_start__ = .;
        *(.dtcm_data)
        . = ALIGN(4);
        __dtcm_data_end__ = .;        
    } > m_dtcm

    __DTCM_START = ORIGIN(m_dtcm_ivt);
    __DTCM_END = ORIGIN(m_stack) + LENGTH(m_stack);
    __DTCM_ROM_END = __DTCM_ROM_START + (__dtcm_data_end__ - __dtcm_data_start__);

    .standby_bss (NOLOAD) :
    {
        . = ALIGN(8);
        __standby_bss_start__ = .;
        *(.standby_bss)
        . = ALIGN(8);
        __standby_bss_end__ = .;
    } > m_standby

    __STANDBY_OCRAM_START = ORIGIN(m_standby);
    __STANDBY_OCRAM_END = ORIGIN(m_standby) + LENGTH(m_standby);

    .nc_bss (NOLOAD) :
    {
		. = ALIGN(16);
        __NC_BSS_START = .;
        __nc_bss_start = .;
        *(.mcal_bss_no_cacheable)      
        . = ALIGN(4);
        __nc_bss_end = .;
        __NC_BSS_END = .;       
    } > m_nc_data

    __NC_DATA_ROM_START = __DTCM_ROM_END;
    .nc_data : AT(__NC_DATA_ROM_START)
    {
        . = ALIGN(4);
        __RAM_START = .;
        __NC_DATA_RAM_START = .;
        __nc_data_start__ = .;
        *(.mcal_data_no_cacheable)
        . = ALIGN(4);
        __nc_data_end__ = .;
    } > m_nc_data
    __NC_DATA_ROM_END = __NC_DATA_ROM_START + (__nc_data_end__ - __nc_data_start__);

    /* Uninitialized data section. */
    .bss (NOLOAD) :
    {
        /* This is used by the startup in order to initialize the .bss section. */
        . = ALIGN(16);
        __BSS_START = .;
        __bss_start__ = .;
        *(.bss)
        *(.bss*)
        *(COMMON)
        . = ALIGN(16);
        *(.mcal_bss)
		. = ALIGN(4);
        __bss_end__ = .;
        __BSS_END = .;
    } > m_data

    __DATA_ROM_START = __NC_DATA_ROM_END;
    .data : AT(__DATA_ROM_START)
    {
        . = ALIGN(4);
        __DATA_RAM_START = .;
        __data_start__ = .;      /* Create a global symbol at data start. */       
        *(.data)                 /* .data sections */
        *(.data*)                /* .data* sections */
        . = ALIGN(4);
        *(.mcal_data)  
		. = ALIGN(4);
        __data_end__ = .;        /* Define a global symbol at data end. */
    } > m_data

    __DATA_ROM_END = __DATA_ROM_START + (__data_end__ - __data_start__);

    __ARRAY_ROM_START = __DATA_ROM_END;
    .array : AT(__ARRAY_ROM_START)
    {
        . = ALIGN(4);
        __ARRAY_RAM_START = .;
        __array_ram_start__ = .;    /* Create a global symbol at array start. */
        /* preinit data */
        PROVIDE_HIDDEN (__preinit_array_start = .);
        KEEP(*(.preinit_array))
        PROVIDE_HIDDEN (__preinit_array_end = .);

        . = ALIGN(4);
        /* init data */
        PROVIDE_HIDDEN (__init_array_start = .);
        KEEP(*(SORT(.init_array.*)))
        KEEP(*(.init_array))
        PROVIDE_HIDDEN (__init_array_end = .);

        . = ALIGN(4);
        /* finit data */
        PROVIDE_HIDDEN (__fini_array_start = .);
        KEEP(*(SORT(.fini_array.*)))
        KEEP(*(.fini_array))
        PROVIDE_HIDDEN (__fini_array_end = .);
        . = ALIGN(4);
        __array_ram_end__ = .;        /* Define a global symbol at array end. */
    } > m_data

    __OCRAM_START = ORIGIN(m_nc_data);
    __OCRAM_END = ORIGIN(m_data) + LENGTH(m_data);
    __ARRAY_ROM_END = __ARRAY_ROM_START + (__array_ram_end__ - __array_ram_start__);

    /* Put heap section after the program data */
    .heap :
    {
        . = ALIGN(8);
        __end__ = .;
        __heap_start__ = .;
        PROVIDE(end = .);
        PROVIDE(_end = .);
        PROVIDE(__end = .);
        __HeapBase = .;
        . += __HeapSize;
        __HeapLimit = .;
        __heap_limit = .;
        __heap_end__ = .;
    } > m_data

    /* Initializes stack on the end of block */
    __StackTop   = ORIGIN(m_stack) + LENGTH(m_stack);
    __StackLimit = __StackTop - __StackSize;
    PROVIDE(__stack = __StackTop);

    .stack __StackLimit :
    {
        . = ALIGN(8);
        __stack_start__ = .;
        . += __StackSize;
        __stack_end__ = .;
    } > m_stack

    .ARM.attributes 0 : { *(.ARM.attributes) }

    /* Memory validation */
    ASSERT(__ARRAY_ROM_END <= (ORIGIN(m_text) + LENGTH(m_text)), "Region m_text overflowed!")

    ASSERT(__array_ram_end__ <= __OCRAM_END, "region m_data overflowed!")
}
