/**
 * @file    Mpu_Lld_M7.C
 * @version V1.0.0
 *
 * @brief   YUNTU Mpu_Lld_M7 module interface
 * @details API implementation for Mpu_Lld_M7 driver
 *
 * @addtogroup MPU_LLD_M7_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Cortex M7 MPU
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V1.0.0
 *
 *   (c) Copyright 2020-2024 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*=================================================================================================
 *                                         INCLUDE FILES
=================================================================================================*/
#include "Mpu_Lld_M7.h"

/*==================================================================================================
 *                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define MPU_LLD_M7_VENDOR_ID_C                      (180)
#define MPU_LLD_M7_AR_RELEASE_MAJOR_VERSION_C       (4)
#define MPU_LLD_M7_AR_RELEASE_MINOR_VERSION_C       (4)
#define MPU_LLD_M7_AR_RELEASE_REVISION_VERSION_C    (0)
#define MPU_LLD_M7_SW_MAJOR_VERSION_C               (1)
#define MPU_LLD_M7_SW_MINOR_VERSION_C               (0)
#define MPU_LLD_M7_SW_PATCH_VERSION_C               (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if Mpu_Lld_M7.c and Mpu_Lld_M7.h are of the same vendor */
#if (MPU_LLD_M7_VENDOR_ID_C != MPU_LLD_M7_VENDOR_ID)
#error "Mpu_Lld_M7.c and Mpu_Lld_M7.h have different vendor ids"
#endif

/* Check if Mpu_Lld_M7.c and Mpu_Lld_M7.h are of the same Autosar version */
#if ((MPU_LLD_M7_AR_RELEASE_MAJOR_VERSION_C != MPU_LLD_M7_AR_RELEASE_MAJOR_VERSION) || \
     (MPU_LLD_M7_AR_RELEASE_MINOR_VERSION_C != MPU_LLD_M7_AR_RELEASE_MINOR_VERSION) || \
     (MPU_LLD_M7_AR_RELEASE_REVISION_VERSION_C != MPU_LLD_M7_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of Mpu_Lld_M7.c and Mpu_Lld_M7.h are different"
#endif

/* Check if Mpu_Lld_M7.c and Mpu_Lld_M7.h are of the same Software version */
#if ((MPU_LLD_M7_SW_MAJOR_VERSION_C != MPU_LLD_M7_SW_MAJOR_VERSION) || \
     (MPU_LLD_M7_SW_MINOR_VERSION_C != MPU_LLD_M7_SW_MINOR_VERSION) || \
     (MPU_LLD_M7_SW_PATCH_VERSION_C != MPU_LLD_M7_SW_PATCH_VERSION))
#error "Software Version Numbers of Mpu_Lld_M7.c and Mpu_Lld_M7.h are different"
#endif

#define PLATFORM_START_SEC_CODE
#include "Platform_MemMap.h"    /* NOSONAR, memory map */
/*==================================================================================================
 *                                         GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief       This function initializes the Cortex M7 MPU.
 * @param[in]   MpuConfig The pointer to the MPU configuration structure.
 * @return      void
 */
PLATFORM_FUNC void Mpu_Lld_M7_Init(const Mpu_Lld_M7_ConfigType *MpuConfig)
{
    uint8 Index;
    uint8 MpuRegionCount = (uint8)(MPU->TYPE >> MPU_TYPE_DREGION_Pos);
    uint32 MpuEnableMask = 0U;

    /* Disable MPU */
    ARM_MPU_Disable();
    /* Disable all regions. */
    for (Index = 0U; Index < MpuRegionCount; ++Index)
    {
        ARM_MPU_ClrRegion(Index);
    }
    /* Load the region configuration */
    for (Index = 0U; Index < MpuConfig->RegionCount; ++Index)
    {
        ARM_MPU_SetRegionEx
        (
            MpuConfig->RegionConfig[Index].RegionIndex,
            MpuConfig->RegionConfig[Index].Region.RBAR,
            MpuConfig->RegionConfig[Index].Region.RASR
        );
    }
    /* Enable MPU with MemManage fault. */
    if (TRUE == MpuConfig->DefaultMapEn)
    {
        MpuEnableMask |= MPU_CTRL_PRIVDEFENA_Msk;
    }
    if (TRUE == MpuConfig->RunHFNMIEn)
    {
        MpuEnableMask |= MPU_CTRL_HFNMIENA_Msk;
    }
    /* Enable MPU */
    ARM_MPU_Enable(MpuEnableMask);
}

#define PLATFORM_STOP_SEC_CODE
#include "Platform_MemMap.h"    /* NOSONAR, memory map */

#ifdef __cplusplus
}
#endif

/** @} */

