/*
* @file    Intm_Lld.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Intm_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.3.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef INTM_LLD_H
#define INTM_LLD_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES                                          
==================================================================================================*/

#include "Intm_Lld_Cfg.h"
#include "Devassert.h"

/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION                                
==================================================================================================*/
#define INTM_LLD_VENDOR_ID                      (180)
#define INTM_LLD_AR_RELEASE_MAJOR_VERSION       (4)
#define INTM_LLD_AR_RELEASE_MINOR_VERSION       (4)
#define INTM_LLD_AR_RELEASE_REVISION_VERSION    (0)
#define INTM_LLD_SW_MAJOR_VERSION               (2)
#define INTM_LLD_SW_MINOR_VERSION               (3)
#define INTM_LLD_SW_PATCH_VERSION               (0)
/*==================================================================================================
                                                DEFINES AND MACROS                                       
==================================================================================================*/
/*==================================================================================================
                                                EXTERNAL CONSTANTS                                        
==================================================================================================*/
/*==================================================================================================
                                                ENUMS                                        
==================================================================================================*/
/*==================================================================================================
                                                FUNCTION PROTOTYPES                                           
==================================================================================================*/

/**
 * @brief         Enables interrupt monitoring feature.
 *
 * @details       This function is non-reentrant and enables the interrupt monitors.
 *
 * @return        void
 *
 * @api
 *
 * @implements    Intm_Lld_EnableMonitor_Activity
 *
 * */
void Intm_Lld_EnableMonitor(void);

/**
 * @brief         Disables interrupt monitoring feature.
 *
 * @details       This function is non-reentrant and disables the interrupt monitors.
 *
 * @return        void
 *
 * @api
 *
 * @implements    Intm_Lld_DisableMonitor_Activity
 *
 * */
void Intm_Lld_DisableMonitor(void);

/**
 * @brief         Acknowledges a monitored interrupt.
 *
 * @details       This function is reentrant; it acknowledges that a monitored interrupt
 *                has been been served.
 *
 * @param[in]     IrqNumber: The interrupt vector acknowledged.
 * @return void
 *
 * @api
 *
 * @implements    Intm_Lld_AckIrq_Activity
 *
 * */
void Intm_Lld_AckIrq(IRQn_Type IrqNumber);

/**
 * @brief         Selects an interrupt to monitor.
 *
 * @details       This function is reentrant; it selects which interrupt is monitored on
 *                a specific interrupt monitor.
 *
 * @param[in]     MonitorIdx: The index of the interrupt monitor used.
 * @param[in]     IrqNumber: The interrupt vector to be monitored.
 * @return void
 *
 * @api
 *
 * @implements    Intm_Lld_SelectMonitoredIrq_Activity
 *
 * */
void Intm_Lld_SelectMonitoredIrq(Intm_Lld_MonitorType MonitorIdx, IRQn_Type IrqNumber);

/**
 * @brief         Sets the latency for a monitored interrupt.
 *
 * @details       This function is reentrant; it sets the accepted latency for the
 *                monitored interrupt.
 *
 * @param[in]     MonitorIdx: The index of the interrupt monitor used.
 * @param[in]     Latency: The accepted latency for the monitored interrupt.
 * @return void
 *
 * @api
 *
 * @implements    Intm_Lld_SetLatency_Activity
 *
 * */
void Intm_Lld_SetLatency(Intm_Lld_MonitorType MonitorIdx, uint32 Latency);

/**
 * @brief         Resets the timer for an interrupt monitor.
 *
 * @details       This function is reentrant; it resets the timer for the interrupt monitor.
 *
 * @param[in]     MonitorIdx: The index of the interrupt monitor used.
 * @return void
 *
 * @api
 *
 * @implements    Intm_Lld_ResetTimer_Activity
 *
 * */
void Intm_Lld_ResetTimer(Intm_Lld_MonitorType MonitorIdx);

/**
 * @brief         Returns the status of an interrupt monitor.
 *
 * @details       This function is reentrant; it returns the status of an interrupt monitor:
 *                0 - latency not exceeded; 1 - timer exceeded latency.
 *
 * @param[in]     MonitorIdx: The index of the interrupt monitor used.
 * @return        uint8: 0 - latency not exceeded; 1 - timer exceeded latency.
 *
 * @api
 *
 * @implements    Intm_Lld_GetStatus_Activity
 *
 * */
 uint8 Intm_Lld_GetStatus(Intm_Lld_MonitorType MonitorIdx);

#ifdef __cplusplus
}
#endif

#endif /* End of file Intm_Lld.h */

