/**
*   @file    pSIP_Ppu.h
*   @version
*
*   @brief   AUTOSAR Ppu register map
*   @details Register map for PPU
*
*   @addtogroup ADC_MODULE
*   @{
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Ppu
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.3.0
*
*   (c) Copyright 2020-2025 YTMicro Semiconductor, Inc.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef PSIP_PPU_H
#define PSIP_PPU_H

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/

#include "Std_Types.h"

/*==================================================================================================
*                              PPU REGISTER MAP
==================================================================================================*/

/**
 * @addtogroup PPU_Peripheral_Access_Layer PPU Peripheral Access Layer
 * @{
 */

typedef struct
{
    uint32 Normal[1024];                        /**< Normal Register, offset: 0x0000 */
    uint32 RESERVED0[1024];                     /**< Reserved0, offset: 0x1000 */
    uint32 Mirror[1024];                        /**< Mirror Register, offset: 0x2000 */
    uint8  SLOCK[1024];                          /**< Soft Lock Register, offset: 0x3000 */
    uint32 RESERVED1[767];                      /**< Reserved1, offset: 0x3400 */
    uint32 PROTCFG;                             /**< Protect Configuration Register, offset: 0x3FFC */
} PPU_Type, *PPU_MemMapPtr;

/** Number of instances of the PPU module. */
#define PPU_INSTANCE_COUNT                                 (1U)

/** Number of ip module of the PPU*/
#define PPU_IP_MODULE_COUNT                                (8U)

/* PPU TMR0 base address */
#define PPU_TMR0_BASE_ADDR32                                (0x40150000U)
#define PPU_TMR0                                            ((volatile PPU_Type*)(PPU_TMR0_BASE_ADDR32))

/* PPU TMR1 base address */
#define PPU_TMR1_BASE_ADDR32                                (0x40154000U)
#define PPU_TMR1                                            ((volatile PPU_Type*)(PPU_TMR1_BASE_ADDR32))

/** PPU SCU base address*/
#define PPU_SCU_BASE_ADDR32                                (0x401F0000U)
#define PPU_SCU                                            ((volatile PPU_Type*)(PPU_SCU_BASE_ADDR32))

/** PPU PCU base address*/
#define PPU_PCU_BASE_ADDR32                                (0x401F8000U)
#define PPU_PCU                                            ((volatile PPU_Type*)(PPU_PCU_BASE_ADDR32))

/** PPU RCU base address*/
#define PPU_RCU_BASE_ADDR32                                (0x401FC000U)
#define PPU_RCU                                            ((volatile PPU_Type*)(PPU_RCU_BASE_ADDR32))

/** PPU EFM base address*/
#define PPU_EFM_BASE_ADDR32                                (0x40040000U)
#define PPU_EFM                                            ((volatile PPU_Type*)(PPU_EFM_BASE_ADDR32))

/** PPU IPC base address*/
#define PPU_IPC_BASE_ADDR32                                (0x401F4000U)
#define PPU_IPC                                            ((volatile PPU_Type*)(PPU_IPC_BASE_ADDR32))

/** PPU CIM base address*/
#define PPU_CIM_BASE_ADDR32                                (0x401EC000U)
#define PPU_CIM                                            ((volatile PPU_Type*)(PPU_CIM_BASE_ADDR32))

/** Array initializer of PPU peripheral base addresses */
#define PPU_BASE_ADDRS                                     { }
/** Array initializer of PPU peripheral base pointers */
#define PPU_BASE_PTRS                                      {PPU_TMR0, PPU_TMR1, PPU_SCU, PPU_PCU, PPU_RCU, PPU_EFM, PPU_IPC, PPU_CIM}

#define PPU_SLOCK_OFFSET8(x)                              (0x3000U + ((x) * (1U)))     /**< Offset for Soft Lock Register */
#define PPU_PROTCFG_OFFSET32                               (0x3FFCU)                    /**< Offset for Protect Configuration Register */


/**
 * @addtogroup PPU Register Mask
 * @{
 */
/* PPU_SLOCK Register */
#define PPU_SLOCK_WREN_MASK                                (0x10U)
#define PPU_SLOCK_WREN_SHIFT                               (4U)
#define PPU_SLOCK_WREN(x)                                  (((uint32)(((uint32)(x)) << PPU_SLOCK_WREN_SHIFT)) & PPU_SLOCK_WREN_MASK)
#define PPU_SLOCK_LOCKBIT_MASK                             (0x1U)
#define PPU_SLOCK_LOCKBIT_SHIFT                            (0U)
#define PPU_SLOCK_LOCKBIT(x)                               (((uint32)(((uint32)(x)) << PPU_SLOCK_LOCKBIT_SHIFT)) & PPU_SLOCK_LOCKBIT_MASK)
/* PPU_PROTCFG Register */
#define PPU_PROTCFG_UAE_MASK                               (0x100U)
#define PPU_PROTCFG_UAE_SHIFT                              (8U)
#define PPU_PROTCFG_UAE(x)                                 (((uint32)(((uint32)(x)) << PPU_PROTCFG_UAE_SHIFT)) & PPU_PROTCFG_UAE_MASK)
#define PPU_PROTCFG_HLOCK_MASK                             (0x1U)
#define PPU_PROTCFG_HLOCK_SHIFT                            (0U)
#define PPU_PROTCFG_HLOCK(x)                               (((uint32)(((uint32)(x)) << PPU_PROTCFG_HLOCK_SHIFT)) & PPU_PROTCFG_HLOCK_MASK)


/**
 * @}
 */ /* end of group PPU_Register_Masks */


/**
 * @}
 */ /* end of group PPU_Peripheral_Access_Layer */

#endif /* PSIP_PPU_H */

