/**
*   @file    pSIP_Mpwm.h
*   @version
*
*   @brief   AUTOSAR Mpwm register map
*   @details Register map for MPWM
*
*   @addtogroup ADC_MODULE
*   @{
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Adc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.3.0
*
*   (c) Copyright 2020-2025 YTMicro Semiconductor, Inc.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef PSIP_MPWM_H
#define PSIP_MPWM_H

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/

#include "Std_Types.h"


/** Extra features */
#define MPWM_NO_LDOK_REG
#define MPWM_NO_CLDOK_REG

/*==================================================================================================
*                              MPWM REGISTER MAP
==================================================================================================*/

/**
 * @addtogroup MPWM_Peripheral_Access_Layer MPWM Peripheral Access Layer
 * @{
 */


/** MPWM - Size of Registers Arrays */
#define MPWM_CH_COUNT           (16)

/* MPWM Register Layout Typedef */
typedef struct {
    uint32 IOSTS;                              /**< IO Status Register, offset: 0x0000 */
    uint32 STS0;                               /**< Status 0 Register, offset: 0x0004 */
    uint32 STS1;                               /**< Status 1 Register, offset: 0x0008 */
    uint32 INTE0;                              /**< Interrupt Enable 0 Register, offset: 0x000C */
    uint32 INTE1;                              /**< Interrupt Enable 1 Register, offset: 0x0010 */
    uint32 CTRL;                               /**< Control Register, offset: 0x0014 */
    uint32 SWT;                                /**< Software Trigger Register, offset: 0x0018 */
    uint32 INIT;                               /**< Channel Initial Register, offset: 0x001C */
    uint32 POL;                                /**< Channel Polarity Register, offset: 0x0020 */
    uint32 Reserved0[ 7];                      /**< Reserved0, offset: 0x0028 */
    struct {
        uint32 CFG;                                /**< Channel Configuration Register, offset: 0x0040 */
        uint32 CNT;                                /**< Counter Status Register, offset: 0x0044 */
        uint32 CAP;                                /**< Capture Status Register, offset: 0x0048 */
    uint32 Reserved1[ 1];                      /**< Reserved1, offset: 0x004C */
        uint32 PERIOD;                             /**< Period Register, offset: 0x0050 */
        uint32 CMP;                                /**< Compare Register, offset: 0x0054 */
    uint32 Reserved2[ 2];                      /**< Reserved2, offset: 0x0058 */

    } CH[16];

} MPWM_Type, *MPWM_MemMapPtr;

/** Number of instances of the MPWM module. */
#define MPWM_INSTANCE_COUNT                        (3u)

/* MPWM0  base address */
#define MPWM0_BASE_ADDR32                                  (0x40144000U)
#define MPWM0                                              ((volatile MPWM_Type *)(MPWM0_BASE_ADDR32))
/* MPWM1  base address */
#define MPWM1_BASE_ADDR32                                  (0x40148000U)
#define MPWM1                                              ((volatile MPWM_Type *)(MPWM1_BASE_ADDR32))
/* MPWM2  base address */
#define MPWM2_BASE_ADDR32                                  (0x4014C000U)
#define MPWM2                                              ((volatile MPWM_Type *)(MPWM2_BASE_ADDR32))

/** Array initializer of MPWM peripheral base addresses */
#define MPWM_BASE_ADDRS                                    { MPWM0_BASE_ADDR32,  MPWM1_BASE_ADDR32,  MPWM2_BASE_ADDR32 }
/** Array initializer of MPWM peripheral base pointers */
#define MPWM_BASE_PTRS                                     { MPWM0,  MPWM1,  MPWM2 }

#define MPWM_IOSTS_OFFSET32                                (0x0000U)                    /**< Offset for IO Status Register */
#define MPWM_STS0_OFFSET32                                 (0x0004U)                    /**< Offset for Status 0 Register */
#define MPWM_STS1_OFFSET32                                 (0x0008U)                    /**< Offset for Status 1 Register */
#define MPWM_INTE0_OFFSET32                                (0x000CU)                    /**< Offset for Interrupt Enable 0 Register */
#define MPWM_INTE1_OFFSET32                                (0x0010U)                    /**< Offset for Interrupt Enable 1 Register */
#define MPWM_CTRL_OFFSET32                                 (0x0014U)                    /**< Offset for Control Register */
#define MPWM_SWT_OFFSET32                                  (0x0018U)                    /**< Offset for Software Trigger Register */
#define MPWM_INIT_OFFSET32                                 (0x001CU)                    /**< Offset for Channel Initial Register */
#define MPWM_POL_OFFSET32                                  (0x0020U)                    /**< Offset for Channel Polarity Register */
#define MPWM_CH_OFFSET32(x)                                (0x0040U + ((x) * (32U)))    /**< Offset for MPWM 16 channels */

#define MPWM_CH_CFG_OFFSET32(x)                            (0x0040U + ((x) * (32U)))    /**< Offset for Channel Configuration Register */
#define MPWM_CH_CNT_OFFSET32(x)                            (0x0044U + ((x) * (32U)))    /**< Offset for Counter Status Register */
#define MPWM_CH_CAP_OFFSET32(x)                            (0x0048U + ((x) * (32U)))    /**< Offset for Capture Status Register */
#define MPWM_CH_PERIOD_OFFSET32(x)                         (0x0050U + ((x) * (32U)))    /**< Offset for Period Register */
#define MPWM_CH_CMP_OFFSET32(x)                            (0x0054U + ((x) * (32U)))    /**< Offset for Compare Register */



/**
 * @addtogroup MPWM Register Mask
 * @{
 */
/* MPWM_IOSTS Register */
#define MPWM_IOSTS_CH15IO_MASK                              (0x8000U)
#define MPWM_IOSTS_CH15IO_SHIFT                             (15U)
#define MPWM_IOSTS_CH15IO(x)                                (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH15IO_SHIFT)) & MPWM_IOSTS_CH15IO_MASK)
#define MPWM_IOSTS_CH14IO_MASK                              (0x4000U)
#define MPWM_IOSTS_CH14IO_SHIFT                             (14U)
#define MPWM_IOSTS_CH14IO(x)                                (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH14IO_SHIFT)) & MPWM_IOSTS_CH14IO_MASK)
#define MPWM_IOSTS_CH13IO_MASK                              (0x2000U)
#define MPWM_IOSTS_CH13IO_SHIFT                             (13U)
#define MPWM_IOSTS_CH13IO(x)                                (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH13IO_SHIFT)) & MPWM_IOSTS_CH13IO_MASK)
#define MPWM_IOSTS_CH12IO_MASK                              (0x1000U)
#define MPWM_IOSTS_CH12IO_SHIFT                             (12U)
#define MPWM_IOSTS_CH12IO(x)                                (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH12IO_SHIFT)) & MPWM_IOSTS_CH12IO_MASK)
#define MPWM_IOSTS_CH11IO_MASK                              (0x800U)
#define MPWM_IOSTS_CH11IO_SHIFT                             (11U)
#define MPWM_IOSTS_CH11IO(x)                                (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH11IO_SHIFT)) & MPWM_IOSTS_CH11IO_MASK)
#define MPWM_IOSTS_CH10IO_MASK                              (0x400U)
#define MPWM_IOSTS_CH10IO_SHIFT                             (10U)
#define MPWM_IOSTS_CH10IO(x)                                (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH10IO_SHIFT)) & MPWM_IOSTS_CH10IO_MASK)
#define MPWM_IOSTS_CH9IO_MASK                               (0x200U)
#define MPWM_IOSTS_CH9IO_SHIFT                              (9U)
#define MPWM_IOSTS_CH9IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH9IO_SHIFT)) & MPWM_IOSTS_CH9IO_MASK)
#define MPWM_IOSTS_CH8IO_MASK                               (0x100U)
#define MPWM_IOSTS_CH8IO_SHIFT                              (8U)
#define MPWM_IOSTS_CH8IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH8IO_SHIFT)) & MPWM_IOSTS_CH8IO_MASK)
#define MPWM_IOSTS_CH7IO_MASK                               (0x80U)
#define MPWM_IOSTS_CH7IO_SHIFT                              (7U)
#define MPWM_IOSTS_CH7IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH7IO_SHIFT)) & MPWM_IOSTS_CH7IO_MASK)
#define MPWM_IOSTS_CH6IO_MASK                               (0x40U)
#define MPWM_IOSTS_CH6IO_SHIFT                              (6U)
#define MPWM_IOSTS_CH6IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH6IO_SHIFT)) & MPWM_IOSTS_CH6IO_MASK)
#define MPWM_IOSTS_CH5IO_MASK                               (0x20U)
#define MPWM_IOSTS_CH5IO_SHIFT                              (5U)
#define MPWM_IOSTS_CH5IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH5IO_SHIFT)) & MPWM_IOSTS_CH5IO_MASK)
#define MPWM_IOSTS_CH4IO_MASK                               (0x10U)
#define MPWM_IOSTS_CH4IO_SHIFT                              (4U)
#define MPWM_IOSTS_CH4IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH4IO_SHIFT)) & MPWM_IOSTS_CH4IO_MASK)
#define MPWM_IOSTS_CH3IO_MASK                               (0x8U)
#define MPWM_IOSTS_CH3IO_SHIFT                              (3U)
#define MPWM_IOSTS_CH3IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH3IO_SHIFT)) & MPWM_IOSTS_CH3IO_MASK)
#define MPWM_IOSTS_CH2IO_MASK                               (0x4U)
#define MPWM_IOSTS_CH2IO_SHIFT                              (2U)
#define MPWM_IOSTS_CH2IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH2IO_SHIFT)) & MPWM_IOSTS_CH2IO_MASK)
#define MPWM_IOSTS_CH1IO_MASK                               (0x2U)
#define MPWM_IOSTS_CH1IO_SHIFT                              (1U)
#define MPWM_IOSTS_CH1IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH1IO_SHIFT)) & MPWM_IOSTS_CH1IO_MASK)
#define MPWM_IOSTS_CH0IO_MASK                               (0x1U)
#define MPWM_IOSTS_CH0IO_SHIFT                              (0U)
#define MPWM_IOSTS_CH0IO(x)                                 (((uint32)(((uint32)(x)) << MPWM_IOSTS_CH0IO_SHIFT)) & MPWM_IOSTS_CH0IO_MASK)
/* MPWM_STS0 Register */
#define MPWM_STS0_CH15TOF_MASK                              (0x8000U)
#define MPWM_STS0_CH15TOF_SHIFT                             (15U)
#define MPWM_STS0_CH15TOF(x)                                (((uint32)(((uint32)(x)) << MPWM_STS0_CH15TOF_SHIFT)) & MPWM_STS0_CH15TOF_MASK)
#define MPWM_STS0_CH14TOF_MASK                              (0x4000U)
#define MPWM_STS0_CH14TOF_SHIFT                             (14U)
#define MPWM_STS0_CH14TOF(x)                                (((uint32)(((uint32)(x)) << MPWM_STS0_CH14TOF_SHIFT)) & MPWM_STS0_CH14TOF_MASK)
#define MPWM_STS0_CH13TOF_MASK                              (0x2000U)
#define MPWM_STS0_CH13TOF_SHIFT                             (13U)
#define MPWM_STS0_CH13TOF(x)                                (((uint32)(((uint32)(x)) << MPWM_STS0_CH13TOF_SHIFT)) & MPWM_STS0_CH13TOF_MASK)
#define MPWM_STS0_CH12TOF_MASK                              (0x1000U)
#define MPWM_STS0_CH12TOF_SHIFT                             (12U)
#define MPWM_STS0_CH12TOF(x)                                (((uint32)(((uint32)(x)) << MPWM_STS0_CH12TOF_SHIFT)) & MPWM_STS0_CH12TOF_MASK)
#define MPWM_STS0_CH11TOF_MASK                              (0x800U)
#define MPWM_STS0_CH11TOF_SHIFT                             (11U)
#define MPWM_STS0_CH11TOF(x)                                (((uint32)(((uint32)(x)) << MPWM_STS0_CH11TOF_SHIFT)) & MPWM_STS0_CH11TOF_MASK)
#define MPWM_STS0_CH10TOF_MASK                              (0x400U)
#define MPWM_STS0_CH10TOF_SHIFT                             (10U)
#define MPWM_STS0_CH10TOF(x)                                (((uint32)(((uint32)(x)) << MPWM_STS0_CH10TOF_SHIFT)) & MPWM_STS0_CH10TOF_MASK)
#define MPWM_STS0_CH9TOF_MASK                               (0x200U)
#define MPWM_STS0_CH9TOF_SHIFT                              (9U)
#define MPWM_STS0_CH9TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH9TOF_SHIFT)) & MPWM_STS0_CH9TOF_MASK)
#define MPWM_STS0_CH8TOF_MASK                               (0x100U)
#define MPWM_STS0_CH8TOF_SHIFT                              (8U)
#define MPWM_STS0_CH8TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH8TOF_SHIFT)) & MPWM_STS0_CH8TOF_MASK)
#define MPWM_STS0_CH7TOF_MASK                               (0x80U)
#define MPWM_STS0_CH7TOF_SHIFT                              (7U)
#define MPWM_STS0_CH7TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH7TOF_SHIFT)) & MPWM_STS0_CH7TOF_MASK)
#define MPWM_STS0_CH6TOF_MASK                               (0x40U)
#define MPWM_STS0_CH6TOF_SHIFT                              (6U)
#define MPWM_STS0_CH6TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH6TOF_SHIFT)) & MPWM_STS0_CH6TOF_MASK)
#define MPWM_STS0_CH5TOF_MASK                               (0x20U)
#define MPWM_STS0_CH5TOF_SHIFT                              (5U)
#define MPWM_STS0_CH5TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH5TOF_SHIFT)) & MPWM_STS0_CH5TOF_MASK)
#define MPWM_STS0_CH4TOF_MASK                               (0x10U)
#define MPWM_STS0_CH4TOF_SHIFT                              (4U)
#define MPWM_STS0_CH4TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH4TOF_SHIFT)) & MPWM_STS0_CH4TOF_MASK)
#define MPWM_STS0_CH3TOF_MASK                               (0x8U)
#define MPWM_STS0_CH3TOF_SHIFT                              (3U)
#define MPWM_STS0_CH3TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH3TOF_SHIFT)) & MPWM_STS0_CH3TOF_MASK)
#define MPWM_STS0_CH2TOF_MASK                               (0x4U)
#define MPWM_STS0_CH2TOF_SHIFT                              (2U)
#define MPWM_STS0_CH2TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH2TOF_SHIFT)) & MPWM_STS0_CH2TOF_MASK)
#define MPWM_STS0_CH1TOF_MASK                               (0x2U)
#define MPWM_STS0_CH1TOF_SHIFT                              (1U)
#define MPWM_STS0_CH1TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH1TOF_SHIFT)) & MPWM_STS0_CH1TOF_MASK)
#define MPWM_STS0_CH0TOF_MASK                               (0x1U)
#define MPWM_STS0_CH0TOF_SHIFT                              (0U)
#define MPWM_STS0_CH0TOF(x)                                 (((uint32)(((uint32)(x)) << MPWM_STS0_CH0TOF_SHIFT)) & MPWM_STS0_CH0TOF_MASK)
/* MPWM_STS1 Register */
#define MPWM_STS1_CH15F_MASK                                (0x8000U)
#define MPWM_STS1_CH15F_SHIFT                               (15U)
#define MPWM_STS1_CH15F(x)                                  (((uint32)(((uint32)(x)) << MPWM_STS1_CH15F_SHIFT)) & MPWM_STS1_CH15F_MASK)
#define MPWM_STS1_CH14F_MASK                                (0x4000U)
#define MPWM_STS1_CH14F_SHIFT                               (14U)
#define MPWM_STS1_CH14F(x)                                  (((uint32)(((uint32)(x)) << MPWM_STS1_CH14F_SHIFT)) & MPWM_STS1_CH14F_MASK)
#define MPWM_STS1_CH13F_MASK                                (0x2000U)
#define MPWM_STS1_CH13F_SHIFT                               (13U)
#define MPWM_STS1_CH13F(x)                                  (((uint32)(((uint32)(x)) << MPWM_STS1_CH13F_SHIFT)) & MPWM_STS1_CH13F_MASK)
#define MPWM_STS1_CH12F_MASK                                (0x1000U)
#define MPWM_STS1_CH12F_SHIFT                               (12U)
#define MPWM_STS1_CH12F(x)                                  (((uint32)(((uint32)(x)) << MPWM_STS1_CH12F_SHIFT)) & MPWM_STS1_CH12F_MASK)
#define MPWM_STS1_CH11F_MASK                                (0x800U)
#define MPWM_STS1_CH11F_SHIFT                               (11U)
#define MPWM_STS1_CH11F(x)                                  (((uint32)(((uint32)(x)) << MPWM_STS1_CH11F_SHIFT)) & MPWM_STS1_CH11F_MASK)
#define MPWM_STS1_CH10F_MASK                                (0x400U)
#define MPWM_STS1_CH10F_SHIFT                               (10U)
#define MPWM_STS1_CH10F(x)                                  (((uint32)(((uint32)(x)) << MPWM_STS1_CH10F_SHIFT)) & MPWM_STS1_CH10F_MASK)
#define MPWM_STS1_CH9F_MASK                                 (0x200U)
#define MPWM_STS1_CH9F_SHIFT                                (9U)
#define MPWM_STS1_CH9F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH9F_SHIFT)) & MPWM_STS1_CH9F_MASK)
#define MPWM_STS1_CH8F_MASK                                 (0x100U)
#define MPWM_STS1_CH8F_SHIFT                                (8U)
#define MPWM_STS1_CH8F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH8F_SHIFT)) & MPWM_STS1_CH8F_MASK)
#define MPWM_STS1_CH7F_MASK                                 (0x80U)
#define MPWM_STS1_CH7F_SHIFT                                (7U)
#define MPWM_STS1_CH7F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH7F_SHIFT)) & MPWM_STS1_CH7F_MASK)
#define MPWM_STS1_CH6F_MASK                                 (0x40U)
#define MPWM_STS1_CH6F_SHIFT                                (6U)
#define MPWM_STS1_CH6F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH6F_SHIFT)) & MPWM_STS1_CH6F_MASK)
#define MPWM_STS1_CH5F_MASK                                 (0x20U)
#define MPWM_STS1_CH5F_SHIFT                                (5U)
#define MPWM_STS1_CH5F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH5F_SHIFT)) & MPWM_STS1_CH5F_MASK)
#define MPWM_STS1_CH4F_MASK                                 (0x10U)
#define MPWM_STS1_CH4F_SHIFT                                (4U)
#define MPWM_STS1_CH4F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH4F_SHIFT)) & MPWM_STS1_CH4F_MASK)
#define MPWM_STS1_CH3F_MASK                                 (0x8U)
#define MPWM_STS1_CH3F_SHIFT                                (3U)
#define MPWM_STS1_CH3F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH3F_SHIFT)) & MPWM_STS1_CH3F_MASK)
#define MPWM_STS1_CH2F_MASK                                 (0x4U)
#define MPWM_STS1_CH2F_SHIFT                                (2U)
#define MPWM_STS1_CH2F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH2F_SHIFT)) & MPWM_STS1_CH2F_MASK)
#define MPWM_STS1_CH1F_MASK                                 (0x2U)
#define MPWM_STS1_CH1F_SHIFT                                (1U)
#define MPWM_STS1_CH1F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH1F_SHIFT)) & MPWM_STS1_CH1F_MASK)
#define MPWM_STS1_CH0F_MASK                                 (0x1U)
#define MPWM_STS1_CH0F_SHIFT                                (0U)
#define MPWM_STS1_CH0F(x)                                   (((uint32)(((uint32)(x)) << MPWM_STS1_CH0F_SHIFT)) & MPWM_STS1_CH0F_MASK)
/* MPWM_INTE0 Register */
#define MPWM_INTE0_CH15TOIE_MASK                            (0x8000U)
#define MPWM_INTE0_CH15TOIE_SHIFT                           (15U)
#define MPWM_INTE0_CH15TOIE(x)                              (((uint32)(((uint32)(x)) << MPWM_INTE0_CH15TOIE_SHIFT)) & MPWM_INTE0_CH15TOIE_MASK)
#define MPWM_INTE0_CH14TOIE_MASK                            (0x4000U)
#define MPWM_INTE0_CH14TOIE_SHIFT                           (14U)
#define MPWM_INTE0_CH14TOIE(x)                              (((uint32)(((uint32)(x)) << MPWM_INTE0_CH14TOIE_SHIFT)) & MPWM_INTE0_CH14TOIE_MASK)
#define MPWM_INTE0_CH13TOIE_MASK                            (0x2000U)
#define MPWM_INTE0_CH13TOIE_SHIFT                           (13U)
#define MPWM_INTE0_CH13TOIE(x)                              (((uint32)(((uint32)(x)) << MPWM_INTE0_CH13TOIE_SHIFT)) & MPWM_INTE0_CH13TOIE_MASK)
#define MPWM_INTE0_CH12TOIE_MASK                            (0x1000U)
#define MPWM_INTE0_CH12TOIE_SHIFT                           (12U)
#define MPWM_INTE0_CH12TOIE(x)                              (((uint32)(((uint32)(x)) << MPWM_INTE0_CH12TOIE_SHIFT)) & MPWM_INTE0_CH12TOIE_MASK)
#define MPWM_INTE0_CH11TOIE_MASK                            (0x800U)
#define MPWM_INTE0_CH11TOIE_SHIFT                           (11U)
#define MPWM_INTE0_CH11TOIE(x)                              (((uint32)(((uint32)(x)) << MPWM_INTE0_CH11TOIE_SHIFT)) & MPWM_INTE0_CH11TOIE_MASK)
#define MPWM_INTE0_CH10TOIE_MASK                            (0x400U)
#define MPWM_INTE0_CH10TOIE_SHIFT                           (10U)
#define MPWM_INTE0_CH10TOIE(x)                              (((uint32)(((uint32)(x)) << MPWM_INTE0_CH10TOIE_SHIFT)) & MPWM_INTE0_CH10TOIE_MASK)
#define MPWM_INTE0_CH9TOIE_MASK                             (0x200U)
#define MPWM_INTE0_CH9TOIE_SHIFT                            (9U)
#define MPWM_INTE0_CH9TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH9TOIE_SHIFT)) & MPWM_INTE0_CH9TOIE_MASK)
#define MPWM_INTE0_CH8TOIE_MASK                             (0x100U)
#define MPWM_INTE0_CH8TOIE_SHIFT                            (8U)
#define MPWM_INTE0_CH8TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH8TOIE_SHIFT)) & MPWM_INTE0_CH8TOIE_MASK)
#define MPWM_INTE0_CH7TOIE_MASK                             (0x80U)
#define MPWM_INTE0_CH7TOIE_SHIFT                            (7U)
#define MPWM_INTE0_CH7TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH7TOIE_SHIFT)) & MPWM_INTE0_CH7TOIE_MASK)
#define MPWM_INTE0_CH6TOIE_MASK                             (0x40U)
#define MPWM_INTE0_CH6TOIE_SHIFT                            (6U)
#define MPWM_INTE0_CH6TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH6TOIE_SHIFT)) & MPWM_INTE0_CH6TOIE_MASK)
#define MPWM_INTE0_CH5TOIE_MASK                             (0x20U)
#define MPWM_INTE0_CH5TOIE_SHIFT                            (5U)
#define MPWM_INTE0_CH5TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH5TOIE_SHIFT)) & MPWM_INTE0_CH5TOIE_MASK)
#define MPWM_INTE0_CH4TOIE_MASK                             (0x10U)
#define MPWM_INTE0_CH4TOIE_SHIFT                            (4U)
#define MPWM_INTE0_CH4TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH4TOIE_SHIFT)) & MPWM_INTE0_CH4TOIE_MASK)
#define MPWM_INTE0_CH3TOIE_MASK                             (0x8U)
#define MPWM_INTE0_CH3TOIE_SHIFT                            (3U)
#define MPWM_INTE0_CH3TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH3TOIE_SHIFT)) & MPWM_INTE0_CH3TOIE_MASK)
#define MPWM_INTE0_CH2TOIE_MASK                             (0x4U)
#define MPWM_INTE0_CH2TOIE_SHIFT                            (2U)
#define MPWM_INTE0_CH2TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH2TOIE_SHIFT)) & MPWM_INTE0_CH2TOIE_MASK)
#define MPWM_INTE0_CH1TOIE_MASK                             (0x2U)
#define MPWM_INTE0_CH1TOIE_SHIFT                            (1U)
#define MPWM_INTE0_CH1TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH1TOIE_SHIFT)) & MPWM_INTE0_CH1TOIE_MASK)
#define MPWM_INTE0_CH0TOIE_MASK                             (0x1U)
#define MPWM_INTE0_CH0TOIE_SHIFT                            (0U)
#define MPWM_INTE0_CH0TOIE(x)                               (((uint32)(((uint32)(x)) << MPWM_INTE0_CH0TOIE_SHIFT)) & MPWM_INTE0_CH0TOIE_MASK)
/* MPWM_INTE1 Register */
#define MPWM_INTE1_CH15IE_MASK                              (0x8000U)
#define MPWM_INTE1_CH15IE_SHIFT                             (15U)
#define MPWM_INTE1_CH15IE(x)                                (((uint32)(((uint32)(x)) << MPWM_INTE1_CH15IE_SHIFT)) & MPWM_INTE1_CH15IE_MASK)
#define MPWM_INTE1_CH14IE_MASK                              (0x4000U)
#define MPWM_INTE1_CH14IE_SHIFT                             (14U)
#define MPWM_INTE1_CH14IE(x)                                (((uint32)(((uint32)(x)) << MPWM_INTE1_CH14IE_SHIFT)) & MPWM_INTE1_CH14IE_MASK)
#define MPWM_INTE1_CH13IE_MASK                              (0x2000U)
#define MPWM_INTE1_CH13IE_SHIFT                             (13U)
#define MPWM_INTE1_CH13IE(x)                                (((uint32)(((uint32)(x)) << MPWM_INTE1_CH13IE_SHIFT)) & MPWM_INTE1_CH13IE_MASK)
#define MPWM_INTE1_CH12IE_MASK                              (0x1000U)
#define MPWM_INTE1_CH12IE_SHIFT                             (12U)
#define MPWM_INTE1_CH12IE(x)                                (((uint32)(((uint32)(x)) << MPWM_INTE1_CH12IE_SHIFT)) & MPWM_INTE1_CH12IE_MASK)
#define MPWM_INTE1_CH11IE_MASK                              (0x800U)
#define MPWM_INTE1_CH11IE_SHIFT                             (11U)
#define MPWM_INTE1_CH11IE(x)                                (((uint32)(((uint32)(x)) << MPWM_INTE1_CH11IE_SHIFT)) & MPWM_INTE1_CH11IE_MASK)
#define MPWM_INTE1_CH10IE_MASK                              (0x400U)
#define MPWM_INTE1_CH10IE_SHIFT                             (10U)
#define MPWM_INTE1_CH10IE(x)                                (((uint32)(((uint32)(x)) << MPWM_INTE1_CH10IE_SHIFT)) & MPWM_INTE1_CH10IE_MASK)
#define MPWM_INTE1_CH9IE_MASK                               (0x200U)
#define MPWM_INTE1_CH9IE_SHIFT                              (9U)
#define MPWM_INTE1_CH9IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH9IE_SHIFT)) & MPWM_INTE1_CH9IE_MASK)
#define MPWM_INTE1_CH8IE_MASK                               (0x100U)
#define MPWM_INTE1_CH8IE_SHIFT                              (8U)
#define MPWM_INTE1_CH8IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH8IE_SHIFT)) & MPWM_INTE1_CH8IE_MASK)
#define MPWM_INTE1_CH7IE_MASK                               (0x80U)
#define MPWM_INTE1_CH7IE_SHIFT                              (7U)
#define MPWM_INTE1_CH7IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH7IE_SHIFT)) & MPWM_INTE1_CH7IE_MASK)
#define MPWM_INTE1_CH6IE_MASK                               (0x40U)
#define MPWM_INTE1_CH6IE_SHIFT                              (6U)
#define MPWM_INTE1_CH6IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH6IE_SHIFT)) & MPWM_INTE1_CH6IE_MASK)
#define MPWM_INTE1_CH5IE_MASK                               (0x20U)
#define MPWM_INTE1_CH5IE_SHIFT                              (5U)
#define MPWM_INTE1_CH5IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH5IE_SHIFT)) & MPWM_INTE1_CH5IE_MASK)
#define MPWM_INTE1_CH4IE_MASK                               (0x10U)
#define MPWM_INTE1_CH4IE_SHIFT                              (4U)
#define MPWM_INTE1_CH4IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH4IE_SHIFT)) & MPWM_INTE1_CH4IE_MASK)
#define MPWM_INTE1_CH3IE_MASK                               (0x8U)
#define MPWM_INTE1_CH3IE_SHIFT                              (3U)
#define MPWM_INTE1_CH3IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH3IE_SHIFT)) & MPWM_INTE1_CH3IE_MASK)
#define MPWM_INTE1_CH2IE_MASK                               (0x4U)
#define MPWM_INTE1_CH2IE_SHIFT                              (2U)
#define MPWM_INTE1_CH2IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH2IE_SHIFT)) & MPWM_INTE1_CH2IE_MASK)
#define MPWM_INTE1_CH1IE_MASK                               (0x2U)
#define MPWM_INTE1_CH1IE_SHIFT                              (1U)
#define MPWM_INTE1_CH1IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH1IE_SHIFT)) & MPWM_INTE1_CH1IE_MASK)
#define MPWM_INTE1_CH0IE_MASK                               (0x1U)
#define MPWM_INTE1_CH0IE_SHIFT                              (0U)
#define MPWM_INTE1_CH0IE(x)                                 (((uint32)(((uint32)(x)) << MPWM_INTE1_CH0IE_SHIFT)) & MPWM_INTE1_CH0IE_MASK)
/* MPWM_CTRL Register */
#define MPWM_CTRL_CH15EN_MASK                               (0x8000U)
#define MPWM_CTRL_CH15EN_SHIFT                              (15U)
#define MPWM_CTRL_CH15EN(x)                                 (((uint32)(((uint32)(x)) << MPWM_CTRL_CH15EN_SHIFT)) & MPWM_CTRL_CH15EN_MASK)
#define MPWM_CTRL_CH14EN_MASK                               (0x4000U)
#define MPWM_CTRL_CH14EN_SHIFT                              (14U)
#define MPWM_CTRL_CH14EN(x)                                 (((uint32)(((uint32)(x)) << MPWM_CTRL_CH14EN_SHIFT)) & MPWM_CTRL_CH14EN_MASK)
#define MPWM_CTRL_CH13EN_MASK                               (0x2000U)
#define MPWM_CTRL_CH13EN_SHIFT                              (13U)
#define MPWM_CTRL_CH13EN(x)                                 (((uint32)(((uint32)(x)) << MPWM_CTRL_CH13EN_SHIFT)) & MPWM_CTRL_CH13EN_MASK)
#define MPWM_CTRL_CH12EN_MASK                               (0x1000U)
#define MPWM_CTRL_CH12EN_SHIFT                              (12U)
#define MPWM_CTRL_CH12EN(x)                                 (((uint32)(((uint32)(x)) << MPWM_CTRL_CH12EN_SHIFT)) & MPWM_CTRL_CH12EN_MASK)
#define MPWM_CTRL_CH11EN_MASK                               (0x800U)
#define MPWM_CTRL_CH11EN_SHIFT                              (11U)
#define MPWM_CTRL_CH11EN(x)                                 (((uint32)(((uint32)(x)) << MPWM_CTRL_CH11EN_SHIFT)) & MPWM_CTRL_CH11EN_MASK)
#define MPWM_CTRL_CH10EN_MASK                               (0x400U)
#define MPWM_CTRL_CH10EN_SHIFT                              (10U)
#define MPWM_CTRL_CH10EN(x)                                 (((uint32)(((uint32)(x)) << MPWM_CTRL_CH10EN_SHIFT)) & MPWM_CTRL_CH10EN_MASK)
#define MPWM_CTRL_CH9EN_MASK                                (0x200U)
#define MPWM_CTRL_CH9EN_SHIFT                               (9U)
#define MPWM_CTRL_CH9EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH9EN_SHIFT)) & MPWM_CTRL_CH9EN_MASK)
#define MPWM_CTRL_CH8EN_MASK                                (0x100U)
#define MPWM_CTRL_CH8EN_SHIFT                               (8U)
#define MPWM_CTRL_CH8EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH8EN_SHIFT)) & MPWM_CTRL_CH8EN_MASK)
#define MPWM_CTRL_CH7EN_MASK                                (0x80U)
#define MPWM_CTRL_CH7EN_SHIFT                               (7U)
#define MPWM_CTRL_CH7EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH7EN_SHIFT)) & MPWM_CTRL_CH7EN_MASK)
#define MPWM_CTRL_CH6EN_MASK                                (0x40U)
#define MPWM_CTRL_CH6EN_SHIFT                               (6U)
#define MPWM_CTRL_CH6EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH6EN_SHIFT)) & MPWM_CTRL_CH6EN_MASK)
#define MPWM_CTRL_CH5EN_MASK                                (0x20U)
#define MPWM_CTRL_CH5EN_SHIFT                               (5U)
#define MPWM_CTRL_CH5EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH5EN_SHIFT)) & MPWM_CTRL_CH5EN_MASK)
#define MPWM_CTRL_CH4EN_MASK                                (0x10U)
#define MPWM_CTRL_CH4EN_SHIFT                               (4U)
#define MPWM_CTRL_CH4EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH4EN_SHIFT)) & MPWM_CTRL_CH4EN_MASK)
#define MPWM_CTRL_CH3EN_MASK                                (0x8U)
#define MPWM_CTRL_CH3EN_SHIFT                               (3U)
#define MPWM_CTRL_CH3EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH3EN_SHIFT)) & MPWM_CTRL_CH3EN_MASK)
#define MPWM_CTRL_CH2EN_MASK                                (0x4U)
#define MPWM_CTRL_CH2EN_SHIFT                               (2U)
#define MPWM_CTRL_CH2EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH2EN_SHIFT)) & MPWM_CTRL_CH2EN_MASK)
#define MPWM_CTRL_CH1EN_MASK                                (0x2U)
#define MPWM_CTRL_CH1EN_SHIFT                               (1U)
#define MPWM_CTRL_CH1EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH1EN_SHIFT)) & MPWM_CTRL_CH1EN_MASK)
#define MPWM_CTRL_CH0EN_MASK                                (0x1U)
#define MPWM_CTRL_CH0EN_SHIFT                               (0U)
#define MPWM_CTRL_CH0EN(x)                                  (((uint32)(((uint32)(x)) << MPWM_CTRL_CH0EN_SHIFT)) & MPWM_CTRL_CH0EN_MASK)
/* MPWM_SWT Register */
#define MPWM_SWT_CH15SWT_MASK                               (0x8000U)
#define MPWM_SWT_CH15SWT_SHIFT                              (15U)
#define MPWM_SWT_CH15SWT(x)                                 (((uint32)(((uint32)(x)) << MPWM_SWT_CH15SWT_SHIFT)) & MPWM_SWT_CH15SWT_MASK)
#define MPWM_SWT_CH14SWT_MASK                               (0x4000U)
#define MPWM_SWT_CH14SWT_SHIFT                              (14U)
#define MPWM_SWT_CH14SWT(x)                                 (((uint32)(((uint32)(x)) << MPWM_SWT_CH14SWT_SHIFT)) & MPWM_SWT_CH14SWT_MASK)
#define MPWM_SWT_CH13SWT_MASK                               (0x2000U)
#define MPWM_SWT_CH13SWT_SHIFT                              (13U)
#define MPWM_SWT_CH13SWT(x)                                 (((uint32)(((uint32)(x)) << MPWM_SWT_CH13SWT_SHIFT)) & MPWM_SWT_CH13SWT_MASK)
#define MPWM_SWT_CH12SWT_MASK                               (0x1000U)
#define MPWM_SWT_CH12SWT_SHIFT                              (12U)
#define MPWM_SWT_CH12SWT(x)                                 (((uint32)(((uint32)(x)) << MPWM_SWT_CH12SWT_SHIFT)) & MPWM_SWT_CH12SWT_MASK)
#define MPWM_SWT_CH11SWT_MASK                               (0x800U)
#define MPWM_SWT_CH11SWT_SHIFT                              (11U)
#define MPWM_SWT_CH11SWT(x)                                 (((uint32)(((uint32)(x)) << MPWM_SWT_CH11SWT_SHIFT)) & MPWM_SWT_CH11SWT_MASK)
#define MPWM_SWT_CH10SWT_MASK                               (0x400U)
#define MPWM_SWT_CH10SWT_SHIFT                              (10U)
#define MPWM_SWT_CH10SWT(x)                                 (((uint32)(((uint32)(x)) << MPWM_SWT_CH10SWT_SHIFT)) & MPWM_SWT_CH10SWT_MASK)
#define MPWM_SWT_CH9SWT_MASK                                (0x200U)
#define MPWM_SWT_CH9SWT_SHIFT                               (9U)
#define MPWM_SWT_CH9SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH9SWT_SHIFT)) & MPWM_SWT_CH9SWT_MASK)
#define MPWM_SWT_CH8SWT_MASK                                (0x100U)
#define MPWM_SWT_CH8SWT_SHIFT                               (8U)
#define MPWM_SWT_CH8SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH8SWT_SHIFT)) & MPWM_SWT_CH8SWT_MASK)
#define MPWM_SWT_CH7SWT_MASK                                (0x80U)
#define MPWM_SWT_CH7SWT_SHIFT                               (7U)
#define MPWM_SWT_CH7SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH7SWT_SHIFT)) & MPWM_SWT_CH7SWT_MASK)
#define MPWM_SWT_CH6SWT_MASK                                (0x40U)
#define MPWM_SWT_CH6SWT_SHIFT                               (6U)
#define MPWM_SWT_CH6SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH6SWT_SHIFT)) & MPWM_SWT_CH6SWT_MASK)
#define MPWM_SWT_CH5SWT_MASK                                (0x20U)
#define MPWM_SWT_CH5SWT_SHIFT                               (5U)
#define MPWM_SWT_CH5SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH5SWT_SHIFT)) & MPWM_SWT_CH5SWT_MASK)
#define MPWM_SWT_CH4SWT_MASK                                (0x10U)
#define MPWM_SWT_CH4SWT_SHIFT                               (4U)
#define MPWM_SWT_CH4SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH4SWT_SHIFT)) & MPWM_SWT_CH4SWT_MASK)
#define MPWM_SWT_CH3SWT_MASK                                (0x8U)
#define MPWM_SWT_CH3SWT_SHIFT                               (3U)
#define MPWM_SWT_CH3SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH3SWT_SHIFT)) & MPWM_SWT_CH3SWT_MASK)
#define MPWM_SWT_CH2SWT_MASK                                (0x4U)
#define MPWM_SWT_CH2SWT_SHIFT                               (2U)
#define MPWM_SWT_CH2SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH2SWT_SHIFT)) & MPWM_SWT_CH2SWT_MASK)
#define MPWM_SWT_CH1SWT_MASK                                (0x2U)
#define MPWM_SWT_CH1SWT_SHIFT                               (1U)
#define MPWM_SWT_CH1SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH1SWT_SHIFT)) & MPWM_SWT_CH1SWT_MASK)
#define MPWM_SWT_CH0SWT_MASK                                (0x1U)
#define MPWM_SWT_CH0SWT_SHIFT                               (0U)
#define MPWM_SWT_CH0SWT(x)                                  (((uint32)(((uint32)(x)) << MPWM_SWT_CH0SWT_SHIFT)) & MPWM_SWT_CH0SWT_MASK)
/* MPWM_INIT Register */
#define MPWM_INIT_CH15INIT_MASK                             (0x8000U)
#define MPWM_INIT_CH15INIT_SHIFT                            (15U)
#define MPWM_INIT_CH15INIT(x)                               (((uint32)(((uint32)(x)) << MPWM_INIT_CH15INIT_SHIFT)) & MPWM_INIT_CH15INIT_MASK)
#define MPWM_INIT_CH14INIT_MASK                             (0x4000U)
#define MPWM_INIT_CH14INIT_SHIFT                            (14U)
#define MPWM_INIT_CH14INIT(x)                               (((uint32)(((uint32)(x)) << MPWM_INIT_CH14INIT_SHIFT)) & MPWM_INIT_CH14INIT_MASK)
#define MPWM_INIT_CH13INIT_MASK                             (0x2000U)
#define MPWM_INIT_CH13INIT_SHIFT                            (13U)
#define MPWM_INIT_CH13INIT(x)                               (((uint32)(((uint32)(x)) << MPWM_INIT_CH13INIT_SHIFT)) & MPWM_INIT_CH13INIT_MASK)
#define MPWM_INIT_CH12INIT_MASK                             (0x1000U)
#define MPWM_INIT_CH12INIT_SHIFT                            (12U)
#define MPWM_INIT_CH12INIT(x)                               (((uint32)(((uint32)(x)) << MPWM_INIT_CH12INIT_SHIFT)) & MPWM_INIT_CH12INIT_MASK)
#define MPWM_INIT_CH11INIT_MASK                             (0x800U)
#define MPWM_INIT_CH11INIT_SHIFT                            (11U)
#define MPWM_INIT_CH11INIT(x)                               (((uint32)(((uint32)(x)) << MPWM_INIT_CH11INIT_SHIFT)) & MPWM_INIT_CH11INIT_MASK)
#define MPWM_INIT_CH10INIT_MASK                             (0x400U)
#define MPWM_INIT_CH10INIT_SHIFT                            (10U)
#define MPWM_INIT_CH10INIT(x)                               (((uint32)(((uint32)(x)) << MPWM_INIT_CH10INIT_SHIFT)) & MPWM_INIT_CH10INIT_MASK)
#define MPWM_INIT_CH9INIT_MASK                              (0x200U)
#define MPWM_INIT_CH9INIT_SHIFT                             (9U)
#define MPWM_INIT_CH9INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH9INIT_SHIFT)) & MPWM_INIT_CH9INIT_MASK)
#define MPWM_INIT_CH8INIT_MASK                              (0x100U)
#define MPWM_INIT_CH8INIT_SHIFT                             (8U)
#define MPWM_INIT_CH8INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH8INIT_SHIFT)) & MPWM_INIT_CH8INIT_MASK)
#define MPWM_INIT_CH7INIT_MASK                              (0x80U)
#define MPWM_INIT_CH7INIT_SHIFT                             (7U)
#define MPWM_INIT_CH7INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH7INIT_SHIFT)) & MPWM_INIT_CH7INIT_MASK)
#define MPWM_INIT_CH6INIT_MASK                              (0x40U)
#define MPWM_INIT_CH6INIT_SHIFT                             (6U)
#define MPWM_INIT_CH6INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH6INIT_SHIFT)) & MPWM_INIT_CH6INIT_MASK)
#define MPWM_INIT_CH5INIT_MASK                              (0x20U)
#define MPWM_INIT_CH5INIT_SHIFT                             (5U)
#define MPWM_INIT_CH5INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH5INIT_SHIFT)) & MPWM_INIT_CH5INIT_MASK)
#define MPWM_INIT_CH4INIT_MASK                              (0x10U)
#define MPWM_INIT_CH4INIT_SHIFT                             (4U)
#define MPWM_INIT_CH4INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH4INIT_SHIFT)) & MPWM_INIT_CH4INIT_MASK)
#define MPWM_INIT_CH3INIT_MASK                              (0x8U)
#define MPWM_INIT_CH3INIT_SHIFT                             (3U)
#define MPWM_INIT_CH3INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH3INIT_SHIFT)) & MPWM_INIT_CH3INIT_MASK)
#define MPWM_INIT_CH2INIT_MASK                              (0x4U)
#define MPWM_INIT_CH2INIT_SHIFT                             (2U)
#define MPWM_INIT_CH2INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH2INIT_SHIFT)) & MPWM_INIT_CH2INIT_MASK)
#define MPWM_INIT_CH1INIT_MASK                              (0x2U)
#define MPWM_INIT_CH1INIT_SHIFT                             (1U)
#define MPWM_INIT_CH1INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH1INIT_SHIFT)) & MPWM_INIT_CH1INIT_MASK)
#define MPWM_INIT_CH0INIT_MASK                              (0x1U)
#define MPWM_INIT_CH0INIT_SHIFT                             (0U)
#define MPWM_INIT_CH0INIT(x)                                (((uint32)(((uint32)(x)) << MPWM_INIT_CH0INIT_SHIFT)) & MPWM_INIT_CH0INIT_MASK)
/* MPWM_POL Register */
#define MPWM_POL_CH15POL_MASK                               (0x8000U)
#define MPWM_POL_CH15POL_SHIFT                              (15U)
#define MPWM_POL_CH15POL(x)                                 (((uint32)(((uint32)(x)) << MPWM_POL_CH15POL_SHIFT)) & MPWM_POL_CH15POL_MASK)
#define MPWM_POL_CH14POL_MASK                               (0x4000U)
#define MPWM_POL_CH14POL_SHIFT                              (14U)
#define MPWM_POL_CH14POL(x)                                 (((uint32)(((uint32)(x)) << MPWM_POL_CH14POL_SHIFT)) & MPWM_POL_CH14POL_MASK)
#define MPWM_POL_CH13POL_MASK                               (0x2000U)
#define MPWM_POL_CH13POL_SHIFT                              (13U)
#define MPWM_POL_CH13POL(x)                                 (((uint32)(((uint32)(x)) << MPWM_POL_CH13POL_SHIFT)) & MPWM_POL_CH13POL_MASK)
#define MPWM_POL_CH12POL_MASK                               (0x1000U)
#define MPWM_POL_CH12POL_SHIFT                              (12U)
#define MPWM_POL_CH12POL(x)                                 (((uint32)(((uint32)(x)) << MPWM_POL_CH12POL_SHIFT)) & MPWM_POL_CH12POL_MASK)
#define MPWM_POL_CH11POL_MASK                               (0x800U)
#define MPWM_POL_CH11POL_SHIFT                              (11U)
#define MPWM_POL_CH11POL(x)                                 (((uint32)(((uint32)(x)) << MPWM_POL_CH11POL_SHIFT)) & MPWM_POL_CH11POL_MASK)
#define MPWM_POL_CH10POL_MASK                               (0x400U)
#define MPWM_POL_CH10POL_SHIFT                              (10U)
#define MPWM_POL_CH10POL(x)                                 (((uint32)(((uint32)(x)) << MPWM_POL_CH10POL_SHIFT)) & MPWM_POL_CH10POL_MASK)
#define MPWM_POL_CH9POL_MASK                                (0x200U)
#define MPWM_POL_CH9POL_SHIFT                               (9U)
#define MPWM_POL_CH9POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH9POL_SHIFT)) & MPWM_POL_CH9POL_MASK)
#define MPWM_POL_CH8POL_MASK                                (0x100U)
#define MPWM_POL_CH8POL_SHIFT                               (8U)
#define MPWM_POL_CH8POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH8POL_SHIFT)) & MPWM_POL_CH8POL_MASK)
#define MPWM_POL_CH7POL_MASK                                (0x80U)
#define MPWM_POL_CH7POL_SHIFT                               (7U)
#define MPWM_POL_CH7POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH7POL_SHIFT)) & MPWM_POL_CH7POL_MASK)
#define MPWM_POL_CH6POL_MASK                                (0x40U)
#define MPWM_POL_CH6POL_SHIFT                               (6U)
#define MPWM_POL_CH6POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH6POL_SHIFT)) & MPWM_POL_CH6POL_MASK)
#define MPWM_POL_CH5POL_MASK                                (0x20U)
#define MPWM_POL_CH5POL_SHIFT                               (5U)
#define MPWM_POL_CH5POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH5POL_SHIFT)) & MPWM_POL_CH5POL_MASK)
#define MPWM_POL_CH4POL_MASK                                (0x10U)
#define MPWM_POL_CH4POL_SHIFT                               (4U)
#define MPWM_POL_CH4POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH4POL_SHIFT)) & MPWM_POL_CH4POL_MASK)
#define MPWM_POL_CH3POL_MASK                                (0x8U)
#define MPWM_POL_CH3POL_SHIFT                               (3U)
#define MPWM_POL_CH3POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH3POL_SHIFT)) & MPWM_POL_CH3POL_MASK)
#define MPWM_POL_CH2POL_MASK                                (0x4U)
#define MPWM_POL_CH2POL_SHIFT                               (2U)
#define MPWM_POL_CH2POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH2POL_SHIFT)) & MPWM_POL_CH2POL_MASK)
#define MPWM_POL_CH1POL_MASK                                (0x2U)
#define MPWM_POL_CH1POL_SHIFT                               (1U)
#define MPWM_POL_CH1POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH1POL_SHIFT)) & MPWM_POL_CH1POL_MASK)
#define MPWM_POL_CH0POL_MASK                                (0x1U)
#define MPWM_POL_CH0POL_SHIFT                               (0U)
#define MPWM_POL_CH0POL(x)                                  (((uint32)(((uint32)(x)) << MPWM_POL_CH0POL_SHIFT)) & MPWM_POL_CH0POL_MASK)
/* MPWM_CH_CFG Register */
#define MPWM_CH_CFG_ACLR_MASK                                  (0x8000000U)
#define MPWM_CH_CFG_ACLR_SHIFT                                 (27U)
#define MPWM_CH_CFG_ACLR(x)                                    (((uint32)(((uint32)(x)) << MPWM_CH_CFG_ACLR_SHIFT)) & MPWM_CH_CFG_ACLR_MASK)
#define MPWM_CH_CFG_OVWR_MASK                                  (0x4000000U)
#define MPWM_CH_CFG_OVWR_SHIFT                                 (26U)
#define MPWM_CH_CFG_OVWR(x)                                    (((uint32)(((uint32)(x)) << MPWM_CH_CFG_OVWR_SHIFT)) & MPWM_CH_CFG_OVWR_MASK)
#define MPWM_CH_CFG_CAPEDGE_WIDTH                              (2)
#define MPWM_CH_CFG_CAPEDGE_MASK                               (0x3000000U)
#define MPWM_CH_CFG_CAPEDGE_SHIFT                              (24U)
#define MPWM_CH_CFG_CAPEDGE(x)                                 (((uint32)(((uint32)(x)) << MPWM_CH_CFG_CAPEDGE_SHIFT)) & MPWM_CH_CFG_CAPEDGE_MASK)
#define MPWM_CH_CFG_CMPOUT_WIDTH                               (2)
#define MPWM_CH_CFG_CMPOUT_MASK                                (0x300000U)
#define MPWM_CH_CFG_CMPOUT_SHIFT                               (20U)
#define MPWM_CH_CFG_CMPOUT(x)                                  (((uint32)(((uint32)(x)) << MPWM_CH_CFG_CMPOUT_SHIFT)) & MPWM_CH_CFG_CMPOUT_MASK)
#define MPWM_CH_CFG_RLDSRC_WIDTH                               (2)
#define MPWM_CH_CFG_RLDSRC_MASK                                (0x30000U)
#define MPWM_CH_CFG_RLDSRC_SHIFT                               (16U)
#define MPWM_CH_CFG_RLDSRC(x)                                  (((uint32)(((uint32)(x)) << MPWM_CH_CFG_RLDSRC_SHIFT)) & MPWM_CH_CFG_RLDSRC_MASK)
#define MPWM_CH_CFG_DMAEN_MASK                                 (0x8000U)
#define MPWM_CH_CFG_DMAEN_SHIFT                                (15U)
#define MPWM_CH_CFG_DMAEN(x)                                   (((uint32)(((uint32)(x)) << MPWM_CH_CFG_DMAEN_SHIFT)) & MPWM_CH_CFG_DMAEN_MASK)
#define MPWM_CH_CFG_DBGDIS_MASK                                (0x4000U)
#define MPWM_CH_CFG_DBGDIS_SHIFT                               (14U)
#define MPWM_CH_CFG_DBGDIS(x)                                  (((uint32)(((uint32)(x)) << MPWM_CH_CFG_DBGDIS_SHIFT)) & MPWM_CH_CFG_DBGDIS_MASK)
#define MPWM_CH_CFG_CLDOK_MASK                                 (0x2000U)
#define MPWM_CH_CFG_CLDOK_SHIFT                                (13U)
#define MPWM_CH_CFG_CLDOK(x)                                   (((uint32)(((uint32)(x)) << MPWM_CH_CFG_CLDOK_SHIFT)) & MPWM_CH_CFG_CLDOK_MASK)
#define MPWM_CH_CFG_LDOK_MASK                                  (0x1000U)
#define MPWM_CH_CFG_LDOK_SHIFT                                 (12U)
#define MPWM_CH_CFG_LDOK(x)                                    (((uint32)(((uint32)(x)) << MPWM_CH_CFG_LDOK_SHIFT)) & MPWM_CH_CFG_LDOK_MASK)
#define MPWM_CH_CFG_HWTRIG_MASK                                (0x400U)
#define MPWM_CH_CFG_HWTRIG_SHIFT                               (10U)
#define MPWM_CH_CFG_HWTRIG(x)                                  (((uint32)(((uint32)(x)) << MPWM_CH_CFG_HWTRIG_SHIFT)) & MPWM_CH_CFG_HWTRIG_MASK)
#define MPWM_CH_CFG_CMPTOEN_MASK                               (0x200U)
#define MPWM_CH_CFG_CMPTOEN_SHIFT                              (9U)
#define MPWM_CH_CFG_CMPTOEN(x)                                 (((uint32)(((uint32)(x)) << MPWM_CH_CFG_CMPTOEN_SHIFT)) & MPWM_CH_CFG_CMPTOEN_MASK)
#define MPWM_CH_CFG_PERTOEN_MASK                               (0x100U)
#define MPWM_CH_CFG_PERTOEN_SHIFT                              (8U)
#define MPWM_CH_CFG_PERTOEN(x)                                 (((uint32)(((uint32)(x)) << MPWM_CH_CFG_PERTOEN_SHIFT)) & MPWM_CH_CFG_PERTOEN_MASK)
#define MPWM_CH_CFG_CLKPRS_WIDTH                               (3)
#define MPWM_CH_CFG_CLKPRS_MASK                                (0x70U)
#define MPWM_CH_CFG_CLKPRS_SHIFT                               (4U)
#define MPWM_CH_CFG_CLKPRS(x)                                  (((uint32)(((uint32)(x)) << MPWM_CH_CFG_CLKPRS_SHIFT)) & MPWM_CH_CFG_CLKPRS_MASK)
#define MPWM_CH_CFG_ONESHOT_MASK                               (0x8U)
#define MPWM_CH_CFG_ONESHOT_SHIFT                              (3U)
#define MPWM_CH_CFG_ONESHOT(x)                                 (((uint32)(((uint32)(x)) << MPWM_CH_CFG_ONESHOT_SHIFT)) & MPWM_CH_CFG_ONESHOT_MASK)
#define MPWM_CH_CFG_CHMODE_WIDTH                               (3)
#define MPWM_CH_CFG_CHMODE_MASK                                (0x7U)
#define MPWM_CH_CFG_CHMODE_SHIFT                               (0U)
#define MPWM_CH_CFG_CHMODE(x)                                  (((uint32)(((uint32)(x)) << MPWM_CH_CFG_CHMODE_SHIFT)) & MPWM_CH_CFG_CHMODE_MASK)
/* MPWM_CH_CNT Register */
#define MPWM_CH_CNT_CNT_MASK                                   (0xFFFFU)
#define MPWM_CH_CNT_CNT_SHIFT                                  (0U)
#define MPWM_CH_CNT_CNT(x)                                     (((uint32)(((uint32)(x)) << MPWM_CH_CNT_CNT_SHIFT)) & MPWM_CH_CNT_CNT_MASK)
/* MPWM_CH_CAP Register */
#define MPWM_CH_CAP_CAP_MASK                                   (0xFFFFU)
#define MPWM_CH_CAP_CAP_SHIFT                                  (0U)
#define MPWM_CH_CAP_CAP(x)                                     (((uint32)(((uint32)(x)) << MPWM_CH_CAP_CAP_SHIFT)) & MPWM_CH_CAP_CAP_MASK)
/* MPWM_CH_PERIOD Register */
#define MPWM_CH_PERIOD_PERIOD_MASK                             (0xFFFFU)
#define MPWM_CH_PERIOD_PERIOD_SHIFT                            (0U)
#define MPWM_CH_PERIOD_PERIOD(x)                               (((uint32)(((uint32)(x)) << MPWM_CH_PERIOD_PERIOD_SHIFT)) & MPWM_CH_PERIOD_PERIOD_MASK)
/* MPWM_CH_CMP Register */
#define MPWM_CH_CMP_CMP_MASK                                   (0xFFFFU)
#define MPWM_CH_CMP_CMP_SHIFT                                  (0U)
#define MPWM_CH_CMP_CMP(x)                                     (((uint32)(((uint32)(x)) << MPWM_CH_CMP_CMP_SHIFT)) & MPWM_CH_CMP_CMP_MASK)



/**
 * @}
 */ /* end of group MPWM_Register_Masks */


/**
 * @}
 */ /* end of group MPWM_Peripheral_Access_Layer */

#endif /* PSIP_MPWM_H */

