/**
*   @file    pSIP_Enet.h
*   @version
*
*   @brief   AUTOSAR Enet register map
*   @details Register map for ENET
*
*   @addtogroup ADC_MODULE
*   @{
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Adc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.3.0
*
*   (c) Copyright 2020-2025 YTMicro Semiconductor, Inc.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef PSIP_ENET_H
#define PSIP_ENET_H

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/

#include "Std_Types.h"

/*==================================================================================================
*                              ENET REGISTER MAP
==================================================================================================*/

/**
 * @addtogroup ENET_Peripheral_Access_Layer ENET Peripheral Access Layer
 * @{
 */


/** ENET - Size of Registers Arrays */
#define ENET_MAC_HASH_TABLE_REG_COUNT           (8)
#define ENET_MAC_TX_FLOW_CTRL_Q_COUNT           (2)
#define ENET_MAC_ADDRESS_COUNT           (64)
#define ENET_MAC_EQOS_L3_L4_FILTER_COUNT           (4)
#define ENET_MAC_PPS_COUNT           (4)
#define ENET_MTL_QUEUE_COUNT           (2)
#define ENET_DMA_CH_COUNT           (2)

/* ENET Register Layout Typedef */
typedef struct {
    uint32 MAC_CONFIGURATION;                  /**< MAC Configuration Register, offset: 0x0000 */
    uint32 MAC_EXT_CONFIGURATION;              /**< MAC Extended Configuration Register, offset: 0x0004 */
    uint32 MAC_PACKET_FILTER;                  /**< MAC Packet Filter, offset: 0x0008 */
    uint32 MAC_WATCHDOG_TIMEOUT;               /**< MAC Watchdog Timeout, offset: 0x000C */
    uint32 MAC_HASH_TABLE_REG[ 8];             /**< MAC Hash Table Register, offset: 0x0010 */
    uint32 Reserved1[ 8];                      /**< Reserved1, offset: 0x0030 */
    union {
        uint32 TAG;                                /**< MAC VLAN Tag, offset: 0x0050 */
        uint32 CTRL;                               /**< VLAN Tag Control Register, offset: 0x0050 */

    } MAC_VLAN_TAG;
    uint32 MAC_VLAN_TAG_DATA;                  /**< MAC VLAN Tag Data Register, offset: 0x0054 */
    uint32 MAC_VLAN_HASH_TABLE;                /**< MAC VLAN Hash Table Register, offset: 0x0058 */
    uint32 Reserved2;                          /**< Reserved2, offset: 0x005C */
    uint32 MAC_VLAN_INCL;                      /**< MAC VLAN Inclusion or Exclusion Register, offset: 0x0060 */
    uint32 MAC_INNER_VLAN_INCL;                /**< Inner VLAN inclusion register, offset: 0x0064 */
    uint32 Reserved3[ 2];                      /**< Reserved, offset: 0x0068 */
    uint32 MAC_TX_FLOW_CTRL_Q[ 2];             /**< Transmit flow control register, offset: 0x0070 */
    uint32 Reserved4[ 6];                      /**< Reserved, offset: 0x0078 */
    uint32 MAC_RX_FLOW_CTRL;                   /**< Receive flow control register, offset: 0x0090 */
    uint32 MAC_RXQ_CTRL4;                      /**< Receive Queue Control Register 4, offset: 0x0094 */
    uint32 MAC_TXQ_PRTY_MAP0;                  /**< Transmit Queue Priority Mapping Register, offset: 0x0098 */
    uint32 MAC_TXQ_PRTY_MAP1;                  /**< Reserved, offset: 0x009C */
    uint32 MAC_RXQ_CTRL0;                      /**< Receive Queue Control Register 0, offset: 0x00A0 */
    uint32 MAC_RXQ_CTRL1;                      /**< The Receive Queue Control 1, offset: 0x00A4 */
    uint32 MAC_RXQ_CTRL2;                      /**< Rx Queue Control 2, offset: 0x00A8 */
    uint32 MAC_RXQ_CTRL3;                      /**< , offset: 0x00AC */
    uint32 MAC_INTERRUPT_STATUS;               /**< MAC Interrupt Status, offset: 0x00B0 */
    uint32 MAC_INTERRUPT_ENABLE;               /**< MAC Interrupt Enable Register, offset: 0x00B4 */
    uint32 MAC_RX_TX_STATUS;                   /**< MAC Receive and Transmit Status Register, offset: 0x00B8 */
    uint32 Reserved5;                          /**< Reserved, offset: 0x00BC */
    uint32 MAC_PMT_CONTROL_STATUS;             /**< MAC PMT Control and Status Register, offset: 0x00C0 */
    uint32 MAC_RWK_PACKET_FILTER;              /**< MAC Remote Wake-Up Packet Filter Register, offset: 0x00C4 */
    uint32 Reserved6[ 2];                      /**< Reserved, offset: 0x00C8 */
    uint32 MAC_LPI_CONTROL_STATUS;             /**< MAC LPI Control and Status Register, offset: 0x00D0 */
    uint32 MAC_LPI_TIMERS_CONTROL;             /**< LPI Timers Control, offset: 0x00D4 */
    uint32 MAC_LPI_ENTRY_TIMER;                /**< LPI Entry Timer, offset: 0x00D8 */
    uint32 MAC_1US_TIC_COUNTER;                /**< 1us TIC Counter, offset: 0x00DC */
    uint32 MAC_AN_CONTROL;                     /**< Auto-Negotiation Control, offset: 0x00E0 */
    uint32 MAC_AN_STATUS;                      /**< Auto-Negotiation Status Register, offset: 0x00E4 */
    uint32 MAC_AN_ADVERTISEMENT;               /**< Auto-Negotiation Advertisement Register, offset: 0x00E8 */
    uint32 MAC_AN_LINK_PARTNER_ABILITY;        /**< Auto-Negotiation Link Partner Ability Register, offset: 0x00EC */
    uint32 MAC_AN_EXPANSION;                   /**< Auto-Negotiation Expansion Register, offset: 0x00F0 */
    uint32 MAC_TBI_EXTENDED_STATUS;            /**< TBI Extended Status Register, offset: 0x00F4 */
    uint32 MAC_PHYIF_CONTROL_STATUS;           /**< PHY Interface Control and Status Register, offset: 0x00F8 */
    uint32 Reserved7[ 5];                      /**< Reserved, offset: 0x00FC */
    uint32 MAC_VERSION;                        /**< Version Register, offset: 0x0110 */
    uint32 MAC_DEBUG;                          /**< Debug Register, offset: 0x0114 */
    uint32 Reserved8;                          /**< Reserved, offset: 0x0118 */
    uint32 MAC_HW_FEATURE0;                    /**< Hardware Feature Register 0, offset: 0x011C */
    uint32 MAC_HW_FEATURE1;                    /**< MAC HW Feature 1, offset: 0x0120 */
    uint32 MAC_HW_FEATURE2;                    /**< MAC HW Feature 2 Register, offset: 0x0124 */
    uint32 MAC_HW_FEATURE3;                    /**< MAC HW Feature 3 Register, offset: 0x0128 */
    uint32 Reserved9[ 5];                      /**< , offset: 0x012C */
    uint32 MAC_DPP_FSM_INTERRUPT_STATUS;       /**< , offset: 0x0140 */
    uint32 MAC_AXI_SLV_DPE_ADDR_STATUS;        /**< MAC AXI Slave Data Path Parity Error Address Status, offset: 0x0144 */
    uint32 MAC_FSM_CONTROL;                    /**< MAC FSM Control, offset: 0x0148 */
    uint32 MAC_FSM_ACT_TIMER;                  /**< FSM Action Timer, offset: 0x014C */
    uint32 Reserved10[44];                     /**< Reserved, offset: 0x0150 */
    uint32 MAC_MDIO_ADDRESS;                   /**< MDIO Address, offset: 0x0200 */
    uint32 MAC_MDIO_DATA;                      /**< MDIO Data Register, offset: 0x0204 */
    uint32 MAC_GPIO_CONTROL;                   /**< GPIO Control Register, offset: 0x0208 */
    uint32 MAC_GPIO_STATUS;                    /**< GPIO Status Register, offset: 0x020C */
    uint32 RESERVED_59;                        /**< RESERVED_59, offset: 0x0210 */
    uint32 Reserved11[ 7];                     /**< Reserved, offset: 0x0214 */
    uint32 MAC_CSR_SW_CTRL;                    /**< , offset: 0x0230 */
    uint32 MAC_FPE_CTRL_STS;                   /**< FPE Control and Status Register, offset: 0x0234 */
    uint32 MAC_EXT_CFG1;                       /**< Extended Configuration Register 1, offset: 0x0238 */
    uint32 Reserved12;                         /**< Reserved, offset: 0x023C */
    uint32 MAC_PRESN_TIME_NS;                  /**< Presentation Time Nanoseconds, offset: 0x0240 */
    uint32 MAC_PRESN_TIME_UPDT;                /**< Presentation Time Update, offset: 0x0244 */
    uint32 Reserved13[46];                     /**< Reserved, offset: 0x0248 */
    struct {
        uint32 HIGH;                               /**< MAC Address High, offset: 0x0300 */
        uint32 LOW;                                /**< MAC Address Low, offset: 0x0304 */

    } MAC_ADDRESS[64];
    uint32 Reserved14[128];                     /**< Reserved, offset: 0x0500 */
    uint32 MMC_CONTROL;                        /**< MMC Control, offset: 0x0700 */
    uint32 MMC_RX_INTERRUPT;                   /**< MMC Rx Interrupt, offset: 0x0704 */
    uint32 MMC_TX_INTERRUPT;                   /**< MMC Transmit Interrupt Register, offset: 0x0708 */
    uint32 MMC_RX_INTERRUPT_MASK;              /**< MMC Receive Interrupt Mask Register, offset: 0x070C */
    uint32 MMC_TX_INTERRUPT_MASK;              /**< MMC Tx Interrupt Mask Register, offset: 0x0710 */
    uint32 TX_OCTET_COUNT_GOOD_BAD;            /**< Tx Octet Count Good Bad, offset: 0x0714 */
    uint32 TX_PACKET_COUNT_GOOD_BAD;           /**< Tx Packet Count Good Bad, offset: 0x0718 */
    uint32 TX_BROADCAST_PACKETS_GOOD;          /**< Tx Broadcast Packets Good, offset: 0x071C */
    uint32 TX_MULTICAST_PACKETS_GOOD;          /**< Tx Multicast Packets Good, offset: 0x0720 */
    uint32 TX_64OCTETS_PACKETS_GOOD_BAD;       /**< Tx 64Octets Packets Good Bad, offset: 0x0724 */
    uint32 TX_65TO127OCTETS_PACKETS_GOOD_BAD;  /**< Tx 65to127Octets Packets Good Bad, offset: 0x0728 */
    uint32 TX_128TO255OCTETS_PACKETS_GOOD_BAD; /**< Tx 128to255Octets Packets Good Bad, offset: 0x072C */
    uint32 TX_256TO511OCTETS_PACKETS_GOOD_BAD; /**< Tx 256to511Octets Packets Good Bad, offset: 0x0730 */
    uint32 TX_512TO1023OCTETS_PACKETS_GOOD_BAD; /**< Tx 512to1023Octets Packets Good Bad, offset: 0x0734 */
    uint32 TX_1024TOMAXOCTETS_PACKETS_GOOD_BAD; /**< Tx 1024toMaxOctets Packets Good Bad, offset: 0x0738 */
    uint32 TX_UNICAST_PACKETS_GOOD_BAD;        /**< Tx Unicast Packets Good Bad, offset: 0x073C */
    uint32 TX_MULTICAST_PACKETS_GOOD_BAD;      /**< Tx Multicast Packets Good Bad, offset: 0x0740 */
    uint32 TX_BROADCAST_PACKETS_GOOD_BAD;      /**< Tx Broadcast Packets Good Bad, offset: 0x0744 */
    uint32 TX_UNDERFLOW_ERROR_PACKETS;         /**< Tx Underflow Error Packets, offset: 0x0748 */
    uint32 TX_SINGLE_COLLISION_GOOD_PACKETS;   /**< Tx Single Collision Good Packets, offset: 0x074C */
    uint32 TX_MULTIPLE_COLLISION_GOOD_PACKETS; /**< Tx Multiple Collision Good Packets, offset: 0x0750 */
    uint32 TX_DEFERRED_PACKETS;                /**< Tx Deferred Packets, offset: 0x0754 */
    uint32 TX_LATE_COLLISION_PACKETS;          /**< Tx Late Collision Packets, offset: 0x0758 */
    uint32 TX_EXCESSIVE_COLLISION_PACKETS;     /**< Tx Excessive Collision Packets, offset: 0x075C */
    uint32 TX_CARRIER_ERROR_PACKETS;           /**< Tx Carrier Error Packets, offset: 0x0760 */
    uint32 TX_OCTET_COUNT_GOOD;                /**< Tx Octet Count Good, offset: 0x0764 */
    uint32 TX_PACKET_COUNT_GOOD;               /**< Tx Packet Count Good, offset: 0x0768 */
    uint32 TX_EXCESSIVE_DEFERRAL_ERROR;        /**< Tx Excessive Deferral Error, offset: 0x076C */
    uint32 TX_PAUSE_PACKETS;                   /**< Tx Pause Packets, offset: 0x0770 */
    uint32 TX_VLAN_PACKETS_GOOD;               /**< Tx VLAN Packets Good, offset: 0x0774 */
    uint32 TX_OSIZE_PACKETS_GOOD;              /**< Tx OSize Packets Good, offset: 0x0778 */
    uint32 Reserved15;                         /**< Reserved, offset: 0x077C */
    uint32 RX_PACKETS_COUNT_GOOD_BAD;          /**< Rx Packets Count Good Bad, offset: 0x0780 */
    uint32 RX_OCTET_COUNT_GOOD_BAD;            /**< Rx Octet Count Good Bad, offset: 0x0784 */
    uint32 RX_OCTET_COUNT_GOOD;                /**< Rx Octet Count Good, offset: 0x0788 */
    uint32 RX_BROADCAST_PACKETS_GOOD;          /**< Rx Broadcast Packets Good, offset: 0x078C */
    uint32 RX_MULTICAST_PACKETS_GOOD;          /**< Rx Multicast Packets Good, offset: 0x0790 */
    uint32 RX_CRC_ERROR_PACKETS;               /**< Rx CRC Error Packets, offset: 0x0794 */
    uint32 RX_ALIGNMENT_ERROR_PACKETS;         /**< Rx Alignment Error Packets, offset: 0x0798 */
    uint32 RX_RUNT_ERROR_PACKETS;              /**< Rx Runt Error Packets, offset: 0x079C */
    uint32 RX_JABBER_ERROR_PACKETS;            /**< Rx Jabber Error Packets, offset: 0x07A0 */
    uint32 RX_UNDERSIZE_PACKETS_GOOD;          /**< Rx Undersize Packets Good, offset: 0x07A4 */
    uint32 RX_OVERSIZE_PACKETS_GOOD;           /**< Rx Oversize Packets Good, offset: 0x07A8 */
    uint32 RX_64OCTETS_PACKETS_GOOD_BAD;       /**< Rx 64 Octets Packets Good Bad, offset: 0x07AC */
    uint32 RX_65TO127OCTETS_PACKETS_GOOD_BAD;  /**< Rx 65 to 127 Octets Packets Good Bad, offset: 0x07B0 */
    uint32 RX_128TO255OCTETS_PACKETS_GOOD_BAD; /**< Rx 128 to 255 Octets Packets Good Bad, offset: 0x07B4 */
    uint32 RX_256TO511OCTETS_PACKETS_GOOD_BAD; /**< Rx 256 to 511 Octets Packets Good Bad, offset: 0x07B8 */
    uint32 RX_512TO1023OCTETS_PACKETS_GOOD_BAD; /**< Rx 512 to 1023 Octets Packets Good Bad, offset: 0x07BC */
    uint32 RX_1024TOMAXOCTETS_PACKETS_GOOD_BAD; /**< Rx 1024 to Max Octets Packets Good Bad, offset: 0x07C0 */
    uint32 RX_UNICAST_PACKETS_GOOD;            /**< Rx Unicast Packets Good, offset: 0x07C4 */
    uint32 RX_LENGTH_ERROR_PACKETS;            /**< Rx Length Error Packets, offset: 0x07C8 */
    uint32 RX_OUT_OF_RANGE_TYPE_PACKETS;       /**< Rx Out Of Range Type Packets, offset: 0x07CC */
    uint32 RX_PAUSE_PACKETS;                   /**< Rx Pause Packets, offset: 0x07D0 */
    uint32 RX_FIFO_OVERFLOW_PACKETS;           /**< Rx FIFO Overflow Packets, offset: 0x07D4 */
    uint32 RX_VLAN_PACKETS_GOOD_BAD;           /**< Rx VLAN Packets Good Bad, offset: 0x07D8 */
    uint32 RX_WATCHDOG_ERROR_PACKETS;          /**< Rx Watchdog Error Packets, offset: 0x07DC */
    uint32 RX_RECEIVE_ERROR_PACKETS;           /**< Rx Receive Error Packets, offset: 0x07E0 */
    uint32 RX_CONTROL_PACKETS_GOOD;            /**< Rx Control Packets Good, offset: 0x07E4 */
    uint32 Reserved16;                         /**< Reserved, offset: 0x07E8 */
    uint32 TX_LPI_USEC_CNTR;                   /**< Tx LPI USEC Cntr, offset: 0x07EC */
    uint32 TX_LPI_TRAN_CNTR;                   /**< Tx LPI Tran Cntr, offset: 0x07F0 */
    uint32 RX_LPI_USEC_CNTR;                   /**< Rx LPI USEC Cntr, offset: 0x07F4 */
    uint32 RX_LPI_TRAN_CNTR;                   /**< Rx LPI Tran Cntr, offset: 0x07F8 */
    uint32 Reserved17;                         /**< Reserved, offset: 0x07FC */
    uint32 MMC_IPC_RX_INTERRUPT_MASK;          /**< MMC IPC Rx Interrupt Mask, offset: 0x0800 */
    uint32 Reserved17a;                        /**< Reserved, offset: 0x0804 */
    uint32 MMC_IPC_RX_INTERRUPT;               /**< MMC IPC Receive Interrupt Register, offset: 0x0808 */
    uint32 Reserved17b;                        /**< Reserved, offset: 0x080C */
    uint32 RXIPV4_GOOD_PACKETS;                /**< MMC Receive IPV4 Good Packet Counter, offset: 0x0810 */
    uint32 RXIPV4_HEADER_ERROR_PACKETS;        /**< MMC Receive IPV4 Header Error Packet Counter, offset: 0x0814 */
    uint32 RXIPV4_NO_PAYLOAD_PACKETS;          /**< MMC Receive IPV4 No Payload Packet Counter, offset: 0x0818 */
    uint32 RXIPV4_FRAGMENTED_PACKETS;          /**< MMC Receive IPV4 Fragmented Packet Counter, offset: 0x081C */
    uint32 RXIPV4_UDP_CHECKSUM_DISABLED_PACKETS; /**< MMC Receive IPV4 UDP Checksum Disabled Packet Counter, offset: 0x0820 */
    uint32 RXIPV6_GOOD_PACKETS;                /**< MMC Receive IPV6 Good Packet Counter, offset: 0x0824 */
    uint32 RXIPV6_HEADER_ERROR_PACKETS;        /**< MMC Receive IPV6 Header Error Packet Counter, offset: 0x0828 */
    uint32 RXIPV6_NO_PAYLOAD_PACKETS;          /**< MMC Receive IPV6 No Payload Packet Counter, offset: 0x082C */
    uint32 RXUDP_GOOD_PACKETS;                 /**< MMC Receive UDP Good Packet Counter, offset: 0x0830 */
    uint32 RXUDP_ERROR_PACKETS;                /**< MMC Receive UDP Error Packet Counter, offset: 0x0834 */
    uint32 RXTCP_GOOD_PACKETS;                 /**< MMC Receive TCP Good Packet Counter, offset: 0x0838 */
    uint32 RXTCP_ERROR_PACKETS;                /**< MMC Receive TCP Error Packet Counter, offset: 0x083C */
    uint32 RXICMP_GOOD_PACKETS;                /**< MMC Receive ICMP Good Packet Counter, offset: 0x0840 */
    uint32 RXICMP_ERROR_PACKETS;               /**< MMC Receive ICMP Error Packet Counter, offset: 0x0844 */
    uint32 Reserved18[ 2];                     /**< , offset: 0x0848 */
    uint32 RXIPV4_GOOD_OCTETS;                 /**< MMC Receive IPV4 Good Octet Counter, offset: 0x0850 */
    uint32 RXIPV4_HEADER_ERROR_OCTETS;         /**< MMC Receive IPV4 Header Error Octet Counter, offset: 0x0854 */
    uint32 RXIPV4_NOPAYLOAD_OCTETS;            /**< MMC Receive IPV4 No Payload Octet Counter, offset: 0x0858 */
    uint32 RXIPV4_FRAGMENTED_OCTETS;           /**< MMC Receive IPV4 Fragmented Octet Counter, offset: 0x085C */
    uint32 RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS; /**< MMC Receive IPV4 UDP Checksum Disabled Octet Counter, offset: 0x0860 */
    uint32 RXIPV6_GOOD_OCTETS;                 /**< MMC Receive IPV6 Good Octet Counter, offset: 0x0864 */
    uint32 RXIPV6_HEADER_ERROR_OCTETS;         /**< MMC Receive IPV6 Header Error Octet Counter, offset: 0x0868 */
    uint32 RXIPV6_NO_PAYLOAD_OCTETS;           /**< MMC Receive IPV6 No Payload Octet Counter, offset: 0x086C */
    uint32 RXUDP_GOOD_OCTETS;                  /**< MMC Receive UDP Good Octet Counter, offset: 0x0870 */
    uint32 RXUDP_ERROR_OCTETS;                 /**< MMC Receive UDP Error Octet Counter, offset: 0x0874 */
    uint32 RXTCP_GOOD_OCTETS;                  /**< MMC Receive TCP Good Octet Counter, offset: 0x0878 */
    uint32 RXTCP_ERROR_OCTETS;                 /**< MMC Receive TCP Error Octet Counter, offset: 0x087C */
    uint32 RXICMP_GOOD_OCTETS;                 /**< MMC Receive ICMP Good Octet Counter, offset: 0x0880 */
    uint32 RXICMP_ERROR_OCTETS;                /**< MMC Receive ICMP Error Octet Counter, offset: 0x0884 */
    uint32 Reserved19[ 6];                     /**< , offset: 0x0888 */
    uint32 MMC_FPE_TX_INTERRUPT;               /**< MMC PFE Tx Interrupt, offset: 0x08A0 */
    uint32 MMC_FPE_TX_INTERRUPT_MASK;          /**< MMC FPE Tx Interrupt Mask, offset: 0x08A4 */
    uint32 MMC_TX_FPE_FRAGMENT_CNTR;           /**< MMC Transmit FPE Fragment Counter, offset: 0x08A8 */
    uint32 MMC_TX_HOLD_REQ_CNTR;               /**< MMC Transmit Hold Request Counter, offset: 0x08AC */
    uint32 Reserved20[ 4];                     /**< , offset: 0x08B0 */
    uint32 MMC_FPE_RX_INTERRUPT;               /**< MMC FPE Rx Interrupt, offset: 0x08C0 */
    uint32 MMC_FPE_RX_INTERRUPT_MASK;          /**< MMC FPE Rx Interrupt Mask, offset: 0x08C4 */
    uint32 MMC_RX_PACKET_ASSEMBLY_ERR_CNTR;    /**< MMC Rx Packet Assembly Error Counter, offset: 0x08C8 */
    uint32 MMC_RX_PACKET_SMD_ERR_CNTR;         /**< MMC Rx Packet SMD Error Counter, offset: 0x08CC */
    uint32 MMC_RX_PACKET_ASSEMBLY_OK_CNTR;     /**< MMC Rx Packet Assembly OK Counter, offset: 0x08D0 */
    uint32 MMC_RX_FPE_FRAGMENT_CNTR;           /**< MMC Rx FPE Fragment Counter, offset: 0x08D4 */
    uint32 Reserved21[10];                     /**< Reserved, offset: 0x08D8 */
    struct {
        uint32 CONTROL;                            /**< MAC EQOS L3 L4 Filter Control, offset: 0x0900 */
        uint32 LAYER4_ADDRESS;                     /**< Layer 4 Address Register, offset: 0x0904 */
    uint32 Reserved22[ 2];                     /**< Reserved, offset: 0x0908 */
    uint32 LAYER3_ADDR[ 4];                    /**< Layer 3 Address Register, offset: 0x0910 */
    uint32 Reserved22a[ 4];                    /**< Reserved, offset: 0x0920 */

    } MAC_EQOS_L3_L4_FILTER[4];
    uint32 Reserved23[44];                     /**< Reserved, offset: 0x09C0 */
    uint32 MAC_INDIR_ACCESS_CTRL;              /**< Indirect Access Control Register, offset: 0x0A70 */
    uint32 MAC_TMRQ_REGS;                      /**< TMRQ register, offset: 0x0A74 */
    uint32 reserved22b[34];                    /**< reserved, offset: 0x0A78 */
    uint32 MAC_TIMESTAMP_CONTROL;              /**< Timestamp Control Register, offset: 0x0B00 */
    uint32 MAC_SUB_SECOND_INCREMENT;           /**< Sub-Second Increment Register, offset: 0x0B04 */
    uint32 MAC_SYSTEM_TIME_SECONDS;            /**< System Time Seconds Register, offset: 0x0B08 */
    uint32 MAC_SYSTEM_TIME_NANOSECONDS;        /**< System Time Nanoseconds Register, offset: 0x0B0C */
    uint32 MAC_SYSTEM_TIME_SECONDS_UPDATE;     /**< System Time Seconds Update Register, offset: 0x0B10 */
    uint32 MAC_SYSTEM_TIME_NANOSECONDS_UPDATE; /**< System Time Nanoseconds Update Register, offset: 0x0B14 */
    uint32 MAC_TIMESTAMP_ADDEND;               /**< Timestamp Addend Register, offset: 0x0B18 */
    uint32 MAC_SYSTEM_TIME_HIGHER_WORD_SECONDS; /**< System Time Higher Word Seconds Register, offset: 0x0B1C */
    uint32 MAC_TIMESTAMP_STATUS;               /**< Timestamp Status Register, offset: 0x0B20 */
    uint32 Reserved23a[ 3];                    /**< , offset: 0x0B38 */
    uint32 MAC_TX_TIMESTAMP_STATUS_NANOSECONDS; /**< Tx Timestamp Status Nanoseconds, offset: 0x0B30 */
    uint32 MAC_TX_TIMESTAMP_STATUS_SECONDS;    /**< Tx Timestamp Status Seconds, offset: 0x0B34 */
    uint32 Reserved24[ 2];                     /**< Reserved, offset: 0x0B38 */
    uint32 MAC_AUXILIARY_CONTROL;              /**< MAC Auxiliary Control, offset: 0x0B40 */
    uint32 Reserved25;                         /**< Reserved, offset: 0x0B44 */
    uint32 MAC_AUXILIARY_TIMESTAMP_NANOSECONDS; /**< Auxiliary Timestamp Nanoseconds, offset: 0x0B48 */
    uint32 MAC_AUXILIARY_TIMESTAMP_SECONDS;    /**< Auxiliary Timestamp Seconds, offset: 0x0B4C */
    uint32 MAC_TIMESTAMP_INGRESS_ASYM_CORR;    /**< Timestamp Ingress Asymmetry Correction, offset: 0x0B50 */
    uint32 MAC_TIMESTAMP_EGRESS_ASYM_CORR;     /**< Timestamp Egress Asymmetry Correction, offset: 0x0B54 */
    uint32 MAC_TIMESTAMP_INGRESS_CORR_NANOSECOND; /**< Timestamp Ingress Correction Nanoseconds, offset: 0x0B58 */
    uint32 MAC_TIMESTAMP_EGRESS_CORR_NANOSECOND; /**< Timestamp Egress Correction Nanoseconds, offset: 0x0B5C */
    uint32 MAC_TIMESTAMP_INGRESS_CORR_SUBNANOSEC; /**< Timestamp Ingress Correction Subnanoseconds, offset: 0x0B60 */
    uint32 MAC_TIMESTAMP_EGRESS_CORR_SUBNANOSEC; /**< Timestamp Egress Correction Subnanoseconds, offset: 0x0B64 */
    uint32 MAC_TIMESTAMP_INGRESS_LATENCY;      /**< Timestamp Ingress Latency, offset: 0x0B68 */
    uint32 MAC_TIMESTAMP_EGRESS_LATENCY;       /**< Timestamp Egress Latency, offset: 0x0B6C */
    uint32 MAC_PPS_CONTROL;                    /**< PPS Control, offset: 0x0B70 */
    uint32 Reserved26[ 3];                     /**< Reserved, offset: 0x0B74 */
    struct {
        uint32 TARGET_TIME_SECONDS;                /**< PPS Target Time Seconds, offset: 0x0B80 */
        uint32 TARGET_TIME_NANOSECONDS;            /**< PPS Target Time Nanoseconds, offset: 0x0B84 */
        uint32 INTERVAL;                           /**< PPS Interval, offset: 0x0B88 */
        uint32 WIDTH;                              /**< PPS Width, offset: 0x0B8C */

    } MAC_PPS[4];
    uint32 MAC_PTO_CONTROL;                    /**< MAC PTO Control Register, offset: 0x0BC0 */
    uint32 MAC_SOURCE_PORT_IDENTITY;           /**< MAC Source Port Identity Register, offset: 0x0BC4 */
    uint32 Reserved27a[ 2];                    /**< Reserved, offset: 0x0BC8 */
    uint32 MAC_LOG_MESSAGE_INTERVAL;           /**< MAC Log Message Interval Register, offset: 0x0BD0 */
    uint32 Reserved27[11];                     /**< Reserved, offset: 0x0BD4 */
    uint32 MTL_OPERATION_MODE;                 /**< MTL Operation Mode Register, offset: 0x0C00 */
    uint32 Reserved28;                         /**< Reserved, offset: 0x0C04 */
    uint32 MTL_DBG_CTL;                        /**< MTL Debug Control Register, offset: 0x0C08 */
    uint32 MTL_DBG_STS;                        /**< MTL Debug Status Register, offset: 0x0C0C */
    uint32 MTL_DBG_DATA;                       /**< MTL Debug Data Register, offset: 0x0C10 */
    uint32 Reserved29[ 3];                     /**< Reserved, offset: 0x0C10 */
    uint32 MTL_INTERRUPT_STATUS;               /**< MTL Interrupt Status Register, offset: 0x0C20 */
    uint32 Reserved30[ 3];                     /**< Reserved, offset: 0x0C24 */
    uint32 MTL_RXQ_DMA_MAP0;                   /**< MTL Rx Queue DMA Map Register, offset: 0x0C30 */
    uint32 MTL_RXQ_DMA_MAP1;                   /**< MTL RxQ DMA Map Register 1, offset: 0x0C34 */
    uint32 Reserved31[ 2];                     /**< Reserved, offset: 0x0C38 */
    uint32 MTL_TBS_CTRL;                       /**< Time Based Scheduling Control Register, offset: 0x0C40 */
    uint32 Reserved32[ 3];                     /**< Reserved, offset: 0x0C44 */
    uint32 MTL_EST_CONTROL;                    /**< EST Control Register, offset: 0x0C50 */
    uint32 MTL_EST_EXT_CONTROL;                /**< EST Extend Control Register, offset: 0x0C54 */
    uint32 MTL_EST_STATUS;                     /**< EST Status Register, offset: 0x0C58 */
    uint32 Reserved33;                         /**< Reserved, offset: 0x0C5C */
    uint32 MTL_EST_SCH_ERROR;                  /**< EST Scheduling Error Register, offset: 0x0C60 */
    uint32 MTL_EST_FRM_SIZE_ERROR;             /**< EST Frame Size Error Register, offset: 0x0C64 */
    uint32 MTL_EST_FRM_SIZE_CAPTURE;           /**< EST Frame Size Capture Register, offset: 0x0C68 */
    uint32 Reserved34;                         /**< Reserved, offset: 0x0C6C */
    uint32 MTL_EST_INTR_ENABLE;                /**< EST Interrupt Enable Register, offset: 0x0C70 */
    uint32 Reserved35[ 3];                     /**< Reserved, offset: 0x0C74 */
    uint32 MTL_EST_GCL_CONTROL;                /**< EST GCL Control Register, offset: 0x0C80 */
    uint32 MTL_EST_GCL_DATA;                   /**< EST GCL Data Register, offset: 0x0C84 */
    uint32 Reserved36[ 2];                     /**< Reserved, offset: 0x0C88 */
    uint32 MTL_FPE_CTRL_STS;                   /**< MTL FPE Control and Status Register, offset: 0x0C90 */
    uint32 MTL_FPE_ADVANCE;                    /**< MTL FPE Advance Register, offset: 0x0C94 */
    uint32 Reserved37[ 2];                     /**< Reserved, offset: 0x0C98 */
    uint32 MTL_RXP_CONTROL_STATUS;             /**< MTL RX Parser Control and Status Register, offset: 0x0CA0 */
    uint32 MTL_RXP_INTERRUPT_CONTROL_STATUS;   /**< MTL RX Parser Interrupt Control and Status Register, offset: 0x0CA4 */
    uint32 MTL_RXP_DROP_CNT;                   /**< MTL RX Parser Drop Count Register, offset: 0x0CA8 */
    uint32 MTL_RXP_ERROR_CNT;                  /**< MTL RX Parser Error Count Register, offset: 0x0CAC */
    uint32 MTL_RXP_INDIRECT_ACC_CONTROL_STATUS; /**< MTL RX Parser Indirect Access Control and Status Register, offset: 0x0CB0 */
    uint32 MTL_RXP_INDIRECT_ACC_DATA;          /**< MTL RX Parser Indirect Access Data Register, offset: 0x0CB4 */
    uint32 MTL_RXP_BYPASS_CNT;                 /**< MTL RX Parser Bypass Count Register, offset: 0x0CB8 */
    uint32 Reserved38;                         /**< Reserved, offset: 0x0CBC */
    uint32 MTL_ECC_CONTROL;                    /**< MTL ECC Control Register, offset: 0x0CC0 */
    uint32 MTL_SAFETY_INTERRUPT_STATUS;        /**< MTL Safety Interrupt Status Register, offset: 0x0CC4 */
    uint32 MTL_ECC_INTERRUPT_ENABLE;           /**< MTL ECC Interrupt Enable Register, offset: 0x0CC8 */
    uint32 MTL_ECC_INTERRUPT_STATUS;           /**< MTL ECC Interrupt Status Register, offset: 0x0CCC */
    uint32 MTL_ECC_ERR_STS_RCTL;               /**< MTL ECC Error Status Register, offset: 0x0CD0 */
    uint32 MTL_ECC_ERR_ADDR_STATUS;            /**< MTL ECC Error Address Status Register, offset: 0x0CD4 */
    uint32 MTL_ECC_ERR_CNTR_STATUS;            /**< MTL ECC Error Counter Status Register, offset: 0x0CD8 */
    uint32 Reserved40;                         /**< Reserved, offset: 0x0CDC */
    uint32 MTL_DPP_CONTROL;                    /**< MTL DPP Control Register, offset: 0x0CE0 */
    uint32 MTL_DPP_ECC_EIC;                    /**< MTL DPP ECC EIC Register, offset: 0x0CE4 */
    uint32 Reserved4A[ 6];                     /**< Reserved, offset: 0x0CE8 */
    struct {
        uint32 TX_OPERATION_MODE;                  /**< TX Operation Mode, offset: 0x0D00 */
        uint32 TX_UNDERFLOW;                       /**< TX Underflow Packet Counter, offset: 0x0D04 */
        uint32 TX_DEBUG;                           /**< TX Debug Register, offset: 0x0D08 */
    uint32 Reserved41[ 1];                     /**< Reserved, offset: 0x0D0C */
        uint32 TX_ETS_CONTROL;                     /**< TX ETS Control Register, offset: 0x0D10 */
        uint32 TX_ETS_STATUS;                      /**< TX ETS Status Register, offset: 0x0D14 */
        uint32 TX_QUANTUM_WEIGHT;                  /**< TX Quantum Weight Register, offset: 0x0D18 */
        uint32 SENDSLOPECREDIT;                    /**< Send Slope Credit, offset: 0x0D1C */
        uint32 HICREDIT;                           /**< High Credit, offset: 0x0D20 */
        uint32 LOCREDIT;                           /**< Low Credit, offset: 0x0D24 */
    uint32 Reserved42[ 1];                     /**< Reserved, offset: 0x0D28 */
        uint32 INTERRUPT_CONTROL_STATUS;           /**< Interrupt Control Status Register, offset: 0x0D2C */
        uint32 RX_OPERATION_MODE;                  /**< RX Operation Mode Register, offset: 0x0D30 */
        uint32 RX_MISSED_PACKET_OVERFLOW_CNT;      /**< RX Missed Packet Overflow Counter, offset: 0x0D34 */
        uint32 RX_DEBUG;                           /**< RX Debug, offset: 0x0D38 */
        uint32 RX_CONTROL;                         /**< RX Control, offset: 0x0D3C */

    } MTL_QUEUE[2];
    uint32 Reserved50[160];                     /**< Reserved, offset: 0x0D40 */
    uint32 DMA_MODE;                           /**< DMA Mode, offset: 0x1000 */
    uint32 DMA_SYSBUS_MODE;                    /**< DMA System Bus Mode, offset: 0x1004 */
    uint32 DMA_INTERRUPT_STATUS;               /**< DMA Interrupt Status Register, offset: 0x1008 */
    uint32 DMA_DEBUG_STATUS0;                  /**< DMA Debug Status Register 0, offset: 0x100C */
    uint32 DMA_DEBUG_STATUS1;                  /**< DMA Debug Status Register 1, offset: 0x1010 */
    uint32 DMA_DEBUG_STATUS2;                  /**< DMA Debug Status 2, offset: 0x1014 */
    uint32 Reserved51[ 2];                     /**< Reserved, offset: 0x1018 */
    uint32 AXI4_TX_AR_ACE_CONTROL;             /**< AXI4 Tx AR ACE Control, offset: 0x1020 */
    uint32 AXI4_RX_AW_ACE_CONTROL;             /**< AXI4 Rx AW ACE Control, offset: 0x1024 */
    uint32 AXI4_TXRX_AWAR_ACE_CONTROL;         /**< AXI4 TxRx AWAR ACE Control, offset: 0x1028 */
    uint32 Reserved52[ 5];                     /**< Reserved, offset: 0x1028 */
    uint32 AXI_LPI_ENTRY_INTERVAL;             /**< AXI LPI Entry Interval, offset: 0x1040 */
    uint32 Reserved53[ 3];                     /**< Reserved, offset: 0x1044 */
    uint32 DMA_TBS_CTRL;                       /**< DMA TBS Control, offset: 0x1050 */
    uint32 Reserved54[11];                     /**< Reserved, offset: 0x1054 */
    uint32 DMA_SAFETY_INTERRUPT_STATUS;        /**< DMA Safety Interrupt Status, offset: 0x1080 */
    uint32 DMA_ECC_INTERRUPT_ENABLE;           /**< DMA ECC Interrupt Enable, offset: 0x1084 */
    uint32 DMA_ECC_INTERRUPT_STATUS;           /**< DMA ECC Interrupt Status, offset: 0x1088 */
    uint32 Reserved60[29];                     /**< Reserved, offset: 0x1088 */
    struct {
        uint32 CONTROL;                            /**< Control, offset: 0x1100 */
        uint32 TX_CONTROL;                         /**< Transmit Control, offset: 0x1104 */
        uint32 RX_CONTROL;                         /**< Receive Control Register, offset: 0x1108 */
        uint32 Reserved61;                         /**< Reserved, offset: 0x110C */
        uint32 TXDESC_LIST_HADDRESS;               /**< Transmit Descriptor List High Address Register, offset: 0x1110 */
        uint32 TXDESC_LIST_ADDRESS;                /**< Transmit Descriptor List Address Register, offset: 0x1114 */
        uint32 RXDESC_LIST_HADDRESS;               /**< Receive Descriptor List High Address Register, offset: 0x1118 */
        uint32 RXDESC_LIST_ADDRESS;                /**< Receive Descriptor List Address Register, offset: 0x111C */
        uint32 TXDESC_TAIL_POINTER;                /**< Transmit Descriptor Tail Pointer Register, offset: 0x1120 */
        uint32 Reserved62;                         /**< Reserved, offset: 0x1124 */
        uint32 RXDESC_TAIL_POINTER;                /**< Receive Descriptor Tail Pointer Register, offset: 0x1128 */
        uint32 TXDESC_RING_LENGTH;                 /**< Transmit Descriptor Ring Length Register, offset: 0x112C */
        uint32 RXDESC_RING_LENGTH;                 /**< Receive Descriptor Ring Length Register, offset: 0x1130 */
        uint32 INTERRUPT_ENABLE;                   /**< Interrupt Enable Register, offset: 0x1134 */
        uint32 RX_INTERRUPT_WATCHDOG_TIMER;        /**< Rx Interrupt Watchdog Timer, offset: 0x1138 */
        uint32 SLOT_FUNCTION_CONTROL_STATUS;       /**< Slot Function Control/Status, offset: 0x113C */
        uint32 Reserved63;                         /**< Reserved, offset: 0x1040 */
        uint32 CURRENT_APP_TXDESC;                 /**< Current Application Transmit Descriptor, offset: 0x1144 */
        uint32 Reserved64;                         /**< Reserved, offset: 0x1048 */
        uint32 CURRENT_APP_RXDESC;                 /**< Current Application Receive Descriptor, offset: 0x114C */
        uint32 CURRENT_APP_TXBUFFER_H;             /**< Current Application Transmit Buffer Address High, offset: 0x1150 */
        uint32 CURRENT_APP_TXBUFFER;               /**< Current Application Transmit Buffer Address Low, offset: 0x1154 */
        uint32 Current_App_RxBuffer_H;             /**< Current Application Receive Buffer Address High, offset: 0x1158 */
        uint32 CURRENT_APP_RXBUFFER;               /**< Current Application Receive Buffer Address Low, offset: 0x115C */
        uint32 STATUS;                             /**< Status, offset: 0x1160 */
        uint32 MISS_FRAME_CNT;                     /**< Miss Frame Counter, offset: 0x1164 */
        uint32 RXP_ACCEPT_CNT;                     /**< Receive Packet Accepted Counter, offset: 0x1168 */
        uint32 RX_ERI_CNT;                         /**< Receive Error Counter, offset: 0x116C */
    uint32 Reserved65[ 4];                     /**< Reserved, offset: 0x1170 */

    } DMA_CH[2];

} ENET_Type, *ENET_MemMapPtr;

/** Number of instances of the ENET module. */
#define ENET_INSTANCE_COUNT                        (1u)

/* ENET base address */
#define ENET_BASE_ADDR32                                  (0x401E4000U)
#define ENET                                              ((volatile ENET_Type *)(ENET_BASE_ADDR32))

/** Array initializer of ENET peripheral base addresses */
#define ENET_BASE_ADDRS                                    { ENET_BASE_ADDR32 }
/** Array initializer of ENET peripheral base pointers */
#define ENET_BASE_PTRS                                     { ENET }

#define ENET_MAC_CONFIGURATION_OFFSET32                    (0x0000U)                    /**< Offset for MAC Configuration Register */
#define ENET_MAC_EXT_CONFIGURATION_OFFSET32                (0x0004U)                    /**< Offset for MAC Extended Configuration Register */
#define ENET_MAC_PACKET_FILTER_OFFSET32                    (0x0008U)                    /**< Offset for MAC Packet Filter */
#define ENET_MAC_WATCHDOG_TIMEOUT_OFFSET32                 (0x000CU)                    /**< Offset for MAC Watchdog Timeout */
#define ENET_MAC_HASH_TABLE_REG_OFFSET32(x)                (0x0010U + ((x) * (4U)))     /**< Offset for MAC Hash Table Register */
#define ENET_MAC_VLAN_TAG_OFFSET32                         (0x0050U)                    /**< Offset for MAC VLAN Tag */
#define ENET_MAC_VLAN_TAG_TAG_OFFSET32                     (0x0050U)                    /**< Offset for MAC VLAN Tag */
#define ENET_MAC_VLAN_TAG_CTRL_OFFSET32                    (0x0050U)                    /**< Offset for VLAN Tag Control Register */


#define ENET_MAC_VLAN_TAG_DATA_OFFSET32                    (0x0054U)                    /**< Offset for MAC VLAN Tag Data Register */
#define ENET_MAC_VLAN_HASH_TABLE_OFFSET32                  (0x0058U)                    /**< Offset for MAC VLAN Hash Table Register */
#define ENET_MAC_VLAN_INCL_OFFSET32                        (0x0060U)                    /**< Offset for MAC VLAN Inclusion or Exclusion Register */
#define ENET_MAC_INNER_VLAN_INCL_OFFSET32                  (0x0064U)                    /**< Offset for Inner VLAN inclusion register */
#define ENET_MAC_TX_FLOW_CTRL_Q_OFFSET32(x)                (0x0070U + ((x) * (4U)))     /**< Offset for Transmit flow control register */
#define ENET_MAC_RX_FLOW_CTRL_OFFSET32                     (0x0090U)                    /**< Offset for Receive flow control register */
#define ENET_MAC_RXQ_CTRL4_OFFSET32                        (0x0094U)                    /**< Offset for Receive Queue Control Register 4 */
#define ENET_MAC_TXQ_PRTY_MAP0_OFFSET32                    (0x0098U)                    /**< Offset for Transmit Queue Priority Mapping Register */
#define ENET_MAC_RXQ_CTRL0_OFFSET32                        (0x00A0U)                    /**< Offset for Receive Queue Control Register 0 */
#define ENET_MAC_RXQ_CTRL1_OFFSET32                        (0x00A4U)                    /**< Offset for The Receive Queue Control 1 */
#define ENET_MAC_RXQ_CTRL2_OFFSET32                        (0x00A8U)                    /**< Offset for Rx Queue Control 2 */
#define ENET_MAC_INTERRUPT_STATUS_OFFSET32                 (0x00B0U)                    /**< Offset for MAC Interrupt Status */
#define ENET_MAC_INTERRUPT_ENABLE_OFFSET32                 (0x00B4U)                    /**< Offset for MAC Interrupt Enable Register */
#define ENET_MAC_RX_TX_STATUS_OFFSET32                     (0x00B8U)                    /**< Offset for MAC Receive and Transmit Status Register */
#define ENET_MAC_PMT_CONTROL_STATUS_OFFSET32               (0x00C0U)                    /**< Offset for MAC PMT Control and Status Register */
#define ENET_MAC_RWK_PACKET_FILTER_OFFSET32                (0x00C4U)                    /**< Offset for MAC Remote Wake-Up Packet Filter Register */
#define ENET_MAC_LPI_CONTROL_STATUS_OFFSET32               (0x00D0U)                    /**< Offset for MAC LPI Control and Status Register */
#define ENET_MAC_LPI_TIMERS_CONTROL_OFFSET32               (0x00D4U)                    /**< Offset for LPI Timers Control */
#define ENET_MAC_LPI_ENTRY_TIMER_OFFSET32                  (0x00D8U)                    /**< Offset for LPI Entry Timer */
#define ENET_MAC_1US_TIC_COUNTER_OFFSET32                  (0x00DCU)                    /**< Offset for 1us TIC Counter */
#define ENET_MAC_AN_CONTROL_OFFSET32                       (0x00E0U)                    /**< Offset for Auto-Negotiation Control */
#define ENET_MAC_AN_STATUS_OFFSET32                        (0x00E4U)                    /**< Offset for Auto-Negotiation Status Register */
#define ENET_MAC_AN_ADVERTISEMENT_OFFSET32                 (0x00E8U)                    /**< Offset for Auto-Negotiation Advertisement Register */
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_OFFSET32          (0x00ECU)                    /**< Offset for Auto-Negotiation Link Partner Ability Register */
#define ENET_MAC_AN_EXPANSION_OFFSET32                     (0x00F0U)                    /**< Offset for Auto-Negotiation Expansion Register */
#define ENET_MAC_TBI_EXTENDED_STATUS_OFFSET32              (0x00F4U)                    /**< Offset for TBI Extended Status Register */
#define ENET_MAC_PHYIF_CONTROL_STATUS_OFFSET32             (0x00F8U)                    /**< Offset for PHY Interface Control and Status Register */
#define ENET_MAC_VERSION_OFFSET32                          (0x0110U)                    /**< Offset for Version Register */
#define ENET_MAC_DEBUG_OFFSET32                            (0x0114U)                    /**< Offset for Debug Register */
#define ENET_MAC_HW_FEATURE0_OFFSET32                      (0x011CU)                    /**< Offset for Hardware Feature Register 0 */
#define ENET_MAC_HW_FEATURE1_OFFSET32                      (0x0120U)                    /**< Offset for MAC HW Feature 1 */
#define ENET_MAC_HW_FEATURE2_OFFSET32                      (0x0124U)                    /**< Offset for MAC HW Feature 2 Register */
#define ENET_MAC_HW_FEATURE3_OFFSET32                      (0x0128U)                    /**< Offset for MAC HW Feature 3 Register */
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_OFFSET32         (0x0140U)                    /**< Offset for  */
#define ENET_MAC_AXI_SLV_DPE_ADDR_STATUS_OFFSET32          (0x0144U)                    /**< Offset for MAC AXI Slave Data Path Parity Error Address Status */
#define ENET_MAC_FSM_CONTROL_OFFSET32                      (0x0148U)                    /**< Offset for MAC FSM Control */
#define ENET_MAC_FSM_ACT_TIMER_OFFSET32                    (0x014CU)                    /**< Offset for FSM Action Timer */
#define ENET_MAC_MDIO_ADDRESS_OFFSET32                     (0x0200U)                    /**< Offset for MDIO Address */
#define ENET_MAC_MDIO_DATA_OFFSET32                        (0x0204U)                    /**< Offset for MDIO Data Register */
#define ENET_MAC_CSR_SW_CTRL_OFFSET32                      (0x0230U)                    /**< Offset for  */
#define ENET_MAC_FPE_CTRL_STS_OFFSET32                     (0x0234U)                    /**< Offset for FPE Control and Status Register */
#define ENET_MAC_EXT_CFG1_OFFSET32                         (0x0238U)                    /**< Offset for Extended Configuration Register 1 */
#define ENET_MAC_PRESN_TIME_NS_OFFSET32                    (0x0240U)                    /**< Offset for Presentation Time Nanoseconds */
#define ENET_MAC_PRESN_TIME_UPDT_OFFSET32                  (0x0244U)                    /**< Offset for Presentation Time Update */
#define ENET_MAC_ADDRESS_OFFSET32(x)                       (0x0300U + ((x) * (8U)))    /**< Offset for MAC Address */

#define ENET_MAC_ADDRESS_HIGH_OFFSET32(x)                  (0x0300U + ((x) * (8U)))    /**< Offset for MAC Address High */
#define ENET_MAC_ADDRESS_LOW_OFFSET32(x)                   (0x0304U + ((x) * (8U)))    /**< Offset for MAC Address Low */

#define ENET_MMC_CONTROL_OFFSET32                          (0x0700U)                    /**< Offset for MMC Control */
#define ENET_MMC_RX_INTERRUPT_OFFSET32                     (0x0704U)                    /**< Offset for MMC Rx Interrupt */
#define ENET_MMC_TX_INTERRUPT_OFFSET32                     (0x0708U)                    /**< Offset for MMC Transmit Interrupt Register */
#define ENET_MMC_RX_INTERRUPT_MASK_OFFSET32                (0x070CU)                    /**< Offset for MMC Receive Interrupt Mask Register */
#define ENET_MMC_TX_INTERRUPT_MASK_OFFSET32                (0x0710U)                    /**< Offset for MMC Tx Interrupt Mask Register */
#define ENET_TX_OCTET_COUNT_GOOD_BAD_OFFSET32              (0x0714U)                    /**< Offset for Tx Octet Count Good Bad */
#define ENET_TX_PACKET_COUNT_GOOD_BAD_OFFSET32             (0x0718U)                    /**< Offset for Tx Packet Count Good Bad */
#define ENET_TX_BROADCAST_PACKETS_GOOD_OFFSET32            (0x071CU)                    /**< Offset for Tx Broadcast Packets Good */
#define ENET_TX_MULTICAST_PACKETS_GOOD_OFFSET32            (0x0720U)                    /**< Offset for Tx Multicast Packets Good */
#define ENET_TX_64OCTETS_PACKETS_GOOD_BAD_OFFSET32         (0x0724U)                    /**< Offset for Tx 64Octets Packets Good Bad */
#define ENET_TX_65TO127OCTETS_PACKETS_GOOD_BAD_OFFSET32    (0x0728U)                    /**< Offset for Tx 65to127Octets Packets Good Bad */
#define ENET_TX_128TO255OCTETS_PACKETS_GOOD_BAD_OFFSET32   (0x072CU)                    /**< Offset for Tx 128to255Octets Packets Good Bad */
#define ENET_TX_256TO511OCTETS_PACKETS_GOOD_BAD_OFFSET32   (0x0730U)                    /**< Offset for Tx 256to511Octets Packets Good Bad */
#define ENET_TX_512TO1023OCTETS_PACKETS_GOOD_BAD_OFFSET32  (0x0734U)                    /**< Offset for Tx 512to1023Octets Packets Good Bad */
#define ENET_TX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_OFFSET32  (0x0738U)                    /**< Offset for Tx 1024toMaxOctets Packets Good Bad */
#define ENET_TX_UNICAST_PACKETS_GOOD_BAD_OFFSET32          (0x073CU)                    /**< Offset for Tx Unicast Packets Good Bad */
#define ENET_TX_MULTICAST_PACKETS_GOOD_BAD_OFFSET32        (0x0740U)                    /**< Offset for Tx Multicast Packets Good Bad */
#define ENET_TX_BROADCAST_PACKETS_GOOD_BAD_OFFSET32        (0x0744U)                    /**< Offset for Tx Broadcast Packets Good Bad */
#define ENET_TX_UNDERFLOW_ERROR_PACKETS_OFFSET32           (0x0748U)                    /**< Offset for Tx Underflow Error Packets */
#define ENET_TX_SINGLE_COLLISION_GOOD_PACKETS_OFFSET32     (0x074CU)                    /**< Offset for Tx Single Collision Good Packets */
#define ENET_TX_MULTIPLE_COLLISION_GOOD_PACKETS_OFFSET32   (0x0750U)                    /**< Offset for Tx Multiple Collision Good Packets */
#define ENET_TX_DEFERRED_PACKETS_OFFSET32                  (0x0754U)                    /**< Offset for Tx Deferred Packets */
#define ENET_TX_LATE_COLLISION_PACKETS_OFFSET32            (0x0758U)                    /**< Offset for Tx Late Collision Packets */
#define ENET_TX_EXCESSIVE_COLLISION_PACKETS_OFFSET32       (0x075CU)                    /**< Offset for Tx Excessive Collision Packets */
#define ENET_TX_CARRIER_ERROR_PACKETS_OFFSET32             (0x0760U)                    /**< Offset for Tx Carrier Error Packets */
#define ENET_TX_OCTET_COUNT_GOOD_OFFSET32                  (0x0764U)                    /**< Offset for Tx Octet Count Good */
#define ENET_TX_PACKET_COUNT_GOOD_OFFSET32                 (0x0768U)                    /**< Offset for Tx Packet Count Good */
#define ENET_TX_EXCESSIVE_DEFERRAL_ERROR_OFFSET32          (0x076CU)                    /**< Offset for Tx Excessive Deferral Error */
#define ENET_TX_PAUSE_PACKETS_OFFSET32                     (0x0770U)                    /**< Offset for Tx Pause Packets */
#define ENET_TX_VLAN_PACKETS_GOOD_OFFSET32                 (0x0774U)                    /**< Offset for Tx VLAN Packets Good */
#define ENET_TX_OSIZE_PACKETS_GOOD_OFFSET32                (0x0778U)                    /**< Offset for Tx OSize Packets Good */
#define ENET_RX_PACKETS_COUNT_GOOD_BAD_OFFSET32            (0x0780U)                    /**< Offset for Rx Packets Count Good Bad */
#define ENET_RX_OCTET_COUNT_GOOD_BAD_OFFSET32              (0x0784U)                    /**< Offset for Rx Octet Count Good Bad */
#define ENET_RX_OCTET_COUNT_GOOD_OFFSET32                  (0x0788U)                    /**< Offset for Rx Octet Count Good */
#define ENET_RX_BROADCAST_PACKETS_GOOD_OFFSET32            (0x078CU)                    /**< Offset for Rx Broadcast Packets Good */
#define ENET_RX_MULTICAST_PACKETS_GOOD_OFFSET32            (0x0790U)                    /**< Offset for Rx Multicast Packets Good */
#define ENET_RX_CRC_ERROR_PACKETS_OFFSET32                 (0x0794U)                    /**< Offset for Rx CRC Error Packets */
#define ENET_RX_ALIGNMENT_ERROR_PACKETS_OFFSET32           (0x0798U)                    /**< Offset for Rx Alignment Error Packets */
#define ENET_RX_RUNT_ERROR_PACKETS_OFFSET32                (0x079CU)                    /**< Offset for Rx Runt Error Packets */
#define ENET_RX_JABBER_ERROR_PACKETS_OFFSET32              (0x07A0U)                    /**< Offset for Rx Jabber Error Packets */
#define ENET_RX_UNDERSIZE_PACKETS_GOOD_OFFSET32            (0x07A4U)                    /**< Offset for Rx Undersize Packets Good */
#define ENET_RX_OVERSIZE_PACKETS_GOOD_OFFSET32             (0x07A8U)                    /**< Offset for Rx Oversize Packets Good */
#define ENET_RX_64OCTETS_PACKETS_GOOD_BAD_OFFSET32         (0x07ACU)                    /**< Offset for Rx 64 Octets Packets Good Bad */
#define ENET_RX_65TO127OCTETS_PACKETS_GOOD_BAD_OFFSET32    (0x07B0U)                    /**< Offset for Rx 65 to 127 Octets Packets Good Bad */
#define ENET_RX_128TO255OCTETS_PACKETS_GOOD_BAD_OFFSET32   (0x07B4U)                    /**< Offset for Rx 128 to 255 Octets Packets Good Bad */
#define ENET_RX_256TO511OCTETS_PACKETS_GOOD_BAD_OFFSET32   (0x07B8U)                    /**< Offset for Rx 256 to 511 Octets Packets Good Bad */
#define ENET_RX_512TO1023OCTETS_PACKETS_GOOD_BAD_OFFSET32  (0x07BCU)                    /**< Offset for Rx 512 to 1023 Octets Packets Good Bad */
#define ENET_RX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_OFFSET32  (0x07C0U)                    /**< Offset for Rx 1024 to Max Octets Packets Good Bad */
#define ENET_RX_UNICAST_PACKETS_GOOD_OFFSET32              (0x07C4U)                    /**< Offset for Rx Unicast Packets Good */
#define ENET_RX_LENGTH_ERROR_PACKETS_OFFSET32              (0x07C8U)                    /**< Offset for Rx Length Error Packets */
#define ENET_RX_OUT_OF_RANGE_TYPE_PACKETS_OFFSET32         (0x07CCU)                    /**< Offset for Rx Out Of Range Type Packets */
#define ENET_RX_PAUSE_PACKETS_OFFSET32                     (0x07D0U)                    /**< Offset for Rx Pause Packets */
#define ENET_RX_FIFO_OVERFLOW_PACKETS_OFFSET32             (0x07D4U)                    /**< Offset for Rx FIFO Overflow Packets */
#define ENET_RX_VLAN_PACKETS_GOOD_BAD_OFFSET32             (0x07D8U)                    /**< Offset for Rx VLAN Packets Good Bad */
#define ENET_RX_WATCHDOG_ERROR_PACKETS_OFFSET32            (0x07DCU)                    /**< Offset for Rx Watchdog Error Packets */
#define ENET_RX_RECEIVE_ERROR_PACKETS_OFFSET32             (0x07E0U)                    /**< Offset for Rx Receive Error Packets */
#define ENET_RX_CONTROL_PACKETS_GOOD_OFFSET32              (0x07E4U)                    /**< Offset for Rx Control Packets Good */
#define ENET_TX_LPI_USEC_CNTR_OFFSET32                     (0x07ECU)                    /**< Offset for Tx LPI USEC Cntr */
#define ENET_TX_LPI_TRAN_CNTR_OFFSET32                     (0x07F0U)                    /**< Offset for Tx LPI Tran Cntr */
#define ENET_RX_LPI_USEC_CNTR_OFFSET32                     (0x07F4U)                    /**< Offset for Rx LPI USEC Cntr */
#define ENET_RX_LPI_TRAN_CNTR_OFFSET32                     (0x07F8U)                    /**< Offset for Rx LPI Tran Cntr */
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_OFFSET32            (0x0800U)                    /**< Offset for MMC IPC Rx Interrupt Mask */
#define ENET_MMC_IPC_RX_INTERRUPT_OFFSET32                 (0x0808U)                    /**< Offset for MMC IPC Receive Interrupt Register */
#define ENET_RXIPV4_GOOD_PACKETS_OFFSET32                  (0x0810U)                    /**< Offset for MMC Receive IPV4 Good Packet Counter */
#define ENET_RXIPV4_HEADER_ERROR_PACKETS_OFFSET32          (0x0814U)                    /**< Offset for MMC Receive IPV4 Header Error Packet Counter */
#define ENET_RXIPV4_NO_PAYLOAD_PACKETS_OFFSET32            (0x0818U)                    /**< Offset for MMC Receive IPV4 No Payload Packet Counter */
#define ENET_RXIPV4_FRAGMENTED_PACKETS_OFFSET32            (0x081CU)                    /**< Offset for MMC Receive IPV4 Fragmented Packet Counter */
#define ENET_RXIPV4_UDP_CHECKSUM_DISABLED_PACKETS_OFFSET32 (0x0820U)                    /**< Offset for MMC Receive IPV4 UDP Checksum Disabled Packet Counter */
#define ENET_RXIPV6_GOOD_PACKETS_OFFSET32                  (0x0824U)                    /**< Offset for MMC Receive IPV6 Good Packet Counter */
#define ENET_RXIPV6_HEADER_ERROR_PACKETS_OFFSET32          (0x0828U)                    /**< Offset for MMC Receive IPV6 Header Error Packet Counter */
#define ENET_RXIPV6_NO_PAYLOAD_PACKETS_OFFSET32            (0x082CU)                    /**< Offset for MMC Receive IPV6 No Payload Packet Counter */
#define ENET_RXUDP_GOOD_PACKETS_OFFSET32                   (0x0830U)                    /**< Offset for MMC Receive UDP Good Packet Counter */
#define ENET_RXUDP_ERROR_PACKETS_OFFSET32                  (0x0834U)                    /**< Offset for MMC Receive UDP Error Packet Counter */
#define ENET_RXTCP_GOOD_PACKETS_OFFSET32                   (0x0838U)                    /**< Offset for MMC Receive TCP Good Packet Counter */
#define ENET_RXTCP_ERROR_PACKETS_OFFSET32                  (0x083CU)                    /**< Offset for MMC Receive TCP Error Packet Counter */
#define ENET_RXICMP_GOOD_PACKETS_OFFSET32                  (0x0840U)                    /**< Offset for MMC Receive ICMP Good Packet Counter */
#define ENET_RXICMP_ERROR_PACKETS_OFFSET32                 (0x0844U)                    /**< Offset for MMC Receive ICMP Error Packet Counter */
#define ENET_RXIPV4_GOOD_OCTETS_OFFSET32                   (0x0850U)                    /**< Offset for MMC Receive IPV4 Good Octet Counter */
#define ENET_RXIPV4_HEADER_ERROR_OCTETS_OFFSET32           (0x0854U)                    /**< Offset for MMC Receive IPV4 Header Error Octet Counter */
#define ENET_RXIPV4_NOPAYLOAD_OCTETS_OFFSET32              (0x0858U)                    /**< Offset for MMC Receive IPV4 No Payload Octet Counter */
#define ENET_RXIPV4_FRAGMENTED_OCTETS_OFFSET32             (0x085CU)                    /**< Offset for MMC Receive IPV4 Fragmented Octet Counter */
#define ENET_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_OFFSET32   (0x0860U)                    /**< Offset for MMC Receive IPV4 UDP Checksum Disabled Octet Counter */
#define ENET_RXIPV6_GOOD_OCTETS_OFFSET32                   (0x0864U)                    /**< Offset for MMC Receive IPV6 Good Octet Counter */
#define ENET_RXIPV6_HEADER_ERROR_OCTETS_OFFSET32           (0x0868U)                    /**< Offset for MMC Receive IPV6 Header Error Octet Counter */
#define ENET_RXIPV6_NO_PAYLOAD_OCTETS_OFFSET32             (0x086CU)                    /**< Offset for MMC Receive IPV6 No Payload Octet Counter */
#define ENET_RXUDP_GOOD_OCTETS_OFFSET32                    (0x0870U)                    /**< Offset for MMC Receive UDP Good Octet Counter */
#define ENET_RXUDP_ERROR_OCTETS_OFFSET32                   (0x0874U)                    /**< Offset for MMC Receive UDP Error Octet Counter */
#define ENET_RXTCP_GOOD_OCTETS_OFFSET32                    (0x0878U)                    /**< Offset for MMC Receive TCP Good Octet Counter */
#define ENET_RXTCP_ERROR_OCTETS_OFFSET32                   (0x087CU)                    /**< Offset for MMC Receive TCP Error Octet Counter */
#define ENET_RXICMP_GOOD_OCTETS_OFFSET32                   (0x0880U)                    /**< Offset for MMC Receive ICMP Good Octet Counter */
#define ENET_RXICMP_ERROR_OCTETS_OFFSET32                  (0x0884U)                    /**< Offset for MMC Receive ICMP Error Octet Counter */
#define ENET_MMC_FPE_TX_INTERRUPT_OFFSET32                 (0x08A0U)                    /**< Offset for MMC PFE Tx Interrupt */
#define ENET_MMC_FPE_TX_INTERRUPT_MASK_OFFSET32            (0x08A4U)                    /**< Offset for MMC FPE Tx Interrupt Mask */
#define ENET_MMC_TX_FPE_FRAGMENT_CNTR_OFFSET32             (0x08A8U)                    /**< Offset for MMC Transmit FPE Fragment Counter */
#define ENET_MMC_TX_HOLD_REQ_CNTR_OFFSET32                 (0x08ACU)                    /**< Offset for MMC Transmit Hold Request Counter */
#define ENET_MMC_FPE_RX_INTERRUPT_OFFSET32                 (0x08C0U)                    /**< Offset for MMC FPE Rx Interrupt */
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_OFFSET32            (0x08C4U)                    /**< Offset for MMC FPE Rx Interrupt Mask */
#define ENET_MMC_RX_PACKET_ASSEMBLY_ERR_CNTR_OFFSET32      (0x08C8U)                    /**< Offset for MMC Rx Packet Assembly Error Counter */
#define ENET_MMC_RX_PACKET_SMD_ERR_CNTR_OFFSET32           (0x08CCU)                    /**< Offset for MMC Rx Packet SMD Error Counter */
#define ENET_MMC_RX_PACKET_ASSEMBLY_OK_CNTR_OFFSET32       (0x08D0U)                    /**< Offset for MMC Rx Packet Assembly OK Counter */
#define ENET_MMC_RX_FPE_FRAGMENT_CNTR_OFFSET32             (0x08D4U)                    /**< Offset for MMC Rx FPE Fragment Counter */
#define ENET_MAC_EQOS_L3_L4_FILTER_OFFSET32(x)             (0x0900U + ((x) * (48U)))    /**< Offset for MAC EQOS L3 L4 Filter */

#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_OFFSET32(x)     (0x0900U + ((x) * (48U)))    /**< Offset for MAC EQOS L3 L4 Filter Control */
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_OFFSET32(x) (0x0904U + ((x) * (48U)))    /**< Offset for Layer 4 Address Register */
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER3_ADDR_OFFSET32(x) (0x0910U + ((x) * (52U)))     /**< Offset for Layer 3 Address Register */

#define ENET_MAC_INDIR_ACCESS_CTRL_OFFSET32                (0x0A70U)                    /**< Offset for Indirect Access Control Register */
#define ENET_MAC_TMRQ_REGS_OFFSET32                        (0x0A74U)                    /**< Offset for TMRQ register */
#define ENET_MAC_TIMESTAMP_CONTROL_OFFSET32                (0x0B00U)                    /**< Offset for Timestamp Control Register */
#define ENET_MAC_SUB_SECOND_INCREMENT_OFFSET32             (0x0B04U)                    /**< Offset for Sub-Second Increment Register */
#define ENET_MAC_SYSTEM_TIME_SECONDS_OFFSET32              (0x0B08U)                    /**< Offset for System Time Seconds Register */
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_OFFSET32          (0x0B0CU)                    /**< Offset for System Time Nanoseconds Register */
#define ENET_MAC_SYSTEM_TIME_SECONDS_UPDATE_OFFSET32       (0x0B10U)                    /**< Offset for System Time Seconds Update Register */
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_OFFSET32   (0x0B14U)                    /**< Offset for System Time Nanoseconds Update Register */
#define ENET_MAC_TIMESTAMP_ADDEND_OFFSET32                 (0x0B18U)                    /**< Offset for Timestamp Addend Register */
#define ENET_MAC_SYSTEM_TIME_HIGHER_WORD_SECONDS_OFFSET32  (0x0B1CU)                    /**< Offset for System Time Higher Word Seconds Register */
#define ENET_MAC_TIMESTAMP_STATUS_OFFSET32                 (0x0B20U)                    /**< Offset for Timestamp Status Register */
#define ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_OFFSET32  (0x0B30U)                    /**< Offset for Tx Timestamp Status Nanoseconds */
#define ENET_MAC_TX_TIMESTAMP_STATUS_SECONDS_OFFSET32      (0x0B34U)                    /**< Offset for Tx Timestamp Status Seconds */
#define ENET_MAC_AUXILIARY_CONTROL_OFFSET32                (0x0B40U)                    /**< Offset for MAC Auxiliary Control */
#define ENET_MAC_AUXILIARY_TIMESTAMP_NANOSECONDS_OFFSET32  (0x0B48U)                    /**< Offset for Auxiliary Timestamp Nanoseconds */
#define ENET_MAC_AUXILIARY_TIMESTAMP_SECONDS_OFFSET32      (0x0B4CU)                    /**< Offset for Auxiliary Timestamp Seconds */
#define ENET_MAC_TIMESTAMP_INGRESS_ASYM_CORR_OFFSET32      (0x0B50U)                    /**< Offset for Timestamp Ingress Asymmetry Correction */
#define ENET_MAC_TIMESTAMP_EGRESS_ASYM_CORR_OFFSET32       (0x0B54U)                    /**< Offset for Timestamp Egress Asymmetry Correction */
#define ENET_MAC_TIMESTAMP_INGRESS_CORR_NANOSECOND_OFFSET32 (0x0B58U)                    /**< Offset for Timestamp Ingress Correction Nanoseconds */
#define ENET_MAC_TIMESTAMP_EGRESS_CORR_NANOSECOND_OFFSET32 (0x0B5CU)                    /**< Offset for Timestamp Egress Correction Nanoseconds */
#define ENET_MAC_TIMESTAMP_INGRESS_CORR_SUBNANOSEC_OFFSET32 (0x0B60U)                    /**< Offset for Timestamp Ingress Correction Subnanoseconds */
#define ENET_MAC_TIMESTAMP_EGRESS_CORR_SUBNANOSEC_OFFSET32 (0x0B64U)                    /**< Offset for Timestamp Egress Correction Subnanoseconds */
#define ENET_MAC_TIMESTAMP_INGRESS_LATENCY_OFFSET32        (0x0B68U)                    /**< Offset for Timestamp Ingress Latency */
#define ENET_MAC_TIMESTAMP_EGRESS_LATENCY_OFFSET32         (0x0B6CU)                    /**< Offset for Timestamp Egress Latency */
#define ENET_MAC_PPS_CONTROL_OFFSET32                      (0x0B70U)                    /**< Offset for PPS Control */
#define ENET_MAC_PPS_OFFSET32(x)                           (0x0B80U + ((x) * (16U)))    /**< Offset for MAC PPS Control Register */

#define ENET_MAC_PPS_TARGET_TIME_SECONDS_OFFSET32(x)       (0x0B80U + ((x) * (16U)))    /**< Offset for PPS Target Time Seconds */
#define ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_OFFSET32(x)   (0x0B84U + ((x) * (16U)))    /**< Offset for PPS Target Time Nanoseconds */
#define ENET_MAC_PPS_INTERVAL_OFFSET32(x)                  (0x0B88U + ((x) * (16U)))    /**< Offset for PPS Interval */
#define ENET_MAC_PPS_WIDTH_OFFSET32(x)                     (0x0B8CU + ((x) * (16U)))    /**< Offset for PPS Width */

#define ENET_MAC_PTO_CONTROL_OFFSET32                      (0x0BC0U)                    /**< Offset for MAC PTO Control Register */
#define ENET_MAC_SOURCE_PORT_IDENTITY_OFFSET32             (0x0BC4U)                    /**< Offset for MAC Source Port Identity Register */
#define ENET_MAC_LOG_MESSAGE_INTERVAL_OFFSET32             (0x0BD0U)                    /**< Offset for MAC Log Message Interval Register */
#define ENET_MTL_OPERATION_MODE_OFFSET32                   (0x0C00U)                    /**< Offset for MTL Operation Mode Register */
#define ENET_MTL_DBG_CTL_OFFSET32                          (0x0C08U)                    /**< Offset for MTL Debug Control Register */
#define ENET_MTL_DBG_STS_OFFSET32                          (0x0C0CU)                    /**< Offset for MTL Debug Status Register */
#define ENET_MTL_DBG_DATA_OFFSET32                         (0x0C10U)                    /**< Offset for MTL Debug Data Register */
#define ENET_MTL_INTERRUPT_STATUS_OFFSET32                 (0x0C20U)                    /**< Offset for MTL Interrupt Status Register */
#define ENET_MTL_RXQ_DMA_MAP0_OFFSET32                     (0x0C30U)                    /**< Offset for MTL Rx Queue DMA Map Register */
#define ENET_MTL_RXQ_DMA_MAP1_OFFSET32                     (0x0C34U)                    /**< Offset for MTL RxQ DMA Map Register 1 */
#define ENET_MTL_TBS_CTRL_OFFSET32                         (0x0C40U)                    /**< Offset for Time Based Scheduling Control Register */
#define ENET_MTL_EST_CONTROL_OFFSET32                      (0x0C50U)                    /**< Offset for EST Control Register */
#define ENET_MTL_EST_EXT_CONTROL_OFFSET32                  (0x0C54U)                    /**< Offset for EST Extend Control Register */
#define ENET_MTL_EST_STATUS_OFFSET32                       (0x0C58U)                    /**< Offset for EST Status Register */
#define ENET_MTL_EST_SCH_ERROR_OFFSET32                    (0x0C60U)                    /**< Offset for EST Scheduling Error Register */
#define ENET_MTL_EST_FRM_SIZE_ERROR_OFFSET32               (0x0C64U)                    /**< Offset for EST Frame Size Error Register */
#define ENET_MTL_EST_FRM_SIZE_CAPTURE_OFFSET32             (0x0C68U)                    /**< Offset for EST Frame Size Capture Register */
#define ENET_MTL_EST_INTR_ENABLE_OFFSET32                  (0x0C70U)                    /**< Offset for EST Interrupt Enable Register */
#define ENET_MTL_EST_GCL_CONTROL_OFFSET32                  (0x0C80U)                    /**< Offset for EST GCL Control Register */
#define ENET_MTL_EST_GCL_DATA_OFFSET32                     (0x0C84U)                    /**< Offset for EST GCL Data Register */
#define ENET_MTL_FPE_CTRL_STS_OFFSET32                     (0x0C90U)                    /**< Offset for MTL FPE Control and Status Register */
#define ENET_MTL_FPE_ADVANCE_OFFSET32                      (0x0C94U)                    /**< Offset for MTL FPE Advance Register */
#define ENET_MTL_RXP_CONTROL_STATUS_OFFSET32               (0x0CA0U)                    /**< Offset for MTL RX Parser Control and Status Register */
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_OFFSET32     (0x0CA4U)                    /**< Offset for MTL RX Parser Interrupt Control and Status Register */
#define ENET_MTL_RXP_DROP_CNT_OFFSET32                     (0x0CA8U)                    /**< Offset for MTL RX Parser Drop Count Register */
#define ENET_MTL_RXP_ERROR_CNT_OFFSET32                    (0x0CACU)                    /**< Offset for MTL RX Parser Error Count Register */
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_OFFSET32  (0x0CB0U)                    /**< Offset for MTL RX Parser Indirect Access Control and Status Register */
#define ENET_MTL_RXP_INDIRECT_ACC_DATA_OFFSET32            (0x0CB4U)                    /**< Offset for MTL RX Parser Indirect Access Data Register */
#define ENET_MTL_RXP_BYPASS_CNT_OFFSET32                   (0x0CB8U)                    /**< Offset for MTL RX Parser Bypass Count Register */
#define ENET_MTL_ECC_CONTROL_OFFSET32                      (0x0CC0U)                    /**< Offset for MTL ECC Control Register */
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_OFFSET32          (0x0CC4U)                    /**< Offset for MTL Safety Interrupt Status Register */
#define ENET_MTL_ECC_INTERRUPT_ENABLE_OFFSET32             (0x0CC8U)                    /**< Offset for MTL ECC Interrupt Enable Register */
#define ENET_MTL_ECC_INTERRUPT_STATUS_OFFSET32             (0x0CCCU)                    /**< Offset for MTL ECC Interrupt Status Register */
#define ENET_MTL_ECC_ERR_STS_RCTL_OFFSET32                 (0x0CD0U)                    /**< Offset for MTL ECC Error Status Register */
#define ENET_MTL_ECC_ERR_ADDR_STATUS_OFFSET32              (0x0CD4U)                    /**< Offset for MTL ECC Error Address Status Register */
#define ENET_MTL_ECC_ERR_CNTR_STATUS_OFFSET32              (0x0CD8U)                    /**< Offset for MTL ECC Error Counter Status Register */
#define ENET_MTL_DPP_CONTROL_OFFSET32                      (0x0CE0U)                    /**< Offset for MTL DPP Control Register */
#define ENET_MTL_DPP_ECC_EIC_OFFSET32                      (0x0CE4U)                    /**< Offset for MTL DPP ECC EIC Register */
#define ENET_MTL_QUEUE_OFFSET32(x)                         (0x0D00U + ((x) * (64U)))    /**< Offset for MTL Queue Array */

#define ENET_MTL_QUEUE_TX_OPERATION_MODE_OFFSET32(x)       (0x0D00U + ((x) * (64U)))    /**< Offset for TX Operation Mode */
#define ENET_MTL_QUEUE_TX_UNDERFLOW_OFFSET32(x)            (0x0D04U + ((x) * (64U)))    /**< Offset for TX Underflow Packet Counter */
#define ENET_MTL_QUEUE_TX_DEBUG_OFFSET32(x)                (0x0D08U + ((x) * (64U)))    /**< Offset for TX Debug Register */
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_OFFSET32(x)          (0x0D10U + ((x) * (64U)))    /**< Offset for TX ETS Control Register */
#define ENET_MTL_QUEUE_TX_ETS_STATUS_OFFSET32(x)           (0x0D14U + ((x) * (64U)))    /**< Offset for TX ETS Status Register */
#define ENET_MTL_QUEUE_TX_QUANTUM_WEIGHT_OFFSET32(x)       (0x0D18U + ((x) * (64U)))    /**< Offset for TX Quantum Weight Register */
#define ENET_MTL_QUEUE_SENDSLOPECREDIT_OFFSET32(x)         (0x0D1CU + ((x) * (64U)))    /**< Offset for Send Slope Credit */
#define ENET_MTL_QUEUE_HICREDIT_OFFSET32(x)                (0x0D20U + ((x) * (64U)))    /**< Offset for High Credit */
#define ENET_MTL_QUEUE_LOCREDIT_OFFSET32(x)                (0x0D24U + ((x) * (64U)))    /**< Offset for Low Credit */
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_OFFSET32(x) (0x0D2CU + ((x) * (64U)))    /**< Offset for Interrupt Control Status Register */
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_OFFSET32(x)       (0x0D30U + ((x) * (64U)))    /**< Offset for RX Operation Mode Register */
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OFFSET32(x) (0x0D34U + ((x) * (64U)))    /**< Offset for RX Missed Packet Overflow Counter */
#define ENET_MTL_QUEUE_RX_DEBUG_OFFSET32(x)                (0x0D38U + ((x) * (64U)))    /**< Offset for RX Debug */
#define ENET_MTL_QUEUE_RX_CONTROL_OFFSET32(x)              (0x0D3CU + ((x) * (64U)))    /**< Offset for RX Control */

#define ENET_DMA_MODE_OFFSET32                             (0x1000U)                    /**< Offset for DMA Mode */
#define ENET_DMA_SYSBUS_MODE_OFFSET32                      (0x1004U)                    /**< Offset for DMA System Bus Mode */
#define ENET_DMA_INTERRUPT_STATUS_OFFSET32                 (0x1008U)                    /**< Offset for DMA Interrupt Status Register */
#define ENET_DMA_DEBUG_STATUS0_OFFSET32                    (0x100CU)                    /**< Offset for DMA Debug Status Register 0 */
#define ENET_DMA_DEBUG_STATUS1_OFFSET32                    (0x1010U)                    /**< Offset for DMA Debug Status Register 1 */
#define ENET_DMA_DEBUG_STATUS2_OFFSET32                    (0x1014U)                    /**< Offset for DMA Debug Status 2 */
#define ENET_AXI4_TX_AR_ACE_CONTROL_OFFSET32               (0x1020U)                    /**< Offset for AXI4 Tx AR ACE Control */
#define ENET_AXI4_RX_AW_ACE_CONTROL_OFFSET32               (0x1024U)                    /**< Offset for AXI4 Rx AW ACE Control */
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_OFFSET32           (0x1028U)                    /**< Offset for AXI4 TxRx AWAR ACE Control */
#define ENET_AXI_LPI_ENTRY_INTERVAL_OFFSET32               (0x1040U)                    /**< Offset for AXI LPI Entry Interval */
#define ENET_DMA_TBS_CTRL_OFFSET32                         (0x1050U)                    /**< Offset for DMA TBS Control */
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_OFFSET32          (0x1080U)                    /**< Offset for DMA Safety Interrupt Status */
#define ENET_DMA_ECC_INTERRUPT_ENABLE_OFFSET32             (0x1084U)                    /**< Offset for DMA ECC Interrupt Enable */
#define ENET_DMA_ECC_INTERRUPT_STATUS_OFFSET32             (0x1088U)                    /**< Offset for DMA ECC Interrupt Status */
#define ENET_DMA_CH_OFFSET32(x)                            (0x1100U + ((x) * (128U)))    /**< Offset for DMA Channel */

#define ENET_DMA_CH_CONTROL_OFFSET32(x)                    (0x1100U + ((x) * (128U)))    /**< Offset for Control */
#define ENET_DMA_CH_TX_CONTROL_OFFSET32(x)                 (0x1104U + ((x) * (128U)))    /**< Offset for Transmit Control */
#define ENET_DMA_CH_RX_CONTROL_OFFSET32(x)                 (0x1108U + ((x) * (128U)))    /**< Offset for Receive Control Register */
#define ENET_DMA_CH_TXDESC_LIST_HADDRESS_OFFSET32(x)       (0x1110U + ((x) * (128U)))    /**< Offset for Transmit Descriptor List High Address Register */
#define ENET_DMA_CH_TXDESC_LIST_ADDRESS_OFFSET32(x)        (0x1114U + ((x) * (128U)))    /**< Offset for Transmit Descriptor List Address Register */
#define ENET_DMA_CH_RXDESC_LIST_HADDRESS_OFFSET32(x)       (0x1118U + ((x) * (128U)))    /**< Offset for Receive Descriptor List High Address Register */
#define ENET_DMA_CH_RXDESC_LIST_ADDRESS_OFFSET32(x)        (0x111CU + ((x) * (128U)))    /**< Offset for Receive Descriptor List Address Register */
#define ENET_DMA_CH_TXDESC_TAIL_POINTER_OFFSET32(x)        (0x1120U + ((x) * (128U)))    /**< Offset for Transmit Descriptor Tail Pointer Register */
#define ENET_DMA_CH_RXDESC_TAIL_POINTER_OFFSET32(x)        (0x1128U + ((x) * (128U)))    /**< Offset for Receive Descriptor Tail Pointer Register */
#define ENET_DMA_CH_TXDESC_RING_LENGTH_OFFSET32(x)         (0x112CU + ((x) * (128U)))    /**< Offset for Transmit Descriptor Ring Length Register */
#define ENET_DMA_CH_RXDESC_RING_LENGTH_OFFSET32(x)         (0x1130U + ((x) * (128U)))    /**< Offset for Receive Descriptor Ring Length Register */
#define ENET_DMA_CH_INTERRUPT_ENABLE_OFFSET32(x)           (0x1134U + ((x) * (128U)))    /**< Offset for Interrupt Enable Register */
#define ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_OFFSET32(x) (0x1138U + ((x) * (128U)))    /**< Offset for Rx Interrupt Watchdog Timer */
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_OFFSET32(x) (0x113CU + ((x) * (128U)))    /**< Offset for Slot Function Control/Status */
#define ENET_DMA_CH_CURRENT_APP_TXDESC_OFFSET32(x)         (0x1144U + ((x) * (128U)))    /**< Offset for Current Application Transmit Descriptor */
#define ENET_DMA_CH_CURRENT_APP_RXDESC_OFFSET32(x)         (0x114CU + ((x) * (128U)))    /**< Offset for Current Application Receive Descriptor */
#define ENET_DMA_CH_CURRENT_APP_TXBUFFER_H_OFFSET32(x)     (0x1150U + ((x) * (128U)))    /**< Offset for Current Application Transmit Buffer Address High */
#define ENET_DMA_CH_CURRENT_APP_TXBUFFER_OFFSET32(x)       (0x1154U + ((x) * (128U)))    /**< Offset for Current Application Transmit Buffer Address Low */
#define ENET_DMA_CH_Current_App_RxBuffer_H_OFFSET32(x)     (0x1158U + ((x) * (128U)))    /**< Offset for Current Application Receive Buffer Address High */
#define ENET_DMA_CH_CURRENT_APP_RXBUFFER_OFFSET32(x)       (0x115CU + ((x) * (128U)))    /**< Offset for Current Application Receive Buffer Address Low */
#define ENET_DMA_CH_STATUS_OFFSET32(x)                     (0x1160U + ((x) * (128U)))    /**< Offset for Status */
#define ENET_DMA_CH_MISS_FRAME_CNT_OFFSET32(x)             (0x1164U + ((x) * (128U)))    /**< Offset for Miss Frame Counter */
#define ENET_DMA_CH_RXP_ACCEPT_CNT_OFFSET32(x)             (0x1168U + ((x) * (128U)))    /**< Offset for Receive Packet Accepted Counter */
#define ENET_DMA_CH_RX_ERI_CNT_OFFSET32(x)                 (0x116CU + ((x) * (128U)))    /**< Offset for Receive Error Counter */



/**
 * @addtogroup ENET Register Mask
 * @{
 */
/* ENET_MAC_CONFIGURATION Register */
#define ENET_MAC_CONFIGURATION_ARPEN_MASK                     (0x80000000U)
#define ENET_MAC_CONFIGURATION_ARPEN_SHIFT                    (31U)
#define ENET_MAC_CONFIGURATION_ARPEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_ARPEN_SHIFT)) & ENET_MAC_CONFIGURATION_ARPEN_MASK)
#define ENET_MAC_CONFIGURATION_SARC_MASK                      (0x70000000U)
#define ENET_MAC_CONFIGURATION_SARC_SHIFT                     (28U)
#define ENET_MAC_CONFIGURATION_SARC(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_SARC_SHIFT)) & ENET_MAC_CONFIGURATION_SARC_MASK)
#define ENET_MAC_CONFIGURATION_IPC_MASK                       (0x8000000U)
#define ENET_MAC_CONFIGURATION_IPC_SHIFT                      (27U)
#define ENET_MAC_CONFIGURATION_IPC(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_IPC_SHIFT)) & ENET_MAC_CONFIGURATION_IPC_MASK)
#define ENET_MAC_CONFIGURATION_IPG_MASK                       (0x7000000U)
#define ENET_MAC_CONFIGURATION_IPG_SHIFT                      (24U)
#define ENET_MAC_CONFIGURATION_IPG(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_IPG_SHIFT)) & ENET_MAC_CONFIGURATION_IPG_MASK)
#define ENET_MAC_CONFIGURATION_GPSLCE_MASK                    (0x800000U)
#define ENET_MAC_CONFIGURATION_GPSLCE_SHIFT                   (23U)
#define ENET_MAC_CONFIGURATION_GPSLCE(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_GPSLCE_SHIFT)) & ENET_MAC_CONFIGURATION_GPSLCE_MASK)
#define ENET_MAC_CONFIGURATION_S2KP_MASK                      (0x400000U)
#define ENET_MAC_CONFIGURATION_S2KP_SHIFT                     (22U)
#define ENET_MAC_CONFIGURATION_S2KP(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_S2KP_SHIFT)) & ENET_MAC_CONFIGURATION_S2KP_MASK)
#define ENET_MAC_CONFIGURATION_CST_MASK                       (0x200000U)
#define ENET_MAC_CONFIGURATION_CST_SHIFT                      (21U)
#define ENET_MAC_CONFIGURATION_CST(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_CST_SHIFT)) & ENET_MAC_CONFIGURATION_CST_MASK)
#define ENET_MAC_CONFIGURATION_ACS_MASK                       (0x100000U)
#define ENET_MAC_CONFIGURATION_ACS_SHIFT                      (20U)
#define ENET_MAC_CONFIGURATION_ACS(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_ACS_SHIFT)) & ENET_MAC_CONFIGURATION_ACS_MASK)
#define ENET_MAC_CONFIGURATION_WD_MASK                        (0x80000U)
#define ENET_MAC_CONFIGURATION_WD_SHIFT                       (19U)
#define ENET_MAC_CONFIGURATION_WD(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_WD_SHIFT)) & ENET_MAC_CONFIGURATION_WD_MASK)
#define ENET_MAC_CONFIGURATION_BE_MASK                        (0x40000U)
#define ENET_MAC_CONFIGURATION_BE_SHIFT                       (18U)
#define ENET_MAC_CONFIGURATION_BE(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_BE_SHIFT)) & ENET_MAC_CONFIGURATION_BE_MASK)
#define ENET_MAC_CONFIGURATION_JD_MASK                        (0x20000U)
#define ENET_MAC_CONFIGURATION_JD_SHIFT                       (17U)
#define ENET_MAC_CONFIGURATION_JD(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_JD_SHIFT)) & ENET_MAC_CONFIGURATION_JD_MASK)
#define ENET_MAC_CONFIGURATION_JE_MASK                        (0x10000U)
#define ENET_MAC_CONFIGURATION_JE_SHIFT                       (16U)
#define ENET_MAC_CONFIGURATION_JE(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_JE_SHIFT)) & ENET_MAC_CONFIGURATION_JE_MASK)
#define ENET_MAC_CONFIGURATION_PS_MASK                        (0x8000U)
#define ENET_MAC_CONFIGURATION_PS_SHIFT                       (15U)
#define ENET_MAC_CONFIGURATION_PS(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_PS_SHIFT)) & ENET_MAC_CONFIGURATION_PS_MASK)
#define ENET_MAC_CONFIGURATION_FES_MASK                       (0x4000U)
#define ENET_MAC_CONFIGURATION_FES_SHIFT                      (14U)
#define ENET_MAC_CONFIGURATION_FES(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_FES_SHIFT)) & ENET_MAC_CONFIGURATION_FES_MASK)
#define ENET_MAC_CONFIGURATION_DM_MASK                        (0x2000U)
#define ENET_MAC_CONFIGURATION_DM_SHIFT                       (13U)
#define ENET_MAC_CONFIGURATION_DM(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_DM_SHIFT)) & ENET_MAC_CONFIGURATION_DM_MASK)
#define ENET_MAC_CONFIGURATION_LM_MASK                        (0x1000U)
#define ENET_MAC_CONFIGURATION_LM_SHIFT                       (12U)
#define ENET_MAC_CONFIGURATION_LM(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_LM_SHIFT)) & ENET_MAC_CONFIGURATION_LM_MASK)
#define ENET_MAC_CONFIGURATION_ECRSFD_MASK                    (0x800U)
#define ENET_MAC_CONFIGURATION_ECRSFD_SHIFT                   (11U)
#define ENET_MAC_CONFIGURATION_ECRSFD(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_ECRSFD_SHIFT)) & ENET_MAC_CONFIGURATION_ECRSFD_MASK)
#define ENET_MAC_CONFIGURATION_DO_MASK                        (0x400U)
#define ENET_MAC_CONFIGURATION_DO_SHIFT                       (10U)
#define ENET_MAC_CONFIGURATION_DO(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_DO_SHIFT)) & ENET_MAC_CONFIGURATION_DO_MASK)
#define ENET_MAC_CONFIGURATION_DCRS_MASK                      (0x200U)
#define ENET_MAC_CONFIGURATION_DCRS_SHIFT                     (9U)
#define ENET_MAC_CONFIGURATION_DCRS(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_DCRS_SHIFT)) & ENET_MAC_CONFIGURATION_DCRS_MASK)
#define ENET_MAC_CONFIGURATION_DR_MASK                        (0x100U)
#define ENET_MAC_CONFIGURATION_DR_SHIFT                       (8U)
#define ENET_MAC_CONFIGURATION_DR(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_DR_SHIFT)) & ENET_MAC_CONFIGURATION_DR_MASK)
#define ENET_MAC_CONFIGURATION_BL_MASK                        (0x60U)
#define ENET_MAC_CONFIGURATION_BL_SHIFT                       (5U)
#define ENET_MAC_CONFIGURATION_BL(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_BL_SHIFT)) & ENET_MAC_CONFIGURATION_BL_MASK)
#define ENET_MAC_CONFIGURATION_DC_MASK                        (0x10U)
#define ENET_MAC_CONFIGURATION_DC_SHIFT                       (4U)
#define ENET_MAC_CONFIGURATION_DC(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_DC_SHIFT)) & ENET_MAC_CONFIGURATION_DC_MASK)
#define ENET_MAC_CONFIGURATION_PRELEN_MASK                    (0xCU)
#define ENET_MAC_CONFIGURATION_PRELEN_SHIFT                   (2U)
#define ENET_MAC_CONFIGURATION_PRELEN(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_PRELEN_SHIFT)) & ENET_MAC_CONFIGURATION_PRELEN_MASK)
#define ENET_MAC_CONFIGURATION_TE_MASK                        (0x2U)
#define ENET_MAC_CONFIGURATION_TE_SHIFT                       (1U)
#define ENET_MAC_CONFIGURATION_TE(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_TE_SHIFT)) & ENET_MAC_CONFIGURATION_TE_MASK)
#define ENET_MAC_CONFIGURATION_RE_MASK                        (0x1U)
#define ENET_MAC_CONFIGURATION_RE_SHIFT                       (0U)
#define ENET_MAC_CONFIGURATION_RE(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_CONFIGURATION_RE_SHIFT)) & ENET_MAC_CONFIGURATION_RE_MASK)
/* ENET_MAC_EXT_CONFIGURATION Register */
#define ENET_MAC_EXT_CONFIGURATION_FHE_MASK                       (0x80000000U)
#define ENET_MAC_EXT_CONFIGURATION_FHE_SHIFT                      (31U)
#define ENET_MAC_EXT_CONFIGURATION_FHE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_FHE_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_FHE_MASK)
#define ENET_MAC_EXT_CONFIGURATION_EIPG_MASK                      (0x3E000000U)
#define ENET_MAC_EXT_CONFIGURATION_EIPG_SHIFT                     (25U)
#define ENET_MAC_EXT_CONFIGURATION_EIPG(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_EIPG_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_EIPG_MASK)
#define ENET_MAC_EXT_CONFIGURATION_EIPGEN_MASK                    (0x1000000U)
#define ENET_MAC_EXT_CONFIGURATION_EIPGEN_SHIFT                   (24U)
#define ENET_MAC_EXT_CONFIGURATION_EIPGEN(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_EIPGEN_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_EIPGEN_MASK)
#define ENET_MAC_EXT_CONFIGURATION_HDSMS_MASK                     (0x700000U)
#define ENET_MAC_EXT_CONFIGURATION_HDSMS_SHIFT                    (20U)
#define ENET_MAC_EXT_CONFIGURATION_HDSMS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_HDSMS_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_HDSMS_MASK)
#define ENET_MAC_EXT_CONFIGURATION_PDC_MASK                       (0x80000U)
#define ENET_MAC_EXT_CONFIGURATION_PDC_SHIFT                      (19U)
#define ENET_MAC_EXT_CONFIGURATION_PDC(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_PDC_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_PDC_MASK)
#define ENET_MAC_EXT_CONFIGURATION_USP_MASK                       (0x40000U)
#define ENET_MAC_EXT_CONFIGURATION_USP_SHIFT                      (18U)
#define ENET_MAC_EXT_CONFIGURATION_USP(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_USP_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_USP_MASK)
#define ENET_MAC_EXT_CONFIGURATION_SPEN_MASK                      (0x20000U)
#define ENET_MAC_EXT_CONFIGURATION_SPEN_SHIFT                     (17U)
#define ENET_MAC_EXT_CONFIGURATION_SPEN(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_SPEN_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_SPEN_MASK)
#define ENET_MAC_EXT_CONFIGURATION_DCRCC_MASK                     (0x10000U)
#define ENET_MAC_EXT_CONFIGURATION_DCRCC_SHIFT                    (16U)
#define ENET_MAC_EXT_CONFIGURATION_DCRCC(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_DCRCC_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_DCRCC_MASK)
#define ENET_MAC_EXT_CONFIGURATION_GPSL_MASK                      (0x3FFFU)
#define ENET_MAC_EXT_CONFIGURATION_GPSL_SHIFT                     (0U)
#define ENET_MAC_EXT_CONFIGURATION_GPSL(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CONFIGURATION_GPSL_SHIFT)) & ENET_MAC_EXT_CONFIGURATION_GPSL_MASK)
/* ENET_MAC_PACKET_FILTER Register */
#define ENET_MAC_PACKET_FILTER_RA_MASK                        (0x80000000U)
#define ENET_MAC_PACKET_FILTER_RA_SHIFT                       (31U)
#define ENET_MAC_PACKET_FILTER_RA(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_RA_SHIFT)) & ENET_MAC_PACKET_FILTER_RA_MASK)
#define ENET_MAC_PACKET_FILTER_DNTU_MASK                      (0x200000U)
#define ENET_MAC_PACKET_FILTER_DNTU_SHIFT                     (21U)
#define ENET_MAC_PACKET_FILTER_DNTU(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_DNTU_SHIFT)) & ENET_MAC_PACKET_FILTER_DNTU_MASK)
#define ENET_MAC_PACKET_FILTER_IPFE_MASK                      (0x100000U)
#define ENET_MAC_PACKET_FILTER_IPFE_SHIFT                     (20U)
#define ENET_MAC_PACKET_FILTER_IPFE(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_IPFE_SHIFT)) & ENET_MAC_PACKET_FILTER_IPFE_MASK)
#define ENET_MAC_PACKET_FILTER_VTFE_MASK                      (0x10000U)
#define ENET_MAC_PACKET_FILTER_VTFE_SHIFT                     (16U)
#define ENET_MAC_PACKET_FILTER_VTFE(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_VTFE_SHIFT)) & ENET_MAC_PACKET_FILTER_VTFE_MASK)
#define ENET_MAC_PACKET_FILTER_HPF_MASK                       (0x400U)
#define ENET_MAC_PACKET_FILTER_HPF_SHIFT                      (10U)
#define ENET_MAC_PACKET_FILTER_HPF(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_HPF_SHIFT)) & ENET_MAC_PACKET_FILTER_HPF_MASK)
#define ENET_MAC_PACKET_FILTER_SAF_MASK                       (0x200U)
#define ENET_MAC_PACKET_FILTER_SAF_SHIFT                      (9U)
#define ENET_MAC_PACKET_FILTER_SAF(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_SAF_SHIFT)) & ENET_MAC_PACKET_FILTER_SAF_MASK)
#define ENET_MAC_PACKET_FILTER_SAIF_MASK                      (0x100U)
#define ENET_MAC_PACKET_FILTER_SAIF_SHIFT                     (8U)
#define ENET_MAC_PACKET_FILTER_SAIF(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_SAIF_SHIFT)) & ENET_MAC_PACKET_FILTER_SAIF_MASK)
#define ENET_MAC_PACKET_FILTER_PCF_MASK                       (0xC0U)
#define ENET_MAC_PACKET_FILTER_PCF_SHIFT                      (6U)
#define ENET_MAC_PACKET_FILTER_PCF(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_PCF_SHIFT)) & ENET_MAC_PACKET_FILTER_PCF_MASK)
#define ENET_MAC_PACKET_FILTER_DBF_MASK                       (0x20U)
#define ENET_MAC_PACKET_FILTER_DBF_SHIFT                      (5U)
#define ENET_MAC_PACKET_FILTER_DBF(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_DBF_SHIFT)) & ENET_MAC_PACKET_FILTER_DBF_MASK)
#define ENET_MAC_PACKET_FILTER_PM_MASK                        (0x10U)
#define ENET_MAC_PACKET_FILTER_PM_SHIFT                       (4U)
#define ENET_MAC_PACKET_FILTER_PM(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_PM_SHIFT)) & ENET_MAC_PACKET_FILTER_PM_MASK)
#define ENET_MAC_PACKET_FILTER_DAIF_MASK                      (0x8U)
#define ENET_MAC_PACKET_FILTER_DAIF_SHIFT                     (3U)
#define ENET_MAC_PACKET_FILTER_DAIF(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_DAIF_SHIFT)) & ENET_MAC_PACKET_FILTER_DAIF_MASK)
#define ENET_MAC_PACKET_FILTER_HMC_MASK                       (0x4U)
#define ENET_MAC_PACKET_FILTER_HMC_SHIFT                      (2U)
#define ENET_MAC_PACKET_FILTER_HMC(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_HMC_SHIFT)) & ENET_MAC_PACKET_FILTER_HMC_MASK)
#define ENET_MAC_PACKET_FILTER_HUC_MASK                       (0x2U)
#define ENET_MAC_PACKET_FILTER_HUC_SHIFT                      (1U)
#define ENET_MAC_PACKET_FILTER_HUC(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_HUC_SHIFT)) & ENET_MAC_PACKET_FILTER_HUC_MASK)
#define ENET_MAC_PACKET_FILTER_PR_MASK                        (0x1U)
#define ENET_MAC_PACKET_FILTER_PR_SHIFT                       (0U)
#define ENET_MAC_PACKET_FILTER_PR(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_PACKET_FILTER_PR_SHIFT)) & ENET_MAC_PACKET_FILTER_PR_MASK)
/* ENET_MAC_WATCHDOG_TIMEOUT Register */
#define ENET_MAC_WATCHDOG_TIMEOUT_PWE_MASK                       (0x100U)
#define ENET_MAC_WATCHDOG_TIMEOUT_PWE_SHIFT                      (8U)
#define ENET_MAC_WATCHDOG_TIMEOUT_PWE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_WATCHDOG_TIMEOUT_PWE_SHIFT)) & ENET_MAC_WATCHDOG_TIMEOUT_PWE_MASK)
#define ENET_MAC_WATCHDOG_TIMEOUT_WTO_MASK                       (0xFU)
#define ENET_MAC_WATCHDOG_TIMEOUT_WTO_SHIFT                      (0U)
#define ENET_MAC_WATCHDOG_TIMEOUT_WTO(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_WATCHDOG_TIMEOUT_WTO_SHIFT)) & ENET_MAC_WATCHDOG_TIMEOUT_WTO_MASK)
/* ENET_MAC_HASH_TABLE_REG Register */
#define ENET_MAC_HASH_TABLE_REG_HT_MASK                        (0xFFFFFFFFU)
#define ENET_MAC_HASH_TABLE_REG_HT_SHIFT                       (0U)
#define ENET_MAC_HASH_TABLE_REG_HT(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_HASH_TABLE_REG_HT_SHIFT)) & ENET_MAC_HASH_TABLE_REG_HT_MASK)
/* ENET_MAC_VLAN_TAG_TAG Register */
#define ENET_MAC_VLAN_TAG_TAG_EIVLRXS_MASK                               (0x80000000U)
#define ENET_MAC_VLAN_TAG_TAG_EIVLRXS_SHIFT                              (31U)
#define ENET_MAC_VLAN_TAG_TAG_EIVLRXS(x)                                 (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_EIVLRXS_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_EIVLRXS_MASK)
#define ENET_MAC_VLAN_TAG_TAG_EIVLS_MASK                                 (0x30000000U)
#define ENET_MAC_VLAN_TAG_TAG_EIVLS_SHIFT                                (28U)
#define ENET_MAC_VLAN_TAG_TAG_EIVLS(x)                                   (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_EIVLS_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_EIVLS_MASK)
#define ENET_MAC_VLAN_TAG_TAG_ERIVLT_MASK                                (0x8000000U)
#define ENET_MAC_VLAN_TAG_TAG_ERIVLT_SHIFT                               (27U)
#define ENET_MAC_VLAN_TAG_TAG_ERIVLT(x)                                  (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_ERIVLT_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_ERIVLT_MASK)
#define ENET_MAC_VLAN_TAG_TAG_EDVLP_MASK                                 (0x4000000U)
#define ENET_MAC_VLAN_TAG_TAG_EDVLP_SHIFT                                (26U)
#define ENET_MAC_VLAN_TAG_TAG_EDVLP(x)                                   (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_EDVLP_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_EDVLP_MASK)
#define ENET_MAC_VLAN_TAG_TAG_VTHM_MASK                                  (0x2000000U)
#define ENET_MAC_VLAN_TAG_TAG_VTHM_SHIFT                                 (25U)
#define ENET_MAC_VLAN_TAG_TAG_VTHM(x)                                    (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_VTHM_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_VTHM_MASK)
#define ENET_MAC_VLAN_TAG_TAG_EVLRXS_MASK                                (0x1000000U)
#define ENET_MAC_VLAN_TAG_TAG_EVLRXS_SHIFT                               (24U)
#define ENET_MAC_VLAN_TAG_TAG_EVLRXS(x)                                  (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_EVLRXS_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_EVLRXS_MASK)
#define ENET_MAC_VLAN_TAG_TAG_EVLS_MASK                                  (0x600000U)
#define ENET_MAC_VLAN_TAG_TAG_EVLS_SHIFT                                 (21U)
#define ENET_MAC_VLAN_TAG_TAG_EVLS(x)                                    (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_EVLS_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_EVLS_MASK)
#define ENET_MAC_VLAN_TAG_TAG_DOVLTC_MASK                                (0x100000U)
#define ENET_MAC_VLAN_TAG_TAG_DOVLTC_SHIFT                               (20U)
#define ENET_MAC_VLAN_TAG_TAG_DOVLTC(x)                                  (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_DOVLTC_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_DOVLTC_MASK)
#define ENET_MAC_VLAN_TAG_TAG_ERSVLM_MASK                                (0x80000U)
#define ENET_MAC_VLAN_TAG_TAG_ERSVLM_SHIFT                               (19U)
#define ENET_MAC_VLAN_TAG_TAG_ERSVLM(x)                                  (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_ERSVLM_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_ERSVLM_MASK)
#define ENET_MAC_VLAN_TAG_TAG_ESVL_MASK                                  (0x40000U)
#define ENET_MAC_VLAN_TAG_TAG_ESVL_SHIFT                                 (18U)
#define ENET_MAC_VLAN_TAG_TAG_ESVL(x)                                    (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_ESVL_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_ESVL_MASK)
#define ENET_MAC_VLAN_TAG_TAG_VTIM_MASK                                  (0x20000U)
#define ENET_MAC_VLAN_TAG_TAG_VTIM_SHIFT                                 (17U)
#define ENET_MAC_VLAN_TAG_TAG_VTIM(x)                                    (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_VTIM_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_VTIM_MASK)
#define ENET_MAC_VLAN_TAG_TAG_ETV_MASK                                   (0x10000U)
#define ENET_MAC_VLAN_TAG_TAG_ETV_SHIFT                                  (16U)
#define ENET_MAC_VLAN_TAG_TAG_ETV(x)                                     (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_ETV_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_ETV_MASK)
#define ENET_MAC_VLAN_TAG_TAG_VL_MASK                                    (0xFFFFU)
#define ENET_MAC_VLAN_TAG_TAG_VL_SHIFT                                   (0U)
#define ENET_MAC_VLAN_TAG_TAG_VL(x)                                      (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_TAG_VL_SHIFT)) & ENET_MAC_VLAN_TAG_TAG_VL_MASK)
/* ENET_MAC_VLAN_TAG_CTRL Register */
#define ENET_MAC_VLAN_TAG_CTRL_EIVLRXS_MASK                              (0x80000000U)
#define ENET_MAC_VLAN_TAG_CTRL_EIVLRXS_SHIFT                             (31U)
#define ENET_MAC_VLAN_TAG_CTRL_EIVLRXS(x)                                (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_EIVLRXS_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_EIVLRXS_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_EIVLS_MASK                                (0x30000000U)
#define ENET_MAC_VLAN_TAG_CTRL_EIVLS_SHIFT                               (28U)
#define ENET_MAC_VLAN_TAG_CTRL_EIVLS(x)                                  (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_EIVLS_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_EIVLS_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_ERIVLT_MASK                               (0x8000000U)
#define ENET_MAC_VLAN_TAG_CTRL_ERIVLT_SHIFT                              (27U)
#define ENET_MAC_VLAN_TAG_CTRL_ERIVLT(x)                                 (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_ERIVLT_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_ERIVLT_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_EDVLP_MASK                                (0x4000000U)
#define ENET_MAC_VLAN_TAG_CTRL_EDVLP_SHIFT                               (26U)
#define ENET_MAC_VLAN_TAG_CTRL_EDVLP(x)                                  (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_EDVLP_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_EDVLP_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_VTHM_MASK                                 (0x2000000U)
#define ENET_MAC_VLAN_TAG_CTRL_VTHM_SHIFT                                (25U)
#define ENET_MAC_VLAN_TAG_CTRL_VTHM(x)                                   (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_VTHM_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_VTHM_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_EVLRXS_MASK                               (0x1000000U)
#define ENET_MAC_VLAN_TAG_CTRL_EVLRXS_SHIFT                              (24U)
#define ENET_MAC_VLAN_TAG_CTRL_EVLRXS(x)                                 (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_EVLRXS_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_EVLRXS_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_EVLS_MASK                                 (0x600000U)
#define ENET_MAC_VLAN_TAG_CTRL_EVLS_SHIFT                                (21U)
#define ENET_MAC_VLAN_TAG_CTRL_EVLS(x)                                   (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_EVLS_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_EVLS_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_ESVL_MASK                                 (0x40000U)
#define ENET_MAC_VLAN_TAG_CTRL_ESVL_SHIFT                                (18U)
#define ENET_MAC_VLAN_TAG_CTRL_ESVL(x)                                   (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_ESVL_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_ESVL_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_VTIM_MASK                                 (0x20000U)
#define ENET_MAC_VLAN_TAG_CTRL_VTIM_SHIFT                                (17U)
#define ENET_MAC_VLAN_TAG_CTRL_VTIM(x)                                   (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_VTIM_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_VTIM_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_OFS_MASK                                  (0x7CU)
#define ENET_MAC_VLAN_TAG_CTRL_OFS_SHIFT                                 (2U)
#define ENET_MAC_VLAN_TAG_CTRL_OFS(x)                                    (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_OFS_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_OFS_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_CT_MASK                                   (0x2U)
#define ENET_MAC_VLAN_TAG_CTRL_CT_SHIFT                                  (1U)
#define ENET_MAC_VLAN_TAG_CTRL_CT(x)                                     (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_CT_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_CT_MASK)
#define ENET_MAC_VLAN_TAG_CTRL_OB_MASK                                   (0x1U)
#define ENET_MAC_VLAN_TAG_CTRL_OB_SHIFT                                  (0U)
#define ENET_MAC_VLAN_TAG_CTRL_OB(x)                                     (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_CTRL_OB_SHIFT)) & ENET_MAC_VLAN_TAG_CTRL_OB_MASK)

/* ENET_MAC_VLAN_TAG_DATA Register */
#define ENET_MAC_VLAN_TAG_DATA_DMACHN_MASK                    (0x1E000000U)
#define ENET_MAC_VLAN_TAG_DATA_DMACHN_SHIFT                   (25U)
#define ENET_MAC_VLAN_TAG_DATA_DMACHN(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_DATA_DMACHN_SHIFT)) & ENET_MAC_VLAN_TAG_DATA_DMACHN_MASK)
#define ENET_MAC_VLAN_TAG_DATA_DMACHEN_MASK                   (0x1000000U)
#define ENET_MAC_VLAN_TAG_DATA_DMACHEN_SHIFT                  (24U)
#define ENET_MAC_VLAN_TAG_DATA_DMACHEN(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_DATA_DMACHEN_SHIFT)) & ENET_MAC_VLAN_TAG_DATA_DMACHEN_MASK)
#define ENET_MAC_VLAN_TAG_DATA_ERIVLT_MASK                    (0x100000U)
#define ENET_MAC_VLAN_TAG_DATA_ERIVLT_SHIFT                   (20U)
#define ENET_MAC_VLAN_TAG_DATA_ERIVLT(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_DATA_ERIVLT_SHIFT)) & ENET_MAC_VLAN_TAG_DATA_ERIVLT_MASK)
#define ENET_MAC_VLAN_TAG_DATA_ERSVLM_MASK                    (0x80000U)
#define ENET_MAC_VLAN_TAG_DATA_ERSVLM_SHIFT                   (19U)
#define ENET_MAC_VLAN_TAG_DATA_ERSVLM(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_DATA_ERSVLM_SHIFT)) & ENET_MAC_VLAN_TAG_DATA_ERSVLM_MASK)
#define ENET_MAC_VLAN_TAG_DATA_DOVLTC_MASK                    (0x40000U)
#define ENET_MAC_VLAN_TAG_DATA_DOVLTC_SHIFT                   (18U)
#define ENET_MAC_VLAN_TAG_DATA_DOVLTC(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_DATA_DOVLTC_SHIFT)) & ENET_MAC_VLAN_TAG_DATA_DOVLTC_MASK)
#define ENET_MAC_VLAN_TAG_DATA_ETV_MASK                       (0x20000U)
#define ENET_MAC_VLAN_TAG_DATA_ETV_SHIFT                      (17U)
#define ENET_MAC_VLAN_TAG_DATA_ETV(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_DATA_ETV_SHIFT)) & ENET_MAC_VLAN_TAG_DATA_ETV_MASK)
#define ENET_MAC_VLAN_TAG_DATA_VEN_MASK                       (0x10000U)
#define ENET_MAC_VLAN_TAG_DATA_VEN_SHIFT                      (16U)
#define ENET_MAC_VLAN_TAG_DATA_VEN(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_DATA_VEN_SHIFT)) & ENET_MAC_VLAN_TAG_DATA_VEN_MASK)
#define ENET_MAC_VLAN_TAG_DATA_VID_MASK                       (0xFFFFU)
#define ENET_MAC_VLAN_TAG_DATA_VID_SHIFT                      (0U)
#define ENET_MAC_VLAN_TAG_DATA_VID(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_TAG_DATA_VID_SHIFT)) & ENET_MAC_VLAN_TAG_DATA_VID_MASK)
/* ENET_MAC_VLAN_HASH_TABLE Register */
#define ENET_MAC_VLAN_HASH_TABLE_VLHT_MASK                      (0xFFFFU)
#define ENET_MAC_VLAN_HASH_TABLE_VLHT_SHIFT                     (0U)
#define ENET_MAC_VLAN_HASH_TABLE_VLHT(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_HASH_TABLE_VLHT_SHIFT)) & ENET_MAC_VLAN_HASH_TABLE_VLHT_MASK)
/* ENET_MAC_VLAN_INCL Register */
#define ENET_MAC_VLAN_INCL_BUSY_MASK                        (0x80000000U)
#define ENET_MAC_VLAN_INCL_BUSY_SHIFT                       (31U)
#define ENET_MAC_VLAN_INCL_BUSY(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_BUSY_SHIFT)) & ENET_MAC_VLAN_INCL_BUSY_MASK)
#define ENET_MAC_VLAN_INCL_RDWR_MASK                        (0x40000000U)
#define ENET_MAC_VLAN_INCL_RDWR_SHIFT                       (30U)
#define ENET_MAC_VLAN_INCL_RDWR(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_RDWR_SHIFT)) & ENET_MAC_VLAN_INCL_RDWR_MASK)
#define ENET_MAC_VLAN_INCL_ADDR_MASK                        (0x3F000000U)
#define ENET_MAC_VLAN_INCL_ADDR_SHIFT                       (24U)
#define ENET_MAC_VLAN_INCL_ADDR(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_ADDR_SHIFT)) & ENET_MAC_VLAN_INCL_ADDR_MASK)
#define ENET_MAC_VLAN_INCL_CBTI_MASK                        (0x800000U)
#define ENET_MAC_VLAN_INCL_CBTI_SHIFT                       (23U)
#define ENET_MAC_VLAN_INCL_CBTI(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_CBTI_SHIFT)) & ENET_MAC_VLAN_INCL_CBTI_MASK)
#define ENET_MAC_VLAN_INCL_VLTI_MASK                        (0x100000U)
#define ENET_MAC_VLAN_INCL_VLTI_SHIFT                       (20U)
#define ENET_MAC_VLAN_INCL_VLTI(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_VLTI_SHIFT)) & ENET_MAC_VLAN_INCL_VLTI_MASK)
#define ENET_MAC_VLAN_INCL_CSVL_MASK                        (0x80000U)
#define ENET_MAC_VLAN_INCL_CSVL_SHIFT                       (19U)
#define ENET_MAC_VLAN_INCL_CSVL(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_CSVL_SHIFT)) & ENET_MAC_VLAN_INCL_CSVL_MASK)
#define ENET_MAC_VLAN_INCL_VLP_MASK                         (0x40000U)
#define ENET_MAC_VLAN_INCL_VLP_SHIFT                        (18U)
#define ENET_MAC_VLAN_INCL_VLP(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_VLP_SHIFT)) & ENET_MAC_VLAN_INCL_VLP_MASK)
#define ENET_MAC_VLAN_INCL_VLC_MASK                         (0x30000U)
#define ENET_MAC_VLAN_INCL_VLC_SHIFT                        (16U)
#define ENET_MAC_VLAN_INCL_VLC(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_VLC_SHIFT)) & ENET_MAC_VLAN_INCL_VLC_MASK)
#define ENET_MAC_VLAN_INCL_VLT_MASK                         (0xFFFFU)
#define ENET_MAC_VLAN_INCL_VLT_SHIFT                        (0U)
#define ENET_MAC_VLAN_INCL_VLT(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_VLAN_INCL_VLT_SHIFT)) & ENET_MAC_VLAN_INCL_VLT_MASK)
/* ENET_MAC_INNER_VLAN_INCL Register */
#define ENET_MAC_INNER_VLAN_INCL_VLTI_MASK                      (0x100000U)
#define ENET_MAC_INNER_VLAN_INCL_VLTI_SHIFT                     (20U)
#define ENET_MAC_INNER_VLAN_INCL_VLTI(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_INNER_VLAN_INCL_VLTI_SHIFT)) & ENET_MAC_INNER_VLAN_INCL_VLTI_MASK)
#define ENET_MAC_INNER_VLAN_INCL_CSVL_MASK                      (0x80000U)
#define ENET_MAC_INNER_VLAN_INCL_CSVL_SHIFT                     (19U)
#define ENET_MAC_INNER_VLAN_INCL_CSVL(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_INNER_VLAN_INCL_CSVL_SHIFT)) & ENET_MAC_INNER_VLAN_INCL_CSVL_MASK)
#define ENET_MAC_INNER_VLAN_INCL_VLP_MASK                       (0x40000U)
#define ENET_MAC_INNER_VLAN_INCL_VLP_SHIFT                      (18U)
#define ENET_MAC_INNER_VLAN_INCL_VLP(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_INNER_VLAN_INCL_VLP_SHIFT)) & ENET_MAC_INNER_VLAN_INCL_VLP_MASK)
#define ENET_MAC_INNER_VLAN_INCL_VLC_MASK                       (0x30000U)
#define ENET_MAC_INNER_VLAN_INCL_VLC_SHIFT                      (16U)
#define ENET_MAC_INNER_VLAN_INCL_VLC(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_INNER_VLAN_INCL_VLC_SHIFT)) & ENET_MAC_INNER_VLAN_INCL_VLC_MASK)
#define ENET_MAC_INNER_VLAN_INCL_VLT_MASK                       (0xFFFFU)
#define ENET_MAC_INNER_VLAN_INCL_VLT_SHIFT                      (0U)
#define ENET_MAC_INNER_VLAN_INCL_VLT(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_INNER_VLAN_INCL_VLT_SHIFT)) & ENET_MAC_INNER_VLAN_INCL_VLT_MASK)
/* ENET_MAC_TX_FLOW_CTRL_Q Register */
#define ENET_MAC_TX_FLOW_CTRL_Q_PT_MASK                        (0xFFFF0000U)
#define ENET_MAC_TX_FLOW_CTRL_Q_PT_SHIFT                       (16U)
#define ENET_MAC_TX_FLOW_CTRL_Q_PT(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_TX_FLOW_CTRL_Q_PT_SHIFT)) & ENET_MAC_TX_FLOW_CTRL_Q_PT_MASK)
#define ENET_MAC_TX_FLOW_CTRL_Q_DZPQ_MASK                      (0x80U)
#define ENET_MAC_TX_FLOW_CTRL_Q_DZPQ_SHIFT                     (7U)
#define ENET_MAC_TX_FLOW_CTRL_Q_DZPQ(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_TX_FLOW_CTRL_Q_DZPQ_SHIFT)) & ENET_MAC_TX_FLOW_CTRL_Q_DZPQ_MASK)
#define ENET_MAC_TX_FLOW_CTRL_Q_PLT_MASK                       (0x70U)
#define ENET_MAC_TX_FLOW_CTRL_Q_PLT_SHIFT                      (4U)
#define ENET_MAC_TX_FLOW_CTRL_Q_PLT(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_TX_FLOW_CTRL_Q_PLT_SHIFT)) & ENET_MAC_TX_FLOW_CTRL_Q_PLT_MASK)
#define ENET_MAC_TX_FLOW_CTRL_Q_TFE_MASK                       (0x2U)
#define ENET_MAC_TX_FLOW_CTRL_Q_TFE_SHIFT                      (1U)
#define ENET_MAC_TX_FLOW_CTRL_Q_TFE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_TX_FLOW_CTRL_Q_TFE_SHIFT)) & ENET_MAC_TX_FLOW_CTRL_Q_TFE_MASK)
#define ENET_MAC_TX_FLOW_CTRL_Q_FCB_BPA_MASK                   (0x1U)
#define ENET_MAC_TX_FLOW_CTRL_Q_FCB_BPA_SHIFT                  (0U)
#define ENET_MAC_TX_FLOW_CTRL_Q_FCB_BPA(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_TX_FLOW_CTRL_Q_FCB_BPA_SHIFT)) & ENET_MAC_TX_FLOW_CTRL_Q_FCB_BPA_MASK)
/* ENET_MAC_RX_FLOW_CTRL Register */
#define ENET_MAC_RX_FLOW_CTRL_PFCE_MASK                      (0x100U)
#define ENET_MAC_RX_FLOW_CTRL_PFCE_SHIFT                     (8U)
#define ENET_MAC_RX_FLOW_CTRL_PFCE(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_RX_FLOW_CTRL_PFCE_SHIFT)) & ENET_MAC_RX_FLOW_CTRL_PFCE_MASK)
#define ENET_MAC_RX_FLOW_CTRL_UP_MASK                        (0x2U)
#define ENET_MAC_RX_FLOW_CTRL_UP_SHIFT                       (1U)
#define ENET_MAC_RX_FLOW_CTRL_UP(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_RX_FLOW_CTRL_UP_SHIFT)) & ENET_MAC_RX_FLOW_CTRL_UP_MASK)
#define ENET_MAC_RX_FLOW_CTRL_RFE_MASK                       (0x1U)
#define ENET_MAC_RX_FLOW_CTRL_RFE_SHIFT                      (0U)
#define ENET_MAC_RX_FLOW_CTRL_RFE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RX_FLOW_CTRL_RFE_SHIFT)) & ENET_MAC_RX_FLOW_CTRL_RFE_MASK)
/* ENET_MAC_RXQ_CTRL4 Register */
#define ENET_MAC_RXQ_CTRL4_VFFQ_WIDTH                       (1)
#define ENET_MAC_RXQ_CTRL4_VFFQ_MASK                        (0x20000U)
#define ENET_MAC_RXQ_CTRL4_VFFQ_SHIFT                       (17U)
#define ENET_MAC_RXQ_CTRL4_VFFQ(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL4_VFFQ_SHIFT)) & ENET_MAC_RXQ_CTRL4_VFFQ_MASK)
#define ENET_MAC_RXQ_CTRL4_VFFQE_MASK                       (0x10000U)
#define ENET_MAC_RXQ_CTRL4_VFFQE_SHIFT                      (16U)
#define ENET_MAC_RXQ_CTRL4_VFFQE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL4_VFFQE_SHIFT)) & ENET_MAC_RXQ_CTRL4_VFFQE_MASK)
#define ENET_MAC_RXQ_CTRL4_MFFQ_WIDTH                       (1)
#define ENET_MAC_RXQ_CTRL4_MFFQ_MASK                        (0x200U)
#define ENET_MAC_RXQ_CTRL4_MFFQ_SHIFT                       (9U)
#define ENET_MAC_RXQ_CTRL4_MFFQ(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL4_MFFQ_SHIFT)) & ENET_MAC_RXQ_CTRL4_MFFQ_MASK)
#define ENET_MAC_RXQ_CTRL4_MFFQE_MASK                       (0x100U)
#define ENET_MAC_RXQ_CTRL4_MFFQE_SHIFT                      (8U)
#define ENET_MAC_RXQ_CTRL4_MFFQE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL4_MFFQE_SHIFT)) & ENET_MAC_RXQ_CTRL4_MFFQE_MASK)
#define ENET_MAC_RXQ_CTRL4_UFFQ_WIDTH                       (1)
#define ENET_MAC_RXQ_CTRL4_UFFQ_MASK                        (0x2U)
#define ENET_MAC_RXQ_CTRL4_UFFQ_SHIFT                       (1U)
#define ENET_MAC_RXQ_CTRL4_UFFQ(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL4_UFFQ_SHIFT)) & ENET_MAC_RXQ_CTRL4_UFFQ_MASK)
#define ENET_MAC_RXQ_CTRL4_UFFQE_MASK                       (0x1U)
#define ENET_MAC_RXQ_CTRL4_UFFQE_SHIFT                      (0U)
#define ENET_MAC_RXQ_CTRL4_UFFQE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL4_UFFQE_SHIFT)) & ENET_MAC_RXQ_CTRL4_UFFQE_MASK)
/* ENET_MAC_TXQ_PRTY_MAP0 Register */
#define ENET_MAC_TXQ_PRTY_MAP0_PSTQ1_MASK                     (0xFF00U)
#define ENET_MAC_TXQ_PRTY_MAP0_PSTQ1_SHIFT                    (8U)
#define ENET_MAC_TXQ_PRTY_MAP0_PSTQ1(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_TXQ_PRTY_MAP0_PSTQ1_SHIFT)) & ENET_MAC_TXQ_PRTY_MAP0_PSTQ1_MASK)
#define ENET_MAC_TXQ_PRTY_MAP0_PSTQ0_MASK                     (0xFFU)
#define ENET_MAC_TXQ_PRTY_MAP0_PSTQ0_SHIFT                    (0U)
#define ENET_MAC_TXQ_PRTY_MAP0_PSTQ0(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_TXQ_PRTY_MAP0_PSTQ0_SHIFT)) & ENET_MAC_TXQ_PRTY_MAP0_PSTQ0_MASK)
/* ENET_MAC_RXQ_CTRL0 Register */
#define ENET_MAC_RXQ_CTRL0_RQ1EN_MASK                       (0x2U)
#define ENET_MAC_RXQ_CTRL0_RQ1EN_SHIFT                      (1U)
#define ENET_MAC_RXQ_CTRL0_RQ1EN(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL0_RQ1EN_SHIFT)) & ENET_MAC_RXQ_CTRL0_RQ1EN_MASK)
#define ENET_MAC_RXQ_CTRL0_RQ0EN_MASK                       (0x1U)
#define ENET_MAC_RXQ_CTRL0_RQ0EN_SHIFT                      (0U)
#define ENET_MAC_RXQ_CTRL0_RQ0EN(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL0_RQ0EN_SHIFT)) & ENET_MAC_RXQ_CTRL0_RQ0EN_MASK)
/* ENET_MAC_RXQ_CTRL1 Register */
#define ENET_MAC_RXQ_CTRL1_FPRQ_MASK                        (0x7000000U)
#define ENET_MAC_RXQ_CTRL1_FPRQ_SHIFT                       (24U)
#define ENET_MAC_RXQ_CTRL1_FPRQ(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_FPRQ_SHIFT)) & ENET_MAC_RXQ_CTRL1_FPRQ_MASK)
#define ENET_MAC_RXQ_CTRL1_TPQC_MASK                        (0xC00000U)
#define ENET_MAC_RXQ_CTRL1_TPQC_SHIFT                       (22U)
#define ENET_MAC_RXQ_CTRL1_TPQC(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_TPQC_SHIFT)) & ENET_MAC_RXQ_CTRL1_TPQC_MASK)
#define ENET_MAC_RXQ_CTRL1_TACPQE_MASK                      (0x200000U)
#define ENET_MAC_RXQ_CTRL1_TACPQE_SHIFT                     (21U)
#define ENET_MAC_RXQ_CTRL1_TACPQE(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_TACPQE_SHIFT)) & ENET_MAC_RXQ_CTRL1_TACPQE_MASK)
#define ENET_MAC_RXQ_CTRL1_MCBCQEN_MASK                     (0x100000U)
#define ENET_MAC_RXQ_CTRL1_MCBCQEN_SHIFT                    (20U)
#define ENET_MAC_RXQ_CTRL1_MCBCQEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_MCBCQEN_SHIFT)) & ENET_MAC_RXQ_CTRL1_MCBCQEN_MASK)
#define ENET_MAC_RXQ_CTRL1_MCBCQ_MASK                       (0x70000U)
#define ENET_MAC_RXQ_CTRL1_MCBCQ_SHIFT                      (16U)
#define ENET_MAC_RXQ_CTRL1_MCBCQ(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_MCBCQ_SHIFT)) & ENET_MAC_RXQ_CTRL1_MCBCQ_MASK)
#define ENET_MAC_RXQ_CTRL1_UPQ_MASK                         (0x7000U)
#define ENET_MAC_RXQ_CTRL1_UPQ_SHIFT                        (12U)
#define ENET_MAC_RXQ_CTRL1_UPQ(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_UPQ_SHIFT)) & ENET_MAC_RXQ_CTRL1_UPQ_MASK)
#define ENET_MAC_RXQ_CTRL1_DCBCPQ_MASK                      (0x700U)
#define ENET_MAC_RXQ_CTRL1_DCBCPQ_SHIFT                     (8U)
#define ENET_MAC_RXQ_CTRL1_DCBCPQ(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_DCBCPQ_SHIFT)) & ENET_MAC_RXQ_CTRL1_DCBCPQ_MASK)
#define ENET_MAC_RXQ_CTRL1_PTPQ_MASK                        (0x70U)
#define ENET_MAC_RXQ_CTRL1_PTPQ_SHIFT                       (4U)
#define ENET_MAC_RXQ_CTRL1_PTPQ(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_PTPQ_SHIFT)) & ENET_MAC_RXQ_CTRL1_PTPQ_MASK)
#define ENET_MAC_RXQ_CTRL1_AVCPQ_MASK                       (0x7U)
#define ENET_MAC_RXQ_CTRL1_AVCPQ_SHIFT                      (0U)
#define ENET_MAC_RXQ_CTRL1_AVCPQ(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL1_AVCPQ_SHIFT)) & ENET_MAC_RXQ_CTRL1_AVCPQ_MASK)
/* ENET_MAC_RXQ_CTRL2 Register */
#define ENET_MAC_RXQ_CTRL2_PSRQ3_MASK                       (0xFF000000U)
#define ENET_MAC_RXQ_CTRL2_PSRQ3_SHIFT                      (24U)
#define ENET_MAC_RXQ_CTRL2_PSRQ3(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL2_PSRQ3_SHIFT)) & ENET_MAC_RXQ_CTRL2_PSRQ3_MASK)
#define ENET_MAC_RXQ_CTRL2_PSRQ2_MASK                       (0xFF0000U)
#define ENET_MAC_RXQ_CTRL2_PSRQ2_SHIFT                      (16U)
#define ENET_MAC_RXQ_CTRL2_PSRQ2(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL2_PSRQ2_SHIFT)) & ENET_MAC_RXQ_CTRL2_PSRQ2_MASK)
#define ENET_MAC_RXQ_CTRL2_PSRQ1_MASK                       (0xFF00U)
#define ENET_MAC_RXQ_CTRL2_PSRQ1_SHIFT                      (8U)
#define ENET_MAC_RXQ_CTRL2_PSRQ1(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL2_PSRQ1_SHIFT)) & ENET_MAC_RXQ_CTRL2_PSRQ1_MASK)
#define ENET_MAC_RXQ_CTRL2_PSRQ0_MASK                       (0xFFU)
#define ENET_MAC_RXQ_CTRL2_PSRQ0_SHIFT                      (0U)
#define ENET_MAC_RXQ_CTRL2_PSRQ0(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RXQ_CTRL2_PSRQ0_SHIFT)) & ENET_MAC_RXQ_CTRL2_PSRQ0_MASK)
/* ENET_MAC_INTERRUPT_STATUS Register */
#define ENET_MAC_INTERRUPT_STATUS_MFRIS_MASK                     (0x100000U)
#define ENET_MAC_INTERRUPT_STATUS_MFRIS_SHIFT                    (20U)
#define ENET_MAC_INTERRUPT_STATUS_MFRIS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_MFRIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_MFRIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_MFTIS_MASK                     (0x80000U)
#define ENET_MAC_INTERRUPT_STATUS_MFTIS_SHIFT                    (19U)
#define ENET_MAC_INTERRUPT_STATUS_MFTIS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_MFTIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_MFTIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_MDIOIS_MASK                    (0x40000U)
#define ENET_MAC_INTERRUPT_STATUS_MDIOIS_SHIFT                   (18U)
#define ENET_MAC_INTERRUPT_STATUS_MDIOIS(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_MDIOIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_MDIOIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_FPEIS_MASK                     (0x20000U)
#define ENET_MAC_INTERRUPT_STATUS_FPEIS_SHIFT                    (17U)
#define ENET_MAC_INTERRUPT_STATUS_FPEIS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_FPEIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_FPEIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_GPIIS_MASK                     (0x8000U)
#define ENET_MAC_INTERRUPT_STATUS_GPIIS_SHIFT                    (15U)
#define ENET_MAC_INTERRUPT_STATUS_GPIIS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_GPIIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_GPIIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_RXSTSIS_MASK                   (0x4000U)
#define ENET_MAC_INTERRUPT_STATUS_RXSTSIS_SHIFT                  (14U)
#define ENET_MAC_INTERRUPT_STATUS_RXSTSIS(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_RXSTSIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_RXSTSIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_TXSTSIS_MASK                   (0x2000U)
#define ENET_MAC_INTERRUPT_STATUS_TXSTSIS_SHIFT                  (13U)
#define ENET_MAC_INTERRUPT_STATUS_TXSTSIS(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_TXSTSIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_TXSTSIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_TSIS_MASK                      (0x1000U)
#define ENET_MAC_INTERRUPT_STATUS_TSIS_SHIFT                     (12U)
#define ENET_MAC_INTERRUPT_STATUS_TSIS(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_TSIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_TSIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_MMCRXIPIS_MASK                 (0x800U)
#define ENET_MAC_INTERRUPT_STATUS_MMCRXIPIS_SHIFT                (11U)
#define ENET_MAC_INTERRUPT_STATUS_MMCRXIPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_MMCRXIPIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_MMCRXIPIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_MMCTXIS_MASK                   (0x400U)
#define ENET_MAC_INTERRUPT_STATUS_MMCTXIS_SHIFT                  (10U)
#define ENET_MAC_INTERRUPT_STATUS_MMCTXIS(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_MMCTXIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_MMCTXIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_MMCRXIS_MASK                   (0x200U)
#define ENET_MAC_INTERRUPT_STATUS_MMCRXIS_SHIFT                  (9U)
#define ENET_MAC_INTERRUPT_STATUS_MMCRXIS(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_MMCRXIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_MMCRXIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_MMCIS_MASK                     (0x100U)
#define ENET_MAC_INTERRUPT_STATUS_MMCIS_SHIFT                    (8U)
#define ENET_MAC_INTERRUPT_STATUS_MMCIS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_MMCIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_MMCIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_LPIIS_MASK                     (0x20U)
#define ENET_MAC_INTERRUPT_STATUS_LPIIS_SHIFT                    (5U)
#define ENET_MAC_INTERRUPT_STATUS_LPIIS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_LPIIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_LPIIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_PMTIS_MASK                     (0x10U)
#define ENET_MAC_INTERRUPT_STATUS_PMTIS_SHIFT                    (4U)
#define ENET_MAC_INTERRUPT_STATUS_PMTIS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_PMTIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_PMTIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_PHYIS_MASK                     (0x8U)
#define ENET_MAC_INTERRUPT_STATUS_PHYIS_SHIFT                    (3U)
#define ENET_MAC_INTERRUPT_STATUS_PHYIS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_PHYIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_PHYIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_PCSANCIS_MASK                  (0x4U)
#define ENET_MAC_INTERRUPT_STATUS_PCSANCIS_SHIFT                 (2U)
#define ENET_MAC_INTERRUPT_STATUS_PCSANCIS(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_PCSANCIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_PCSANCIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_PCSLCHGIS_MASK                 (0x2U)
#define ENET_MAC_INTERRUPT_STATUS_PCSLCHGIS_SHIFT                (1U)
#define ENET_MAC_INTERRUPT_STATUS_PCSLCHGIS(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_PCSLCHGIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_PCSLCHGIS_MASK)
#define ENET_MAC_INTERRUPT_STATUS_RGSMIIIS_MASK                  (0x1U)
#define ENET_MAC_INTERRUPT_STATUS_RGSMIIIS_SHIFT                 (0U)
#define ENET_MAC_INTERRUPT_STATUS_RGSMIIIS(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_STATUS_RGSMIIIS_SHIFT)) & ENET_MAC_INTERRUPT_STATUS_RGSMIIIS_MASK)
/* ENET_MAC_INTERRUPT_ENABLE Register */
#define ENET_MAC_INTERRUPT_ENABLE_MDIOIE_MASK                    (0x40000U)
#define ENET_MAC_INTERRUPT_ENABLE_MDIOIE_SHIFT                   (18U)
#define ENET_MAC_INTERRUPT_ENABLE_MDIOIE(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_MDIOIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_MDIOIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_FPEIE_MASK                     (0x20000U)
#define ENET_MAC_INTERRUPT_ENABLE_FPEIE_SHIFT                    (17U)
#define ENET_MAC_INTERRUPT_ENABLE_FPEIE(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_FPEIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_FPEIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_RXSTSIE_MASK                   (0x4000U)
#define ENET_MAC_INTERRUPT_ENABLE_RXSTSIE_SHIFT                  (14U)
#define ENET_MAC_INTERRUPT_ENABLE_RXSTSIE(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_RXSTSIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_RXSTSIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_TXSTSIE_MASK                   (0x2000U)
#define ENET_MAC_INTERRUPT_ENABLE_TXSTSIE_SHIFT                  (13U)
#define ENET_MAC_INTERRUPT_ENABLE_TXSTSIE(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_TXSTSIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_TXSTSIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_TSIE_MASK                      (0x1000U)
#define ENET_MAC_INTERRUPT_ENABLE_TSIE_SHIFT                     (12U)
#define ENET_MAC_INTERRUPT_ENABLE_TSIE(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_TSIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_TSIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_LPIIE_MASK                     (0x20U)
#define ENET_MAC_INTERRUPT_ENABLE_LPIIE_SHIFT                    (5U)
#define ENET_MAC_INTERRUPT_ENABLE_LPIIE(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_LPIIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_LPIIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_PMTIE_MASK                     (0x10U)
#define ENET_MAC_INTERRUPT_ENABLE_PMTIE_SHIFT                    (4U)
#define ENET_MAC_INTERRUPT_ENABLE_PMTIE(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_PMTIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_PMTIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_PHYIE_MASK                     (0x8U)
#define ENET_MAC_INTERRUPT_ENABLE_PHYIE_SHIFT                    (3U)
#define ENET_MAC_INTERRUPT_ENABLE_PHYIE(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_PHYIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_PHYIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_PCSANCIE_MASK                  (0x4U)
#define ENET_MAC_INTERRUPT_ENABLE_PCSANCIE_SHIFT                 (2U)
#define ENET_MAC_INTERRUPT_ENABLE_PCSANCIE(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_PCSANCIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_PCSANCIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_PCSLCHGIE_MASK                 (0x2U)
#define ENET_MAC_INTERRUPT_ENABLE_PCSLCHGIE_SHIFT                (1U)
#define ENET_MAC_INTERRUPT_ENABLE_PCSLCHGIE(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_PCSLCHGIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_PCSLCHGIE_MASK)
#define ENET_MAC_INTERRUPT_ENABLE_RGSMIIIE_MASK                  (0x1U)
#define ENET_MAC_INTERRUPT_ENABLE_RGSMIIIE_SHIFT                 (0U)
#define ENET_MAC_INTERRUPT_ENABLE_RGSMIIIE(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_INTERRUPT_ENABLE_RGSMIIIE_SHIFT)) & ENET_MAC_INTERRUPT_ENABLE_RGSMIIIE_MASK)
/* ENET_MAC_RX_TX_STATUS Register */
#define ENET_MAC_RX_TX_STATUS_RWT_MASK                       (0x100U)
#define ENET_MAC_RX_TX_STATUS_RWT_SHIFT                      (8U)
#define ENET_MAC_RX_TX_STATUS_RWT(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RX_TX_STATUS_RWT_SHIFT)) & ENET_MAC_RX_TX_STATUS_RWT_MASK)
#define ENET_MAC_RX_TX_STATUS_EXCOL_MASK                     (0x20U)
#define ENET_MAC_RX_TX_STATUS_EXCOL_SHIFT                    (5U)
#define ENET_MAC_RX_TX_STATUS_EXCOL(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_RX_TX_STATUS_EXCOL_SHIFT)) & ENET_MAC_RX_TX_STATUS_EXCOL_MASK)
#define ENET_MAC_RX_TX_STATUS_LCOL_MASK                      (0x10U)
#define ENET_MAC_RX_TX_STATUS_LCOL_SHIFT                     (4U)
#define ENET_MAC_RX_TX_STATUS_LCOL(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_RX_TX_STATUS_LCOL_SHIFT)) & ENET_MAC_RX_TX_STATUS_LCOL_MASK)
#define ENET_MAC_RX_TX_STATUS_EXDEF_MASK                     (0x8U)
#define ENET_MAC_RX_TX_STATUS_EXDEF_SHIFT                    (3U)
#define ENET_MAC_RX_TX_STATUS_EXDEF(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_RX_TX_STATUS_EXDEF_SHIFT)) & ENET_MAC_RX_TX_STATUS_EXDEF_MASK)
#define ENET_MAC_RX_TX_STATUS_LCARR_MASK                     (0x4U)
#define ENET_MAC_RX_TX_STATUS_LCARR_SHIFT                    (2U)
#define ENET_MAC_RX_TX_STATUS_LCARR(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_RX_TX_STATUS_LCARR_SHIFT)) & ENET_MAC_RX_TX_STATUS_LCARR_MASK)
#define ENET_MAC_RX_TX_STATUS_NCARR_MASK                     (0x2U)
#define ENET_MAC_RX_TX_STATUS_NCARR_SHIFT                    (1U)
#define ENET_MAC_RX_TX_STATUS_NCARR(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_RX_TX_STATUS_NCARR_SHIFT)) & ENET_MAC_RX_TX_STATUS_NCARR_MASK)
#define ENET_MAC_RX_TX_STATUS_TJT_MASK                       (0x1U)
#define ENET_MAC_RX_TX_STATUS_TJT_SHIFT                      (0U)
#define ENET_MAC_RX_TX_STATUS_TJT(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_RX_TX_STATUS_TJT_SHIFT)) & ENET_MAC_RX_TX_STATUS_TJT_MASK)
/* ENET_MAC_PMT_CONTROL_STATUS Register */
#define ENET_MAC_PMT_CONTROL_STATUS_RWKFILTRST_MASK                (0x80000000U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKFILTRST_SHIFT               (31U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKFILTRST(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_RWKFILTRST_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_RWKFILTRST_MASK)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPTR_MASK                    (0x1F000000U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPTR_SHIFT                   (24U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPTR(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_RWKPTR_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_RWKPTR_MASK)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPFE_MASK                    (0x400U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPFE_SHIFT                   (10U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPFE(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_RWKPFE_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_RWKPFE_MASK)
#define ENET_MAC_PMT_CONTROL_STATUS_GLBLUCAST_MASK                 (0x200U)
#define ENET_MAC_PMT_CONTROL_STATUS_GLBLUCAST_SHIFT                (9U)
#define ENET_MAC_PMT_CONTROL_STATUS_GLBLUCAST(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_GLBLUCAST_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_GLBLUCAST_MASK)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPRCVD_MASK                  (0x40U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPRCVD_SHIFT                 (6U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPRCVD(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_RWKPRCVD_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_RWKPRCVD_MASK)
#define ENET_MAC_PMT_CONTROL_STATUS_MGKPRCVD_MASK                  (0x20U)
#define ENET_MAC_PMT_CONTROL_STATUS_MGKPRCVD_SHIFT                 (5U)
#define ENET_MAC_PMT_CONTROL_STATUS_MGKPRCVD(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_MGKPRCVD_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_MGKPRCVD_MASK)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPKTEN_MASK                  (0x4U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPKTEN_SHIFT                 (2U)
#define ENET_MAC_PMT_CONTROL_STATUS_RWKPKTEN(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_RWKPKTEN_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_RWKPKTEN_MASK)
#define ENET_MAC_PMT_CONTROL_STATUS_MGKPKTEN_MASK                  (0x2U)
#define ENET_MAC_PMT_CONTROL_STATUS_MGKPKTEN_SHIFT                 (1U)
#define ENET_MAC_PMT_CONTROL_STATUS_MGKPKTEN(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_MGKPKTEN_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_MGKPKTEN_MASK)
#define ENET_MAC_PMT_CONTROL_STATUS_PWRDWN_MASK                    (0x1U)
#define ENET_MAC_PMT_CONTROL_STATUS_PWRDWN_SHIFT                   (0U)
#define ENET_MAC_PMT_CONTROL_STATUS_PWRDWN(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PMT_CONTROL_STATUS_PWRDWN_SHIFT)) & ENET_MAC_PMT_CONTROL_STATUS_PWRDWN_MASK)
/* ENET_MAC_RWK_PACKET_FILTER Register */
#define ENET_MAC_RWK_PACKET_FILTER_WKUPFRMFTR_MASK                (0xFFFFFFFFU)
#define ENET_MAC_RWK_PACKET_FILTER_WKUPFRMFTR_SHIFT               (0U)
#define ENET_MAC_RWK_PACKET_FILTER_WKUPFRMFTR(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_RWK_PACKET_FILTER_WKUPFRMFTR_SHIFT)) & ENET_MAC_RWK_PACKET_FILTER_WKUPFRMFTR_MASK)
/* ENET_MAC_LPI_CONTROL_STATUS Register */
#define ENET_MAC_LPI_CONTROL_STATUS_LPITCSE_MASK                   (0x200000U)
#define ENET_MAC_LPI_CONTROL_STATUS_LPITCSE_SHIFT                  (21U)
#define ENET_MAC_LPI_CONTROL_STATUS_LPITCSE(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_LPITCSE_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_LPITCSE_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_LPIATE_MASK                    (0x100000U)
#define ENET_MAC_LPI_CONTROL_STATUS_LPIATE_SHIFT                   (20U)
#define ENET_MAC_LPI_CONTROL_STATUS_LPIATE(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_LPIATE_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_LPIATE_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_LPITXA_MASK                    (0x80000U)
#define ENET_MAC_LPI_CONTROL_STATUS_LPITXA_SHIFT                   (19U)
#define ENET_MAC_LPI_CONTROL_STATUS_LPITXA(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_LPITXA_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_LPITXA_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_PLSEN_MASK                     (0x40000U)
#define ENET_MAC_LPI_CONTROL_STATUS_PLSEN_SHIFT                    (18U)
#define ENET_MAC_LPI_CONTROL_STATUS_PLSEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_PLSEN_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_PLSEN_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_PLS_MASK                       (0x20000U)
#define ENET_MAC_LPI_CONTROL_STATUS_PLS_SHIFT                      (17U)
#define ENET_MAC_LPI_CONTROL_STATUS_PLS(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_PLS_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_PLS_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_LPIEN_MASK                     (0x10000U)
#define ENET_MAC_LPI_CONTROL_STATUS_LPIEN_SHIFT                    (16U)
#define ENET_MAC_LPI_CONTROL_STATUS_LPIEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_LPIEN_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_LPIEN_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIST_MASK                    (0x200U)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIST_SHIFT                   (9U)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIST(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_RLPIST_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_RLPIST_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIST_MASK                    (0x100U)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIST_SHIFT                   (8U)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIST(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_TLPIST_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_TLPIST_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIEX_MASK                    (0x8U)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIEX_SHIFT                   (3U)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIEX(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_RLPIEX_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_RLPIEX_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIEN_MASK                    (0x4U)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIEN_SHIFT                   (2U)
#define ENET_MAC_LPI_CONTROL_STATUS_RLPIEN(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_RLPIEN_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_RLPIEN_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIEX_MASK                    (0x2U)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIEX_SHIFT                   (1U)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIEX(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_TLPIEX_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_TLPIEX_MASK)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIEN_MASK                    (0x1U)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIEN_SHIFT                   (0U)
#define ENET_MAC_LPI_CONTROL_STATUS_TLPIEN(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LPI_CONTROL_STATUS_TLPIEN_SHIFT)) & ENET_MAC_LPI_CONTROL_STATUS_TLPIEN_MASK)
/* ENET_MAC_LPI_TIMERS_CONTROL Register */
#define ENET_MAC_LPI_TIMERS_CONTROL_LST_MASK                       (0x3FF0000U)
#define ENET_MAC_LPI_TIMERS_CONTROL_LST_SHIFT                      (16U)
#define ENET_MAC_LPI_TIMERS_CONTROL_LST(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_LPI_TIMERS_CONTROL_LST_SHIFT)) & ENET_MAC_LPI_TIMERS_CONTROL_LST_MASK)
#define ENET_MAC_LPI_TIMERS_CONTROL_TWT_MASK                       (0xFFFFU)
#define ENET_MAC_LPI_TIMERS_CONTROL_TWT_SHIFT                      (0U)
#define ENET_MAC_LPI_TIMERS_CONTROL_TWT(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_LPI_TIMERS_CONTROL_TWT_SHIFT)) & ENET_MAC_LPI_TIMERS_CONTROL_TWT_MASK)
/* ENET_MAC_LPI_ENTRY_TIMER Register */
#define ENET_MAC_LPI_ENTRY_TIMER_LPIET_MASK                     (0xFFFF8U)
#define ENET_MAC_LPI_ENTRY_TIMER_LPIET_SHIFT                    (3U)
#define ENET_MAC_LPI_ENTRY_TIMER_LPIET(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_LPI_ENTRY_TIMER_LPIET_SHIFT)) & ENET_MAC_LPI_ENTRY_TIMER_LPIET_MASK)
/* ENET_MAC_1US_TIC_COUNTER Register */
#define ENET_MAC_1US_TIC_COUNTER_TIC_1US_CNTR_MASK              (0xFFFU)
#define ENET_MAC_1US_TIC_COUNTER_TIC_1US_CNTR_SHIFT             (0U)
#define ENET_MAC_1US_TIC_COUNTER_TIC_1US_CNTR(x)                (((uint32)(((uint32)(x)) << ENET_MAC_1US_TIC_COUNTER_TIC_1US_CNTR_SHIFT)) & ENET_MAC_1US_TIC_COUNTER_TIC_1US_CNTR_MASK)
/* ENET_MAC_AN_CONTROL Register */
#define ENET_MAC_AN_CONTROL_SGMRAL_MASK                     (0x40000U)
#define ENET_MAC_AN_CONTROL_SGMRAL_SHIFT                    (18U)
#define ENET_MAC_AN_CONTROL_SGMRAL(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_AN_CONTROL_SGMRAL_SHIFT)) & ENET_MAC_AN_CONTROL_SGMRAL_MASK)
#define ENET_MAC_AN_CONTROL_LR_MASK                         (0x20000U)
#define ENET_MAC_AN_CONTROL_LR_SHIFT                        (17U)
#define ENET_MAC_AN_CONTROL_LR(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_AN_CONTROL_LR_SHIFT)) & ENET_MAC_AN_CONTROL_LR_MASK)
#define ENET_MAC_AN_CONTROL_ECD_MASK                        (0x10000U)
#define ENET_MAC_AN_CONTROL_ECD_SHIFT                       (16U)
#define ENET_MAC_AN_CONTROL_ECD(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_CONTROL_ECD_SHIFT)) & ENET_MAC_AN_CONTROL_ECD_MASK)
#define ENET_MAC_AN_CONTROL_ELE_MASK                        (0x4000U)
#define ENET_MAC_AN_CONTROL_ELE_SHIFT                       (14U)
#define ENET_MAC_AN_CONTROL_ELE(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_CONTROL_ELE_SHIFT)) & ENET_MAC_AN_CONTROL_ELE_MASK)
#define ENET_MAC_AN_CONTROL_ANE_MASK                        (0x1000U)
#define ENET_MAC_AN_CONTROL_ANE_SHIFT                       (12U)
#define ENET_MAC_AN_CONTROL_ANE(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_CONTROL_ANE_SHIFT)) & ENET_MAC_AN_CONTROL_ANE_MASK)
#define ENET_MAC_AN_CONTROL_RAN_MASK                        (0x200U)
#define ENET_MAC_AN_CONTROL_RAN_SHIFT                       (9U)
#define ENET_MAC_AN_CONTROL_RAN(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_CONTROL_RAN_SHIFT)) & ENET_MAC_AN_CONTROL_RAN_MASK)
/* ENET_MAC_AN_STATUS Register */
#define ENET_MAC_AN_STATUS_ES_MASK                          (0x100U)
#define ENET_MAC_AN_STATUS_ES_SHIFT                         (8U)
#define ENET_MAC_AN_STATUS_ES(x)                            (((uint32)(((uint32)(x)) << ENET_MAC_AN_STATUS_ES_SHIFT)) & ENET_MAC_AN_STATUS_ES_MASK)
#define ENET_MAC_AN_STATUS_ANC_MASK                         (0x20U)
#define ENET_MAC_AN_STATUS_ANC_SHIFT                        (5U)
#define ENET_MAC_AN_STATUS_ANC(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_AN_STATUS_ANC_SHIFT)) & ENET_MAC_AN_STATUS_ANC_MASK)
#define ENET_MAC_AN_STATUS_ANA_MASK                         (0x8U)
#define ENET_MAC_AN_STATUS_ANA_SHIFT                        (3U)
#define ENET_MAC_AN_STATUS_ANA(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_AN_STATUS_ANA_SHIFT)) & ENET_MAC_AN_STATUS_ANA_MASK)
#define ENET_MAC_AN_STATUS_LS_MASK                          (0x4U)
#define ENET_MAC_AN_STATUS_LS_SHIFT                         (2U)
#define ENET_MAC_AN_STATUS_LS(x)                            (((uint32)(((uint32)(x)) << ENET_MAC_AN_STATUS_LS_SHIFT)) & ENET_MAC_AN_STATUS_LS_MASK)
/* ENET_MAC_AN_ADVERTISEMENT Register */
#define ENET_MAC_AN_ADVERTISEMENT_NP_MASK                        (0x8000U)
#define ENET_MAC_AN_ADVERTISEMENT_NP_SHIFT                       (15U)
#define ENET_MAC_AN_ADVERTISEMENT_NP(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_ADVERTISEMENT_NP_SHIFT)) & ENET_MAC_AN_ADVERTISEMENT_NP_MASK)
#define ENET_MAC_AN_ADVERTISEMENT_RFE_MASK                       (0x3000U)
#define ENET_MAC_AN_ADVERTISEMENT_RFE_SHIFT                      (12U)
#define ENET_MAC_AN_ADVERTISEMENT_RFE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_AN_ADVERTISEMENT_RFE_SHIFT)) & ENET_MAC_AN_ADVERTISEMENT_RFE_MASK)
#define ENET_MAC_AN_ADVERTISEMENT_PSE_MASK                       (0x180U)
#define ENET_MAC_AN_ADVERTISEMENT_PSE_SHIFT                      (7U)
#define ENET_MAC_AN_ADVERTISEMENT_PSE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_AN_ADVERTISEMENT_PSE_SHIFT)) & ENET_MAC_AN_ADVERTISEMENT_PSE_MASK)
#define ENET_MAC_AN_ADVERTISEMENT_HD_MASK                        (0x40U)
#define ENET_MAC_AN_ADVERTISEMENT_HD_SHIFT                       (6U)
#define ENET_MAC_AN_ADVERTISEMENT_HD(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_ADVERTISEMENT_HD_SHIFT)) & ENET_MAC_AN_ADVERTISEMENT_HD_MASK)
#define ENET_MAC_AN_ADVERTISEMENT_FD_MASK                        (0x20U)
#define ENET_MAC_AN_ADVERTISEMENT_FD_SHIFT                       (5U)
#define ENET_MAC_AN_ADVERTISEMENT_FD(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_ADVERTISEMENT_FD_SHIFT)) & ENET_MAC_AN_ADVERTISEMENT_FD_MASK)
/* ENET_MAC_AN_LINK_PARTNER_ABILITY Register */
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_NP_MASK                        (0x8000U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_NP_SHIFT                       (15U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_NP(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_LINK_PARTNER_ABILITY_NP_SHIFT)) & ENET_MAC_AN_LINK_PARTNER_ABILITY_NP_MASK)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_RFE_MASK                       (0x3000U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_RFE_SHIFT                      (12U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_RFE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_AN_LINK_PARTNER_ABILITY_RFE_SHIFT)) & ENET_MAC_AN_LINK_PARTNER_ABILITY_RFE_MASK)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_PSE_MASK                       (0x180U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_PSE_SHIFT                      (7U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_PSE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_AN_LINK_PARTNER_ABILITY_PSE_SHIFT)) & ENET_MAC_AN_LINK_PARTNER_ABILITY_PSE_MASK)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_HD_MASK                        (0x40U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_HD_SHIFT                       (6U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_HD(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_LINK_PARTNER_ABILITY_HD_SHIFT)) & ENET_MAC_AN_LINK_PARTNER_ABILITY_HD_MASK)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_FD_MASK                        (0x20U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_FD_SHIFT                       (5U)
#define ENET_MAC_AN_LINK_PARTNER_ABILITY_FD(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_AN_LINK_PARTNER_ABILITY_FD_SHIFT)) & ENET_MAC_AN_LINK_PARTNER_ABILITY_FD_MASK)
/* ENET_MAC_AN_EXPANSION Register */
#define ENET_MAC_AN_EXPANSION_NPA_MASK                       (0x4U)
#define ENET_MAC_AN_EXPANSION_NPA_SHIFT                      (2U)
#define ENET_MAC_AN_EXPANSION_NPA(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_AN_EXPANSION_NPA_SHIFT)) & ENET_MAC_AN_EXPANSION_NPA_MASK)
#define ENET_MAC_AN_EXPANSION_NPR_MASK                       (0x2U)
#define ENET_MAC_AN_EXPANSION_NPR_SHIFT                      (1U)
#define ENET_MAC_AN_EXPANSION_NPR(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_AN_EXPANSION_NPR_SHIFT)) & ENET_MAC_AN_EXPANSION_NPR_MASK)
/* ENET_MAC_TBI_EXTENDED_STATUS Register */
#define ENET_MAC_TBI_EXTENDED_STATUS_GFD_MASK                       (0x8000U)
#define ENET_MAC_TBI_EXTENDED_STATUS_GFD_SHIFT                      (15U)
#define ENET_MAC_TBI_EXTENDED_STATUS_GFD(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_TBI_EXTENDED_STATUS_GFD_SHIFT)) & ENET_MAC_TBI_EXTENDED_STATUS_GFD_MASK)
#define ENET_MAC_TBI_EXTENDED_STATUS_GHD_MASK                       (0x4000U)
#define ENET_MAC_TBI_EXTENDED_STATUS_GHD_SHIFT                      (14U)
#define ENET_MAC_TBI_EXTENDED_STATUS_GHD(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_TBI_EXTENDED_STATUS_GHD_SHIFT)) & ENET_MAC_TBI_EXTENDED_STATUS_GHD_MASK)
/* ENET_MAC_PHYIF_CONTROL_STATUS Register */
#define ENET_MAC_PHYIF_CONTROL_STATUS_FALSCARDET_MASK                (0x200000U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_FALSCARDET_SHIFT               (21U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_FALSCARDET(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_FALSCARDET_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_FALSCARDET_MASK)
#define ENET_MAC_PHYIF_CONTROL_STATUS_JABTO_MASK                     (0x100000U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_JABTO_SHIFT                    (20U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_JABTO(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_JABTO_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_JABTO_MASK)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKSTS_MASK                    (0x80000U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKSTS_SHIFT                   (19U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKSTS(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_LNKSTS_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_LNKSTS_MASK)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKSPEED_MASK                  (0x60000U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKSPEED_SHIFT                 (17U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKSPEED(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_LNKSPEED_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_LNKSPEED_MASK)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKMOD_MASK                    (0x10000U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKMOD_SHIFT                   (16U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LNKMOD(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_LNKMOD_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_LNKMOD_MASK)
#define ENET_MAC_PHYIF_CONTROL_STATUS_SMIDRXS_MASK                   (0x10U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_SMIDRXS_SHIFT                  (4U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_SMIDRXS(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_SMIDRXS_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_SMIDRXS_MASK)
#define ENET_MAC_PHYIF_CONTROL_STATUS_SFTERR_MASK                    (0x4U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_SFTERR_SHIFT                   (2U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_SFTERR(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_SFTERR_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_SFTERR_MASK)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LUD_MASK                       (0x2U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LUD_SHIFT                      (1U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_LUD(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_LUD_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_LUD_MASK)
#define ENET_MAC_PHYIF_CONTROL_STATUS_TC_MASK                        (0x1U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_TC_SHIFT                       (0U)
#define ENET_MAC_PHYIF_CONTROL_STATUS_TC(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_PHYIF_CONTROL_STATUS_TC_SHIFT)) & ENET_MAC_PHYIF_CONTROL_STATUS_TC_MASK)
/* ENET_MAC_VERSION Register */
#define ENET_MAC_VERSION_USERVER_MASK                       (0xFF00U)
#define ENET_MAC_VERSION_USERVER_SHIFT                      (8U)
#define ENET_MAC_VERSION_USERVER(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_VERSION_USERVER_SHIFT)) & ENET_MAC_VERSION_USERVER_MASK)
#define ENET_MAC_VERSION_SNPSVER_MASK                       (0xFFU)
#define ENET_MAC_VERSION_SNPSVER_SHIFT                      (0U)
#define ENET_MAC_VERSION_SNPSVER(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_VERSION_SNPSVER_SHIFT)) & ENET_MAC_VERSION_SNPSVER_MASK)
/* ENET_MAC_DEBUG Register */
#define ENET_MAC_DEBUG_TFCSTS_MASK                          (0x60000U)
#define ENET_MAC_DEBUG_TFCSTS_SHIFT                         (17U)
#define ENET_MAC_DEBUG_TFCSTS(x)                            (((uint32)(((uint32)(x)) << ENET_MAC_DEBUG_TFCSTS_SHIFT)) & ENET_MAC_DEBUG_TFCSTS_MASK)
#define ENET_MAC_DEBUG_TPESTS_MASK                          (0x10000U)
#define ENET_MAC_DEBUG_TPESTS_SHIFT                         (16U)
#define ENET_MAC_DEBUG_TPESTS(x)                            (((uint32)(((uint32)(x)) << ENET_MAC_DEBUG_TPESTS_SHIFT)) & ENET_MAC_DEBUG_TPESTS_MASK)
#define ENET_MAC_DEBUG_RFCFCSTS_MASK                        (0x6U)
#define ENET_MAC_DEBUG_RFCFCSTS_SHIFT                       (1U)
#define ENET_MAC_DEBUG_RFCFCSTS(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_DEBUG_RFCFCSTS_SHIFT)) & ENET_MAC_DEBUG_RFCFCSTS_MASK)
#define ENET_MAC_DEBUG_RPESTS_MASK                          (0x1U)
#define ENET_MAC_DEBUG_RPESTS_SHIFT                         (0U)
#define ENET_MAC_DEBUG_RPESTS(x)                            (((uint32)(((uint32)(x)) << ENET_MAC_DEBUG_RPESTS_SHIFT)) & ENET_MAC_DEBUG_RPESTS_MASK)
/* ENET_MAC_HW_FEATURE0 Register */
#define ENET_MAC_HW_FEATURE0_ACTPHYSEL_MASK                 (0x70000000U)
#define ENET_MAC_HW_FEATURE0_ACTPHYSEL_SHIFT                (28U)
#define ENET_MAC_HW_FEATURE0_ACTPHYSEL(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_ACTPHYSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_ACTPHYSEL_MASK)
#define ENET_MAC_HW_FEATURE0_SAVLANINS_MASK                 (0x8000000U)
#define ENET_MAC_HW_FEATURE0_SAVLANINS_SHIFT                (27U)
#define ENET_MAC_HW_FEATURE0_SAVLANINS(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_SAVLANINS_SHIFT)) & ENET_MAC_HW_FEATURE0_SAVLANINS_MASK)
#define ENET_MAC_HW_FEATURE0_TSSTSSEL_MASK                  (0x6000000U)
#define ENET_MAC_HW_FEATURE0_TSSTSSEL_SHIFT                 (25U)
#define ENET_MAC_HW_FEATURE0_TSSTSSEL(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_TSSTSSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_TSSTSSEL_MASK)
#define ENET_MAC_HW_FEATURE0_MACADR64SEL_MASK               (0x1000000U)
#define ENET_MAC_HW_FEATURE0_MACADR64SEL_SHIFT              (24U)
#define ENET_MAC_HW_FEATURE0_MACADR64SEL(x)                 (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_MACADR64SEL_SHIFT)) & ENET_MAC_HW_FEATURE0_MACADR64SEL_MASK)
#define ENET_MAC_HW_FEATURE0_MACADR32SEL_MASK               (0x800000U)
#define ENET_MAC_HW_FEATURE0_MACADR32SEL_SHIFT              (23U)
#define ENET_MAC_HW_FEATURE0_MACADR32SEL(x)                 (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_MACADR32SEL_SHIFT)) & ENET_MAC_HW_FEATURE0_MACADR32SEL_MASK)
#define ENET_MAC_HW_FEATURE0_ADDMACADRSEL_MASK              (0x7C0000U)
#define ENET_MAC_HW_FEATURE0_ADDMACADRSEL_SHIFT             (18U)
#define ENET_MAC_HW_FEATURE0_ADDMACADRSEL(x)                (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_ADDMACADRSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_ADDMACADRSEL_MASK)
#define ENET_MAC_HW_FEATURE0_RXCOESEL_MASK                  (0x10000U)
#define ENET_MAC_HW_FEATURE0_RXCOESEL_SHIFT                 (16U)
#define ENET_MAC_HW_FEATURE0_RXCOESEL(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_RXCOESEL_SHIFT)) & ENET_MAC_HW_FEATURE0_RXCOESEL_MASK)
#define ENET_MAC_HW_FEATURE0_TXCOESEL_MASK                  (0x4000U)
#define ENET_MAC_HW_FEATURE0_TXCOESEL_SHIFT                 (14U)
#define ENET_MAC_HW_FEATURE0_TXCOESEL(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_TXCOESEL_SHIFT)) & ENET_MAC_HW_FEATURE0_TXCOESEL_MASK)
#define ENET_MAC_HW_FEATURE0_EEESEL_MASK                    (0x2000U)
#define ENET_MAC_HW_FEATURE0_EEESEL_SHIFT                   (13U)
#define ENET_MAC_HW_FEATURE0_EEESEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_EEESEL_SHIFT)) & ENET_MAC_HW_FEATURE0_EEESEL_MASK)
#define ENET_MAC_HW_FEATURE0_TSSEL_MASK                     (0x1000U)
#define ENET_MAC_HW_FEATURE0_TSSEL_SHIFT                    (12U)
#define ENET_MAC_HW_FEATURE0_TSSEL(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_TSSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_TSSEL_MASK)
#define ENET_MAC_HW_FEATURE0_ARPOFFSEL_MASK                 (0x200U)
#define ENET_MAC_HW_FEATURE0_ARPOFFSEL_SHIFT                (9U)
#define ENET_MAC_HW_FEATURE0_ARPOFFSEL(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_ARPOFFSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_ARPOFFSEL_MASK)
#define ENET_MAC_HW_FEATURE0_MMCSEL_MASK                    (0x100U)
#define ENET_MAC_HW_FEATURE0_MMCSEL_SHIFT                   (8U)
#define ENET_MAC_HW_FEATURE0_MMCSEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_MMCSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_MMCSEL_MASK)
#define ENET_MAC_HW_FEATURE0_MGKSEL_MASK                    (0x80U)
#define ENET_MAC_HW_FEATURE0_MGKSEL_SHIFT                   (7U)
#define ENET_MAC_HW_FEATURE0_MGKSEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_MGKSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_MGKSEL_MASK)
#define ENET_MAC_HW_FEATURE0_RWKSEL_MASK                    (0x40U)
#define ENET_MAC_HW_FEATURE0_RWKSEL_SHIFT                   (6U)
#define ENET_MAC_HW_FEATURE0_RWKSEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_RWKSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_RWKSEL_MASK)
#define ENET_MAC_HW_FEATURE0_SMASEL_MASK                    (0x20U)
#define ENET_MAC_HW_FEATURE0_SMASEL_SHIFT                   (5U)
#define ENET_MAC_HW_FEATURE0_SMASEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_SMASEL_SHIFT)) & ENET_MAC_HW_FEATURE0_SMASEL_MASK)
#define ENET_MAC_HW_FEATURE0_VLHASH_MASK                    (0x10U)
#define ENET_MAC_HW_FEATURE0_VLHASH_SHIFT                   (4U)
#define ENET_MAC_HW_FEATURE0_VLHASH(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_VLHASH_SHIFT)) & ENET_MAC_HW_FEATURE0_VLHASH_MASK)
#define ENET_MAC_HW_FEATURE0_PCSSEL_MASK                    (0x8U)
#define ENET_MAC_HW_FEATURE0_PCSSEL_SHIFT                   (3U)
#define ENET_MAC_HW_FEATURE0_PCSSEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_PCSSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_PCSSEL_MASK)
#define ENET_MAC_HW_FEATURE0_HDSEL_MASK                     (0x4U)
#define ENET_MAC_HW_FEATURE0_HDSEL_SHIFT                    (2U)
#define ENET_MAC_HW_FEATURE0_HDSEL(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_HDSEL_SHIFT)) & ENET_MAC_HW_FEATURE0_HDSEL_MASK)
#define ENET_MAC_HW_FEATURE0_GMIISEL_MASK                   (0x2U)
#define ENET_MAC_HW_FEATURE0_GMIISEL_SHIFT                  (1U)
#define ENET_MAC_HW_FEATURE0_GMIISEL(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_GMIISEL_SHIFT)) & ENET_MAC_HW_FEATURE0_GMIISEL_MASK)
#define ENET_MAC_HW_FEATURE0_MIISEL_MASK                    (0x1U)
#define ENET_MAC_HW_FEATURE0_MIISEL_SHIFT                   (0U)
#define ENET_MAC_HW_FEATURE0_MIISEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE0_MIISEL_SHIFT)) & ENET_MAC_HW_FEATURE0_MIISEL_MASK)
/* ENET_MAC_HW_FEATURE1 Register */
#define ENET_MAC_HW_FEATURE1_L3L4FNUM_MASK                  (0x78000000U)
#define ENET_MAC_HW_FEATURE1_L3L4FNUM_SHIFT                 (27U)
#define ENET_MAC_HW_FEATURE1_L3L4FNUM(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_L3L4FNUM_SHIFT)) & ENET_MAC_HW_FEATURE1_L3L4FNUM_MASK)
#define ENET_MAC_HW_FEATURE1_HASHTBLSZ_MASK                 (0x3000000U)
#define ENET_MAC_HW_FEATURE1_HASHTBLSZ_SHIFT                (24U)
#define ENET_MAC_HW_FEATURE1_HASHTBLSZ(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_HASHTBLSZ_SHIFT)) & ENET_MAC_HW_FEATURE1_HASHTBLSZ_MASK)
#define ENET_MAC_HW_FEATURE1_POUOST_MASK                    (0x800000U)
#define ENET_MAC_HW_FEATURE1_POUOST_SHIFT                   (23U)
#define ENET_MAC_HW_FEATURE1_POUOST(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_POUOST_SHIFT)) & ENET_MAC_HW_FEATURE1_POUOST_MASK)
#define ENET_MAC_HW_FEATURE1_RAVSEL_MASK                    (0x200000U)
#define ENET_MAC_HW_FEATURE1_RAVSEL_SHIFT                   (21U)
#define ENET_MAC_HW_FEATURE1_RAVSEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_RAVSEL_SHIFT)) & ENET_MAC_HW_FEATURE1_RAVSEL_MASK)
#define ENET_MAC_HW_FEATURE1_AVSEL_MASK                     (0x100000U)
#define ENET_MAC_HW_FEATURE1_AVSEL_SHIFT                    (20U)
#define ENET_MAC_HW_FEATURE1_AVSEL(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_AVSEL_SHIFT)) & ENET_MAC_HW_FEATURE1_AVSEL_MASK)
#define ENET_MAC_HW_FEATURE1_DBGMEMA_MASK                   (0x80000U)
#define ENET_MAC_HW_FEATURE1_DBGMEMA_SHIFT                  (19U)
#define ENET_MAC_HW_FEATURE1_DBGMEMA(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_DBGMEMA_SHIFT)) & ENET_MAC_HW_FEATURE1_DBGMEMA_MASK)
#define ENET_MAC_HW_FEATURE1_TSOEN_MASK                     (0x40000U)
#define ENET_MAC_HW_FEATURE1_TSOEN_SHIFT                    (18U)
#define ENET_MAC_HW_FEATURE1_TSOEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_TSOEN_SHIFT)) & ENET_MAC_HW_FEATURE1_TSOEN_MASK)
#define ENET_MAC_HW_FEATURE1_SPHEN_MASK                     (0x20000U)
#define ENET_MAC_HW_FEATURE1_SPHEN_SHIFT                    (17U)
#define ENET_MAC_HW_FEATURE1_SPHEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_SPHEN_SHIFT)) & ENET_MAC_HW_FEATURE1_SPHEN_MASK)
#define ENET_MAC_HW_FEATURE1_DCBEN_MASK                     (0x10000U)
#define ENET_MAC_HW_FEATURE1_DCBEN_SHIFT                    (16U)
#define ENET_MAC_HW_FEATURE1_DCBEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_DCBEN_SHIFT)) & ENET_MAC_HW_FEATURE1_DCBEN_MASK)
#define ENET_MAC_HW_FEATURE1_ADDR64_MASK                    (0xC000U)
#define ENET_MAC_HW_FEATURE1_ADDR64_SHIFT                   (14U)
#define ENET_MAC_HW_FEATURE1_ADDR64(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_ADDR64_SHIFT)) & ENET_MAC_HW_FEATURE1_ADDR64_MASK)
#define ENET_MAC_HW_FEATURE1_ADVTHWORD_MASK                 (0x2000U)
#define ENET_MAC_HW_FEATURE1_ADVTHWORD_SHIFT                (13U)
#define ENET_MAC_HW_FEATURE1_ADVTHWORD(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_ADVTHWORD_SHIFT)) & ENET_MAC_HW_FEATURE1_ADVTHWORD_MASK)
#define ENET_MAC_HW_FEATURE1_PTOEN_MASK                     (0x1000U)
#define ENET_MAC_HW_FEATURE1_PTOEN_SHIFT                    (12U)
#define ENET_MAC_HW_FEATURE1_PTOEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_PTOEN_SHIFT)) & ENET_MAC_HW_FEATURE1_PTOEN_MASK)
#define ENET_MAC_HW_FEATURE1_OSTEN_MASK                     (0x800U)
#define ENET_MAC_HW_FEATURE1_OSTEN_SHIFT                    (11U)
#define ENET_MAC_HW_FEATURE1_OSTEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_OSTEN_SHIFT)) & ENET_MAC_HW_FEATURE1_OSTEN_MASK)
#define ENET_MAC_HW_FEATURE1_TXFIFOSIZE_MASK                (0x700U)
#define ENET_MAC_HW_FEATURE1_TXFIFOSIZE_SHIFT               (8U)
#define ENET_MAC_HW_FEATURE1_TXFIFOSIZE(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_TXFIFOSIZE_SHIFT)) & ENET_MAC_HW_FEATURE1_TXFIFOSIZE_MASK)
#define ENET_MAC_HW_FEATURE1_SPRAM_MASK                     (0x20U)
#define ENET_MAC_HW_FEATURE1_SPRAM_SHIFT                    (5U)
#define ENET_MAC_HW_FEATURE1_SPRAM(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_SPRAM_SHIFT)) & ENET_MAC_HW_FEATURE1_SPRAM_MASK)
#define ENET_MAC_HW_FEATURE1_RXFIFOSIZE_MASK                (0x1FU)
#define ENET_MAC_HW_FEATURE1_RXFIFOSIZE_SHIFT               (0U)
#define ENET_MAC_HW_FEATURE1_RXFIFOSIZE(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE1_RXFIFOSIZE_SHIFT)) & ENET_MAC_HW_FEATURE1_RXFIFOSIZE_MASK)
/* ENET_MAC_HW_FEATURE2 Register */
#define ENET_MAC_HW_FEATURE2_AUXSNAPNUM_MASK                (0x70000000U)
#define ENET_MAC_HW_FEATURE2_AUXSNAPNUM_SHIFT               (28U)
#define ENET_MAC_HW_FEATURE2_AUXSNAPNUM(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE2_AUXSNAPNUM_SHIFT)) & ENET_MAC_HW_FEATURE2_AUXSNAPNUM_MASK)
#define ENET_MAC_HW_FEATURE2_PPSOUTNUM_MASK                 (0x7000000U)
#define ENET_MAC_HW_FEATURE2_PPSOUTNUM_SHIFT                (24U)
#define ENET_MAC_HW_FEATURE2_PPSOUTNUM(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE2_PPSOUTNUM_SHIFT)) & ENET_MAC_HW_FEATURE2_PPSOUTNUM_MASK)
#define ENET_MAC_HW_FEATURE2_TXCHCNT_MASK                   (0x3C0000U)
#define ENET_MAC_HW_FEATURE2_TXCHCNT_SHIFT                  (18U)
#define ENET_MAC_HW_FEATURE2_TXCHCNT(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE2_TXCHCNT_SHIFT)) & ENET_MAC_HW_FEATURE2_TXCHCNT_MASK)
#define ENET_MAC_HW_FEATURE2_RXCHCNT_MASK                   (0xF000U)
#define ENET_MAC_HW_FEATURE2_RXCHCNT_SHIFT                  (12U)
#define ENET_MAC_HW_FEATURE2_RXCHCNT(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE2_RXCHCNT_SHIFT)) & ENET_MAC_HW_FEATURE2_RXCHCNT_MASK)
#define ENET_MAC_HW_FEATURE2_TXQCNT_MASK                    (0x3C0U)
#define ENET_MAC_HW_FEATURE2_TXQCNT_SHIFT                   (6U)
#define ENET_MAC_HW_FEATURE2_TXQCNT(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE2_TXQCNT_SHIFT)) & ENET_MAC_HW_FEATURE2_TXQCNT_MASK)
#define ENET_MAC_HW_FEATURE2_RXQCNT_MASK                    (0xFU)
#define ENET_MAC_HW_FEATURE2_RXQCNT_SHIFT                   (0U)
#define ENET_MAC_HW_FEATURE2_RXQCNT(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE2_RXQCNT_SHIFT)) & ENET_MAC_HW_FEATURE2_RXQCNT_MASK)
/* ENET_MAC_HW_FEATURE3 Register */
#define ENET_MAC_HW_FEATURE3_ASP_MASK                       (0x30000000U)
#define ENET_MAC_HW_FEATURE3_ASP_SHIFT                      (28U)
#define ENET_MAC_HW_FEATURE3_ASP(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_ASP_SHIFT)) & ENET_MAC_HW_FEATURE3_ASP_MASK)
#define ENET_MAC_HW_FEATURE3_TBSSEL_MASK                    (0x8000000U)
#define ENET_MAC_HW_FEATURE3_TBSSEL_SHIFT                   (27U)
#define ENET_MAC_HW_FEATURE3_TBSSEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_TBSSEL_SHIFT)) & ENET_MAC_HW_FEATURE3_TBSSEL_MASK)
#define ENET_MAC_HW_FEATURE3_FPESEL_MASK                    (0x4000000U)
#define ENET_MAC_HW_FEATURE3_FPESEL_SHIFT                   (26U)
#define ENET_MAC_HW_FEATURE3_FPESEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_FPESEL_SHIFT)) & ENET_MAC_HW_FEATURE3_FPESEL_MASK)
#define ENET_MAC_HW_FEATURE3_ESTWID_MASK                    (0x300000U)
#define ENET_MAC_HW_FEATURE3_ESTWID_SHIFT                   (20U)
#define ENET_MAC_HW_FEATURE3_ESTWID(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_ESTWID_SHIFT)) & ENET_MAC_HW_FEATURE3_ESTWID_MASK)
#define ENET_MAC_HW_FEATURE3_ESTDEP_MASK                    (0xE0000U)
#define ENET_MAC_HW_FEATURE3_ESTDEP_SHIFT                   (17U)
#define ENET_MAC_HW_FEATURE3_ESTDEP(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_ESTDEP_SHIFT)) & ENET_MAC_HW_FEATURE3_ESTDEP_MASK)
#define ENET_MAC_HW_FEATURE3_ESTSEL_MASK                    (0x10000U)
#define ENET_MAC_HW_FEATURE3_ESTSEL_SHIFT                   (16U)
#define ENET_MAC_HW_FEATURE3_ESTSEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_ESTSEL_SHIFT)) & ENET_MAC_HW_FEATURE3_ESTSEL_MASK)
#define ENET_MAC_HW_FEATURE3_FRPES_MASK                     (0x6000U)
#define ENET_MAC_HW_FEATURE3_FRPES_SHIFT                    (13U)
#define ENET_MAC_HW_FEATURE3_FRPES(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_FRPES_SHIFT)) & ENET_MAC_HW_FEATURE3_FRPES_MASK)
#define ENET_MAC_HW_FEATURE3_FRPBS_MASK                     (0x1800U)
#define ENET_MAC_HW_FEATURE3_FRPBS_SHIFT                    (11U)
#define ENET_MAC_HW_FEATURE3_FRPBS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_FRPBS_SHIFT)) & ENET_MAC_HW_FEATURE3_FRPBS_MASK)
#define ENET_MAC_HW_FEATURE3_FRPSEL_MASK                    (0x400U)
#define ENET_MAC_HW_FEATURE3_FRPSEL_SHIFT                   (10U)
#define ENET_MAC_HW_FEATURE3_FRPSEL(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_FRPSEL_SHIFT)) & ENET_MAC_HW_FEATURE3_FRPSEL_MASK)
#define ENET_MAC_HW_FEATURE3_PDUPSEL_MASK                   (0x200U)
#define ENET_MAC_HW_FEATURE3_PDUPSEL_SHIFT                  (9U)
#define ENET_MAC_HW_FEATURE3_PDUPSEL(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_PDUPSEL_SHIFT)) & ENET_MAC_HW_FEATURE3_PDUPSEL_MASK)
#define ENET_MAC_HW_FEATURE3_DVLAN_MASK                     (0x20U)
#define ENET_MAC_HW_FEATURE3_DVLAN_SHIFT                    (5U)
#define ENET_MAC_HW_FEATURE3_DVLAN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_DVLAN_SHIFT)) & ENET_MAC_HW_FEATURE3_DVLAN_MASK)
#define ENET_MAC_HW_FEATURE3_CBTISEL_MASK                   (0x10U)
#define ENET_MAC_HW_FEATURE3_CBTISEL_SHIFT                  (4U)
#define ENET_MAC_HW_FEATURE3_CBTISEL(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_CBTISEL_SHIFT)) & ENET_MAC_HW_FEATURE3_CBTISEL_MASK)
#define ENET_MAC_HW_FEATURE3_NRVF_MASK                      (0x7U)
#define ENET_MAC_HW_FEATURE3_NRVF_SHIFT                     (0U)
#define ENET_MAC_HW_FEATURE3_NRVF(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_HW_FEATURE3_NRVF_SHIFT)) & ENET_MAC_HW_FEATURE3_NRVF_MASK)
/* ENET_MAC_DPP_FSM_INTERRUPT_STATUS Register */
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_FSMPES_MASK                    (0x1000000U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_FSMPES_SHIFT                   (24U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_FSMPES(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_FSMPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_FSMPES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_SLVTES_MASK                    (0x20000U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_SLVTES_SHIFT                   (17U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_SLVTES(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_SLVTES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_SLVTES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MSTTES_MASK                    (0x10000U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MSTTES_SHIFT                   (16U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MSTTES(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MSTTES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MSTTES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RVCTES_MASK                    (0x8000U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RVCTES_SHIFT                   (15U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RVCTES(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RVCTES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RVCTES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_R125ES_MASK                    (0x4000U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_R125ES_SHIFT                   (14U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_R125ES(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_R125ES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_R125ES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_T125ES_MASK                    (0x2000U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_T125ES_SHIFT                   (13U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_T125ES(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_T125ES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_T125ES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_PTES_MASK                      (0x1000U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_PTES_SHIFT                     (12U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_PTES(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_PTES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_PTES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATES_MASK                      (0x800U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATES_SHIFT                     (11U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATES(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CTES_MASK                      (0x400U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CTES_SHIFT                     (10U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CTES(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CTES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CTES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RTES_MASK                      (0x200U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RTES_SHIFT                     (9U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RTES(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RTES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RTES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TTES_MASK                      (0x100U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TTES_SHIFT                     (8U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TTES(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TTES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TTES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ASRPES_MASK                    (0x80U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ASRPES_SHIFT                   (7U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ASRPES(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ASRPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ASRPES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CWPES_MASK                     (0x40U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CWPES_SHIFT                    (6U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CWPES(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CWPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_CWPES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ARPES_MASK                     (0x20U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ARPES_SHIFT                    (5U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ARPES(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ARPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ARPES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MTSPES_MASK                    (0x10U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MTSPES_SHIFT                   (4U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MTSPES(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MTSPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MTSPES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MPES_MASK                      (0x8U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MPES_SHIFT                     (3U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MPES(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_MPES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RPES_MASK                      (0x4U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RPES_SHIFT                     (2U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RPES(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_RPES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TPES_MASK                      (0x2U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TPES_SHIFT                     (1U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TPES(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_TPES_MASK)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATPES_MASK                     (0x1U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATPES_SHIFT                    (0U)
#define ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATPES(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATPES_SHIFT)) & ENET_MAC_DPP_FSM_INTERRUPT_STATUS_ATPES_MASK)
/* ENET_MAC_AXI_SLV_DPE_ADDR_STATUS Register */
#define ENET_MAC_AXI_SLV_DPE_ADDR_STATUS_ASPEAS_MASK                    (0x3FFFU)
#define ENET_MAC_AXI_SLV_DPE_ADDR_STATUS_ASPEAS_SHIFT                   (0U)
#define ENET_MAC_AXI_SLV_DPE_ADDR_STATUS_ASPEAS(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_AXI_SLV_DPE_ADDR_STATUS_ASPEAS_SHIFT)) & ENET_MAC_AXI_SLV_DPE_ADDR_STATUS_ASPEAS_MASK)
/* ENET_MAC_FSM_CONTROL Register */
#define ENET_MAC_FSM_CONTROL_RVCLGRNML_MASK                 (0x80000000U)
#define ENET_MAC_FSM_CONTROL_RVCLGRNML_SHIFT                (31U)
#define ENET_MAC_FSM_CONTROL_RVCLGRNML(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_RVCLGRNML_SHIFT)) & ENET_MAC_FSM_CONTROL_RVCLGRNML_MASK)
#define ENET_MAC_FSM_CONTROL_R125LGRNML_MASK                (0x40000000U)
#define ENET_MAC_FSM_CONTROL_R125LGRNML_SHIFT               (30U)
#define ENET_MAC_FSM_CONTROL_R125LGRNML(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_R125LGRNML_SHIFT)) & ENET_MAC_FSM_CONTROL_R125LGRNML_MASK)
#define ENET_MAC_FSM_CONTROL_T125LGRNML_MASK                (0x20000000U)
#define ENET_MAC_FSM_CONTROL_T125LGRNML_SHIFT               (29U)
#define ENET_MAC_FSM_CONTROL_T125LGRNML(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_T125LGRNML_SHIFT)) & ENET_MAC_FSM_CONTROL_T125LGRNML_MASK)
#define ENET_MAC_FSM_CONTROL_PLGRNML_MASK                   (0x10000000U)
#define ENET_MAC_FSM_CONTROL_PLGRNML_SHIFT                  (28U)
#define ENET_MAC_FSM_CONTROL_PLGRNML(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_PLGRNML_SHIFT)) & ENET_MAC_FSM_CONTROL_PLGRNML_MASK)
#define ENET_MAC_FSM_CONTROL_ALGRNML_MASK                   (0x8000000U)
#define ENET_MAC_FSM_CONTROL_ALGRNML_SHIFT                  (27U)
#define ENET_MAC_FSM_CONTROL_ALGRNML(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_ALGRNML_SHIFT)) & ENET_MAC_FSM_CONTROL_ALGRNML_MASK)
#define ENET_MAC_FSM_CONTROL_CLGRNML_MASK                   (0x4000000U)
#define ENET_MAC_FSM_CONTROL_CLGRNML_SHIFT                  (26U)
#define ENET_MAC_FSM_CONTROL_CLGRNML(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_CLGRNML_SHIFT)) & ENET_MAC_FSM_CONTROL_CLGRNML_MASK)
#define ENET_MAC_FSM_CONTROL_RLGRNML_MASK                   (0x2000000U)
#define ENET_MAC_FSM_CONTROL_RLGRNML_SHIFT                  (25U)
#define ENET_MAC_FSM_CONTROL_RLGRNML(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_RLGRNML_SHIFT)) & ENET_MAC_FSM_CONTROL_RLGRNML_MASK)
#define ENET_MAC_FSM_CONTROL_TLGRNML_MASK                   (0x1000000U)
#define ENET_MAC_FSM_CONTROL_TLGRNML_SHIFT                  (24U)
#define ENET_MAC_FSM_CONTROL_TLGRNML(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_TLGRNML_SHIFT)) & ENET_MAC_FSM_CONTROL_TLGRNML_MASK)
#define ENET_MAC_FSM_CONTROL_RVCPEIN_MASK                   (0x800000U)
#define ENET_MAC_FSM_CONTROL_RVCPEIN_SHIFT                  (23U)
#define ENET_MAC_FSM_CONTROL_RVCPEIN(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_RVCPEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_RVCPEIN_MASK)
#define ENET_MAC_FSM_CONTROL_R125PEIN_MASK                  (0x400000U)
#define ENET_MAC_FSM_CONTROL_R125PEIN_SHIFT                 (22U)
#define ENET_MAC_FSM_CONTROL_R125PEIN(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_R125PEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_R125PEIN_MASK)
#define ENET_MAC_FSM_CONTROL_T125PEIN_MASK                  (0x200000U)
#define ENET_MAC_FSM_CONTROL_T125PEIN_SHIFT                 (21U)
#define ENET_MAC_FSM_CONTROL_T125PEIN(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_T125PEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_T125PEIN_MASK)
#define ENET_MAC_FSM_CONTROL_PPEIN_MASK                     (0x100000U)
#define ENET_MAC_FSM_CONTROL_PPEIN_SHIFT                    (20U)
#define ENET_MAC_FSM_CONTROL_PPEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_PPEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_PPEIN_MASK)
#define ENET_MAC_FSM_CONTROL_APEIN_MASK                     (0x80000U)
#define ENET_MAC_FSM_CONTROL_APEIN_SHIFT                    (19U)
#define ENET_MAC_FSM_CONTROL_APEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_APEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_APEIN_MASK)
#define ENET_MAC_FSM_CONTROL_CPEIN_MASK                     (0x40000U)
#define ENET_MAC_FSM_CONTROL_CPEIN_SHIFT                    (18U)
#define ENET_MAC_FSM_CONTROL_CPEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_CPEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_CPEIN_MASK)
#define ENET_MAC_FSM_CONTROL_RPEIN_MASK                     (0x20000U)
#define ENET_MAC_FSM_CONTROL_RPEIN_SHIFT                    (17U)
#define ENET_MAC_FSM_CONTROL_RPEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_RPEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_RPEIN_MASK)
#define ENET_MAC_FSM_CONTROL_TPEIN_MASK                     (0x10000U)
#define ENET_MAC_FSM_CONTROL_TPEIN_SHIFT                    (16U)
#define ENET_MAC_FSM_CONTROL_TPEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_TPEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_TPEIN_MASK)
#define ENET_MAC_FSM_CONTROL_RVCTEIN_MASK                   (0x8000U)
#define ENET_MAC_FSM_CONTROL_RVCTEIN_SHIFT                  (15U)
#define ENET_MAC_FSM_CONTROL_RVCTEIN(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_RVCTEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_RVCTEIN_MASK)
#define ENET_MAC_FSM_CONTROL_R125TEIN_MASK                  (0x4000U)
#define ENET_MAC_FSM_CONTROL_R125TEIN_SHIFT                 (14U)
#define ENET_MAC_FSM_CONTROL_R125TEIN(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_R125TEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_R125TEIN_MASK)
#define ENET_MAC_FSM_CONTROL_T125TEIN_MASK                  (0x2000U)
#define ENET_MAC_FSM_CONTROL_T125TEIN_SHIFT                 (13U)
#define ENET_MAC_FSM_CONTROL_T125TEIN(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_T125TEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_T125TEIN_MASK)
#define ENET_MAC_FSM_CONTROL_PTEIN_MASK                     (0x1000U)
#define ENET_MAC_FSM_CONTROL_PTEIN_SHIFT                    (12U)
#define ENET_MAC_FSM_CONTROL_PTEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_PTEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_PTEIN_MASK)
#define ENET_MAC_FSM_CONTROL_ATEIN_MASK                     (0x800U)
#define ENET_MAC_FSM_CONTROL_ATEIN_SHIFT                    (11U)
#define ENET_MAC_FSM_CONTROL_ATEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_ATEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_ATEIN_MASK)
#define ENET_MAC_FSM_CONTROL_CTEIN_MASK                     (0x400U)
#define ENET_MAC_FSM_CONTROL_CTEIN_SHIFT                    (10U)
#define ENET_MAC_FSM_CONTROL_CTEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_CTEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_CTEIN_MASK)
#define ENET_MAC_FSM_CONTROL_RTEIN_MASK                     (0x200U)
#define ENET_MAC_FSM_CONTROL_RTEIN_SHIFT                    (9U)
#define ENET_MAC_FSM_CONTROL_RTEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_RTEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_RTEIN_MASK)
#define ENET_MAC_FSM_CONTROL_TTEIN_MASK                     (0x100U)
#define ENET_MAC_FSM_CONTROL_TTEIN_SHIFT                    (8U)
#define ENET_MAC_FSM_CONTROL_TTEIN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_TTEIN_SHIFT)) & ENET_MAC_FSM_CONTROL_TTEIN_MASK)
#define ENET_MAC_FSM_CONTROL_PRTYEN_MASK                    (0x2U)
#define ENET_MAC_FSM_CONTROL_PRTYEN_SHIFT                   (1U)
#define ENET_MAC_FSM_CONTROL_PRTYEN(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_PRTYEN_SHIFT)) & ENET_MAC_FSM_CONTROL_PRTYEN_MASK)
#define ENET_MAC_FSM_CONTROL_TMOUTEN_MASK                   (0x1U)
#define ENET_MAC_FSM_CONTROL_TMOUTEN_SHIFT                  (0U)
#define ENET_MAC_FSM_CONTROL_TMOUTEN(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_FSM_CONTROL_TMOUTEN_SHIFT)) & ENET_MAC_FSM_CONTROL_TMOUTEN_MASK)
/* ENET_MAC_FSM_ACT_TIMER Register */
#define ENET_MAC_FSM_ACT_TIMER_LTMRMD_MASK                    (0xF00000U)
#define ENET_MAC_FSM_ACT_TIMER_LTMRMD_SHIFT                   (20U)
#define ENET_MAC_FSM_ACT_TIMER_LTMRMD(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_FSM_ACT_TIMER_LTMRMD_SHIFT)) & ENET_MAC_FSM_ACT_TIMER_LTMRMD_MASK)
#define ENET_MAC_FSM_ACT_TIMER_NTMRMD_MASK                    (0xF0000U)
#define ENET_MAC_FSM_ACT_TIMER_NTMRMD_SHIFT                   (16U)
#define ENET_MAC_FSM_ACT_TIMER_NTMRMD(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_FSM_ACT_TIMER_NTMRMD_SHIFT)) & ENET_MAC_FSM_ACT_TIMER_NTMRMD_MASK)
#define ENET_MAC_FSM_ACT_TIMER_TMR_MASK                       (0x3FFU)
#define ENET_MAC_FSM_ACT_TIMER_TMR_SHIFT                      (0U)
#define ENET_MAC_FSM_ACT_TIMER_TMR(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_FSM_ACT_TIMER_TMR_SHIFT)) & ENET_MAC_FSM_ACT_TIMER_TMR_MASK)
/* ENET_MAC_MDIO_ADDRESS Register */
#define ENET_MAC_MDIO_ADDRESS_PSE_MASK                       (0x8000000U)
#define ENET_MAC_MDIO_ADDRESS_PSE_SHIFT                      (27U)
#define ENET_MAC_MDIO_ADDRESS_PSE(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_PSE_SHIFT)) & ENET_MAC_MDIO_ADDRESS_PSE_MASK)
#define ENET_MAC_MDIO_ADDRESS_BTB_MASK                       (0x4000000U)
#define ENET_MAC_MDIO_ADDRESS_BTB_SHIFT                      (26U)
#define ENET_MAC_MDIO_ADDRESS_BTB(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_BTB_SHIFT)) & ENET_MAC_MDIO_ADDRESS_BTB_MASK)
#define ENET_MAC_MDIO_ADDRESS_PA_MASK                        (0x3E00000U)
#define ENET_MAC_MDIO_ADDRESS_PA_SHIFT                       (21U)
#define ENET_MAC_MDIO_ADDRESS_PA(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_PA_SHIFT)) & ENET_MAC_MDIO_ADDRESS_PA_MASK)
#define ENET_MAC_MDIO_ADDRESS_RDA_MASK                       (0x1F0000U)
#define ENET_MAC_MDIO_ADDRESS_RDA_SHIFT                      (16U)
#define ENET_MAC_MDIO_ADDRESS_RDA(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_RDA_SHIFT)) & ENET_MAC_MDIO_ADDRESS_RDA_MASK)
#define ENET_MAC_MDIO_ADDRESS_NTC_MASK                       (0x7000U)
#define ENET_MAC_MDIO_ADDRESS_NTC_SHIFT                      (12U)
#define ENET_MAC_MDIO_ADDRESS_NTC(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_NTC_SHIFT)) & ENET_MAC_MDIO_ADDRESS_NTC_MASK)
#define ENET_MAC_MDIO_ADDRESS_CR_MASK                        (0xF00U)
#define ENET_MAC_MDIO_ADDRESS_CR_SHIFT                       (8U)
#define ENET_MAC_MDIO_ADDRESS_CR(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_CR_SHIFT)) & ENET_MAC_MDIO_ADDRESS_CR_MASK)
#define ENET_MAC_MDIO_ADDRESS_SKAP_MASK                      (0x10U)
#define ENET_MAC_MDIO_ADDRESS_SKAP_SHIFT                     (4U)
#define ENET_MAC_MDIO_ADDRESS_SKAP(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_SKAP_SHIFT)) & ENET_MAC_MDIO_ADDRESS_SKAP_MASK)
#define ENET_MAC_MDIO_ADDRESS_GOC_1_MASK                     (0x8U)
#define ENET_MAC_MDIO_ADDRESS_GOC_1_SHIFT                    (3U)
#define ENET_MAC_MDIO_ADDRESS_GOC_1(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_GOC_1_SHIFT)) & ENET_MAC_MDIO_ADDRESS_GOC_1_MASK)
#define ENET_MAC_MDIO_ADDRESS_GOC_0_MASK                     (0x4U)
#define ENET_MAC_MDIO_ADDRESS_GOC_0_SHIFT                    (2U)
#define ENET_MAC_MDIO_ADDRESS_GOC_0(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_GOC_0_SHIFT)) & ENET_MAC_MDIO_ADDRESS_GOC_0_MASK)
#define ENET_MAC_MDIO_ADDRESS_C45E_MASK                      (0x2U)
#define ENET_MAC_MDIO_ADDRESS_C45E_SHIFT                     (1U)
#define ENET_MAC_MDIO_ADDRESS_C45E(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_C45E_SHIFT)) & ENET_MAC_MDIO_ADDRESS_C45E_MASK)
#define ENET_MAC_MDIO_ADDRESS_GB_MASK                        (0x1U)
#define ENET_MAC_MDIO_ADDRESS_GB_SHIFT                       (0U)
#define ENET_MAC_MDIO_ADDRESS_GB(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_ADDRESS_GB_SHIFT)) & ENET_MAC_MDIO_ADDRESS_GB_MASK)
/* ENET_MAC_MDIO_DATA Register */
#define ENET_MAC_MDIO_DATA_RA_MASK                          (0xFFFF0000U)
#define ENET_MAC_MDIO_DATA_RA_SHIFT                         (16U)
#define ENET_MAC_MDIO_DATA_RA(x)                            (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_DATA_RA_SHIFT)) & ENET_MAC_MDIO_DATA_RA_MASK)
#define ENET_MAC_MDIO_DATA_GD_MASK                          (0xFFFFU)
#define ENET_MAC_MDIO_DATA_GD_SHIFT                         (0U)
#define ENET_MAC_MDIO_DATA_GD(x)                            (((uint32)(((uint32)(x)) << ENET_MAC_MDIO_DATA_GD_SHIFT)) & ENET_MAC_MDIO_DATA_GD_MASK)
/* ENET_MAC_CSR_SW_CTRL Register */
#define ENET_MAC_CSR_SW_CTRL_SEEN_MASK                      (0x100U)
#define ENET_MAC_CSR_SW_CTRL_SEEN_SHIFT                     (8U)
#define ENET_MAC_CSR_SW_CTRL_SEEN(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_CSR_SW_CTRL_SEEN_SHIFT)) & ENET_MAC_CSR_SW_CTRL_SEEN_MASK)
#define ENET_MAC_CSR_SW_CTRL_RCWE_MASK                      (0x1U)
#define ENET_MAC_CSR_SW_CTRL_RCWE_SHIFT                     (0U)
#define ENET_MAC_CSR_SW_CTRL_RCWE(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_CSR_SW_CTRL_RCWE_SHIFT)) & ENET_MAC_CSR_SW_CTRL_RCWE_MASK)
/* ENET_MAC_FPE_CTRL_STS Register */
#define ENET_MAC_FPE_CTRL_STS_TRSP_MASK                      (0x80000U)
#define ENET_MAC_FPE_CTRL_STS_TRSP_SHIFT                     (19U)
#define ENET_MAC_FPE_CTRL_STS_TRSP(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_FPE_CTRL_STS_TRSP_SHIFT)) & ENET_MAC_FPE_CTRL_STS_TRSP_MASK)
#define ENET_MAC_FPE_CTRL_STS_TVER_MASK                      (0x40000U)
#define ENET_MAC_FPE_CTRL_STS_TVER_SHIFT                     (18U)
#define ENET_MAC_FPE_CTRL_STS_TVER(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_FPE_CTRL_STS_TVER_SHIFT)) & ENET_MAC_FPE_CTRL_STS_TVER_MASK)
#define ENET_MAC_FPE_CTRL_STS_RRSP_MASK                      (0x20000U)
#define ENET_MAC_FPE_CTRL_STS_RRSP_SHIFT                     (17U)
#define ENET_MAC_FPE_CTRL_STS_RRSP(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_FPE_CTRL_STS_RRSP_SHIFT)) & ENET_MAC_FPE_CTRL_STS_RRSP_MASK)
#define ENET_MAC_FPE_CTRL_STS_RVER_MASK                      (0x10000U)
#define ENET_MAC_FPE_CTRL_STS_RVER_SHIFT                     (16U)
#define ENET_MAC_FPE_CTRL_STS_RVER(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_FPE_CTRL_STS_RVER_SHIFT)) & ENET_MAC_FPE_CTRL_STS_RVER_MASK)
#define ENET_MAC_FPE_CTRL_STS_SRSP_MASK                      (0x4U)
#define ENET_MAC_FPE_CTRL_STS_SRSP_SHIFT                     (2U)
#define ENET_MAC_FPE_CTRL_STS_SRSP(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_FPE_CTRL_STS_SRSP_SHIFT)) & ENET_MAC_FPE_CTRL_STS_SRSP_MASK)
#define ENET_MAC_FPE_CTRL_STS_SVER_MASK                      (0x2U)
#define ENET_MAC_FPE_CTRL_STS_SVER_SHIFT                     (1U)
#define ENET_MAC_FPE_CTRL_STS_SVER(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_FPE_CTRL_STS_SVER_SHIFT)) & ENET_MAC_FPE_CTRL_STS_SVER_MASK)
#define ENET_MAC_FPE_CTRL_STS_EFPE_MASK                      (0x1U)
#define ENET_MAC_FPE_CTRL_STS_EFPE_SHIFT                     (0U)
#define ENET_MAC_FPE_CTRL_STS_EFPE(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_FPE_CTRL_STS_EFPE_SHIFT)) & ENET_MAC_FPE_CTRL_STS_EFPE_MASK)
/* ENET_MAC_EXT_CFG1 Register */
#define ENET_MAC_EXT_CFG1_SPLM_MASK                         (0x300U)
#define ENET_MAC_EXT_CFG1_SPLM_SHIFT                        (8U)
#define ENET_MAC_EXT_CFG1_SPLM(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CFG1_SPLM_SHIFT)) & ENET_MAC_EXT_CFG1_SPLM_MASK)
#define ENET_MAC_EXT_CFG1_SPLOFST_MASK                      (0x7FU)
#define ENET_MAC_EXT_CFG1_SPLOFST_SHIFT                     (0U)
#define ENET_MAC_EXT_CFG1_SPLOFST(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_EXT_CFG1_SPLOFST_SHIFT)) & ENET_MAC_EXT_CFG1_SPLOFST_MASK)
/* ENET_MAC_PRESN_TIME_NS Register */
#define ENET_MAC_PRESN_TIME_NS_MPTN_MASK                      (0xFFFFFFFFU)
#define ENET_MAC_PRESN_TIME_NS_MPTN_SHIFT                     (0U)
#define ENET_MAC_PRESN_TIME_NS_MPTN(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_PRESN_TIME_NS_MPTN_SHIFT)) & ENET_MAC_PRESN_TIME_NS_MPTN_MASK)
/* ENET_MAC_PRESN_TIME_UPDT Register */
#define ENET_MAC_PRESN_TIME_UPDT_MPTU_MASK                      (0xFFFFFFFFU)
#define ENET_MAC_PRESN_TIME_UPDT_MPTU_SHIFT                     (0U)
#define ENET_MAC_PRESN_TIME_UPDT_MPTU(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_PRESN_TIME_UPDT_MPTU_SHIFT)) & ENET_MAC_PRESN_TIME_UPDT_MPTU_MASK)
/* ENET_MAC_ADDRESS_HIGH Register */
#define ENET_MAC_ADDRESS_HIGH_AE_MASK                                   (0x80000000U)
#define ENET_MAC_ADDRESS_HIGH_AE_SHIFT                                  (31U)
#define ENET_MAC_ADDRESS_HIGH_AE(x)                                     (((uint32)(((uint32)(x)) << ENET_MAC_ADDRESS_HIGH_AE_SHIFT)) & ENET_MAC_ADDRESS_HIGH_AE_MASK)
#define ENET_MAC_ADDRESS_HIGH_DCS_MASK                                  (0x70000U)
#define ENET_MAC_ADDRESS_HIGH_DCS_SHIFT                                 (16U)
#define ENET_MAC_ADDRESS_HIGH_DCS(x)                                    (((uint32)(((uint32)(x)) << ENET_MAC_ADDRESS_HIGH_DCS_SHIFT)) & ENET_MAC_ADDRESS_HIGH_DCS_MASK)
#define ENET_MAC_ADDRESS_HIGH_ADDRHI_MASK                               (0xFFFFU)
#define ENET_MAC_ADDRESS_HIGH_ADDRHI_SHIFT                              (0U)
#define ENET_MAC_ADDRESS_HIGH_ADDRHI(x)                                 (((uint32)(((uint32)(x)) << ENET_MAC_ADDRESS_HIGH_ADDRHI_SHIFT)) & ENET_MAC_ADDRESS_HIGH_ADDRHI_MASK)
/* ENET_MAC_ADDRESS_LOW Register */
#define ENET_MAC_ADDRESS_LOW_ADDRLO_MASK                                (0xFFFFFFFFU)
#define ENET_MAC_ADDRESS_LOW_ADDRLO_SHIFT                               (0U)
#define ENET_MAC_ADDRESS_LOW_ADDRLO(x)                                  (((uint32)(((uint32)(x)) << ENET_MAC_ADDRESS_LOW_ADDRLO_SHIFT)) & ENET_MAC_ADDRESS_LOW_ADDRLO_MASK)

/* ENET_MMC_CONTROL Register */
#define ENET_MMC_CONTROL_UCDBC_MASK                         (0x100U)
#define ENET_MMC_CONTROL_UCDBC_SHIFT                        (8U)
#define ENET_MMC_CONTROL_UCDBC(x)                           (((uint32)(((uint32)(x)) << ENET_MMC_CONTROL_UCDBC_SHIFT)) & ENET_MMC_CONTROL_UCDBC_MASK)
#define ENET_MMC_CONTROL_CNTPRSTLVL_MASK                    (0x20U)
#define ENET_MMC_CONTROL_CNTPRSTLVL_SHIFT                   (5U)
#define ENET_MMC_CONTROL_CNTPRSTLVL(x)                      (((uint32)(((uint32)(x)) << ENET_MMC_CONTROL_CNTPRSTLVL_SHIFT)) & ENET_MMC_CONTROL_CNTPRSTLVL_MASK)
#define ENET_MMC_CONTROL_CNTPRST_MASK                       (0x10U)
#define ENET_MMC_CONTROL_CNTPRST_SHIFT                      (4U)
#define ENET_MMC_CONTROL_CNTPRST(x)                         (((uint32)(((uint32)(x)) << ENET_MMC_CONTROL_CNTPRST_SHIFT)) & ENET_MMC_CONTROL_CNTPRST_MASK)
#define ENET_MMC_CONTROL_CNTFREEZ_MASK                      (0x8U)
#define ENET_MMC_CONTROL_CNTFREEZ_SHIFT                     (3U)
#define ENET_MMC_CONTROL_CNTFREEZ(x)                        (((uint32)(((uint32)(x)) << ENET_MMC_CONTROL_CNTFREEZ_SHIFT)) & ENET_MMC_CONTROL_CNTFREEZ_MASK)
#define ENET_MMC_CONTROL_RSTONRD_MASK                       (0x4U)
#define ENET_MMC_CONTROL_RSTONRD_SHIFT                      (2U)
#define ENET_MMC_CONTROL_RSTONRD(x)                         (((uint32)(((uint32)(x)) << ENET_MMC_CONTROL_RSTONRD_SHIFT)) & ENET_MMC_CONTROL_RSTONRD_MASK)
#define ENET_MMC_CONTROL_CNTSTOPRO_MASK                     (0x2U)
#define ENET_MMC_CONTROL_CNTSTOPRO_SHIFT                    (1U)
#define ENET_MMC_CONTROL_CNTSTOPRO(x)                       (((uint32)(((uint32)(x)) << ENET_MMC_CONTROL_CNTSTOPRO_SHIFT)) & ENET_MMC_CONTROL_CNTSTOPRO_MASK)
#define ENET_MMC_CONTROL_CNTRST_MASK                        (0x1U)
#define ENET_MMC_CONTROL_CNTRST_SHIFT                       (0U)
#define ENET_MMC_CONTROL_CNTRST(x)                          (((uint32)(((uint32)(x)) << ENET_MMC_CONTROL_CNTRST_SHIFT)) & ENET_MMC_CONTROL_CNTRST_MASK)
/* ENET_MMC_RX_INTERRUPT Register */
#define ENET_MMC_RX_INTERRUPT_RXLPITRCIS_MASK                (0x8000000U)
#define ENET_MMC_RX_INTERRUPT_RXLPITRCIS_SHIFT               (27U)
#define ENET_MMC_RX_INTERRUPT_RXLPITRCIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXLPITRCIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXLPITRCIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXLPIUSCIS_MASK                (0x4000000U)
#define ENET_MMC_RX_INTERRUPT_RXLPIUSCIS_SHIFT               (26U)
#define ENET_MMC_RX_INTERRUPT_RXLPIUSCIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXLPIUSCIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXLPIUSCIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXCTRLPIS_MASK                 (0x2000000U)
#define ENET_MMC_RX_INTERRUPT_RXCTRLPIS_SHIFT                (25U)
#define ENET_MMC_RX_INTERRUPT_RXCTRLPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXCTRLPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXCTRLPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXRCVERRPIS_MASK               (0x1000000U)
#define ENET_MMC_RX_INTERRUPT_RXRCVERRPIS_SHIFT              (24U)
#define ENET_MMC_RX_INTERRUPT_RXRCVERRPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXRCVERRPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXRCVERRPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXWDOGPIS_MASK                 (0x800000U)
#define ENET_MMC_RX_INTERRUPT_RXWDOGPIS_SHIFT                (23U)
#define ENET_MMC_RX_INTERRUPT_RXWDOGPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXWDOGPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXWDOGPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXVLANGBPIS_MASK               (0x400000U)
#define ENET_MMC_RX_INTERRUPT_RXVLANGBPIS_SHIFT              (22U)
#define ENET_MMC_RX_INTERRUPT_RXVLANGBPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXVLANGBPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXVLANGBPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXFOVPIS_MASK                  (0x200000U)
#define ENET_MMC_RX_INTERRUPT_RXFOVPIS_SHIFT                 (21U)
#define ENET_MMC_RX_INTERRUPT_RXFOVPIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXFOVPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXFOVPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXPAUSPIS_MASK                 (0x100000U)
#define ENET_MMC_RX_INTERRUPT_RXPAUSPIS_SHIFT                (20U)
#define ENET_MMC_RX_INTERRUPT_RXPAUSPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXPAUSPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXPAUSPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXORANGEPIS_MASK               (0x80000U)
#define ENET_MMC_RX_INTERRUPT_RXORANGEPIS_SHIFT              (19U)
#define ENET_MMC_RX_INTERRUPT_RXORANGEPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXORANGEPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXORANGEPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXLENERPIS_MASK                (0x40000U)
#define ENET_MMC_RX_INTERRUPT_RXLENERPIS_SHIFT               (18U)
#define ENET_MMC_RX_INTERRUPT_RXLENERPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXLENERPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXLENERPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXUCGPIS_MASK                  (0x20000U)
#define ENET_MMC_RX_INTERRUPT_RXUCGPIS_SHIFT                 (17U)
#define ENET_MMC_RX_INTERRUPT_RXUCGPIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXUCGPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXUCGPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RX1024TMAXOCTGBPIS_MASK           (0x10000U)
#define ENET_MMC_RX_INTERRUPT_RX1024TMAXOCTGBPIS_SHIFT          (16U)
#define ENET_MMC_RX_INTERRUPT_RX1024TMAXOCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RX1024TMAXOCTGBPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RX1024TMAXOCTGBPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RX512T1023OCTGBPIS_MASK           (0x8000U)
#define ENET_MMC_RX_INTERRUPT_RX512T1023OCTGBPIS_SHIFT          (15U)
#define ENET_MMC_RX_INTERRUPT_RX512T1023OCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RX512T1023OCTGBPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RX512T1023OCTGBPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RX256T511OCTGBPIS_MASK           (0x4000U)
#define ENET_MMC_RX_INTERRUPT_RX256T511OCTGBPIS_SHIFT          (14U)
#define ENET_MMC_RX_INTERRUPT_RX256T511OCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RX256T511OCTGBPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RX256T511OCTGBPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RX128T255OCTGBPIS_MASK           (0x2000U)
#define ENET_MMC_RX_INTERRUPT_RX128T255OCTGBPIS_SHIFT          (13U)
#define ENET_MMC_RX_INTERRUPT_RX128T255OCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RX128T255OCTGBPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RX128T255OCTGBPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RX65T127OCTGBPIS_MASK           (0x1000U)
#define ENET_MMC_RX_INTERRUPT_RX65T127OCTGBPIS_SHIFT          (12U)
#define ENET_MMC_RX_INTERRUPT_RX65T127OCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RX65T127OCTGBPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RX65T127OCTGBPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RX64OCTGBPIS_MASK              (0x800U)
#define ENET_MMC_RX_INTERRUPT_RX64OCTGBPIS_SHIFT             (11U)
#define ENET_MMC_RX_INTERRUPT_RX64OCTGBPIS(x)                (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RX64OCTGBPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RX64OCTGBPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXOSIZEGPIS_MASK               (0x400U)
#define ENET_MMC_RX_INTERRUPT_RXOSIZEGPIS_SHIFT              (10U)
#define ENET_MMC_RX_INTERRUPT_RXOSIZEGPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXOSIZEGPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXOSIZEGPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXUSIZEGPIS_MASK               (0x200U)
#define ENET_MMC_RX_INTERRUPT_RXUSIZEGPIS_SHIFT              (9U)
#define ENET_MMC_RX_INTERRUPT_RXUSIZEGPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXUSIZEGPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXUSIZEGPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXJABERPIS_MASK                (0x100U)
#define ENET_MMC_RX_INTERRUPT_RXJABERPIS_SHIFT               (8U)
#define ENET_MMC_RX_INTERRUPT_RXJABERPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXJABERPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXJABERPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXRUNTPIS_MASK                 (0x80U)
#define ENET_MMC_RX_INTERRUPT_RXRUNTPIS_SHIFT                (7U)
#define ENET_MMC_RX_INTERRUPT_RXRUNTPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXRUNTPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXRUNTPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXALGNERPIS_MASK               (0x40U)
#define ENET_MMC_RX_INTERRUPT_RXALGNERPIS_SHIFT              (6U)
#define ENET_MMC_RX_INTERRUPT_RXALGNERPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXALGNERPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXALGNERPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXCRCERPIS_MASK                (0x20U)
#define ENET_MMC_RX_INTERRUPT_RXCRCERPIS_SHIFT               (5U)
#define ENET_MMC_RX_INTERRUPT_RXCRCERPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXCRCERPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXCRCERPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXMCGPIS_MASK                  (0x10U)
#define ENET_MMC_RX_INTERRUPT_RXMCGPIS_SHIFT                 (4U)
#define ENET_MMC_RX_INTERRUPT_RXMCGPIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXMCGPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXMCGPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXBCGPIS_MASK                  (0x8U)
#define ENET_MMC_RX_INTERRUPT_RXBCGPIS_SHIFT                 (3U)
#define ENET_MMC_RX_INTERRUPT_RXBCGPIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXBCGPIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXBCGPIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXGOCTIS_MASK                  (0x4U)
#define ENET_MMC_RX_INTERRUPT_RXGOCTIS_SHIFT                 (2U)
#define ENET_MMC_RX_INTERRUPT_RXGOCTIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXGOCTIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXGOCTIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXGBOCTIS_MASK                 (0x2U)
#define ENET_MMC_RX_INTERRUPT_RXGBOCTIS_SHIFT                (1U)
#define ENET_MMC_RX_INTERRUPT_RXGBOCTIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXGBOCTIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXGBOCTIS_MASK)
#define ENET_MMC_RX_INTERRUPT_RXGBPKTIS_MASK                 (0x1U)
#define ENET_MMC_RX_INTERRUPT_RXGBPKTIS_SHIFT                (0U)
#define ENET_MMC_RX_INTERRUPT_RXGBPKTIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_RXGBPKTIS_SHIFT)) & ENET_MMC_RX_INTERRUPT_RXGBPKTIS_MASK)
/* ENET_MMC_TX_INTERRUPT Register */
#define ENET_MMC_TX_INTERRUPT_TXLPITRCIS_MASK                (0x8000000U)
#define ENET_MMC_TX_INTERRUPT_TXLPITRCIS_SHIFT               (27U)
#define ENET_MMC_TX_INTERRUPT_TXLPITRCIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXLPITRCIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXLPITRCIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXLPIUSCIS_MASK                (0x4000000U)
#define ENET_MMC_TX_INTERRUPT_TXLPIUSCIS_SHIFT               (26U)
#define ENET_MMC_TX_INTERRUPT_TXLPIUSCIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXLPIUSCIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXLPIUSCIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXOSIZEGPIS_MASK               (0x2000000U)
#define ENET_MMC_TX_INTERRUPT_TXOSIZEGPIS_SHIFT              (25U)
#define ENET_MMC_TX_INTERRUPT_TXOSIZEGPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXOSIZEGPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXOSIZEGPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXVLANGPIS_MASK                (0x1000000U)
#define ENET_MMC_TX_INTERRUPT_TXVLANGPIS_SHIFT               (24U)
#define ENET_MMC_TX_INTERRUPT_TXVLANGPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXVLANGPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXVLANGPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXPAUSPIS_MASK                 (0x800000U)
#define ENET_MMC_TX_INTERRUPT_TXPAUSPIS_SHIFT                (23U)
#define ENET_MMC_TX_INTERRUPT_TXPAUSPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXPAUSPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXPAUSPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXEXDEFPIS_MASK                (0x400000U)
#define ENET_MMC_TX_INTERRUPT_TXEXDEFPIS_SHIFT               (22U)
#define ENET_MMC_TX_INTERRUPT_TXEXDEFPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXEXDEFPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXEXDEFPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXGPKTIS_MASK                  (0x200000U)
#define ENET_MMC_TX_INTERRUPT_TXGPKTIS_SHIFT                 (21U)
#define ENET_MMC_TX_INTERRUPT_TXGPKTIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXGPKTIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXGPKTIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXGOCTIS_MASK                  (0x100000U)
#define ENET_MMC_TX_INTERRUPT_TXGOCTIS_SHIFT                 (20U)
#define ENET_MMC_TX_INTERRUPT_TXGOCTIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXGOCTIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXGOCTIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXCARERPIS_MASK                (0x80000U)
#define ENET_MMC_TX_INTERRUPT_TXCARERPIS_SHIFT               (19U)
#define ENET_MMC_TX_INTERRUPT_TXCARERPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXCARERPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXCARERPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXEXCOLPIS_MASK                (0x40000U)
#define ENET_MMC_TX_INTERRUPT_TXEXCOLPIS_SHIFT               (18U)
#define ENET_MMC_TX_INTERRUPT_TXEXCOLPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXEXCOLPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXEXCOLPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXLATCOLPIS_MASK               (0x20000U)
#define ENET_MMC_TX_INTERRUPT_TXLATCOLPIS_SHIFT              (17U)
#define ENET_MMC_TX_INTERRUPT_TXLATCOLPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXLATCOLPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXLATCOLPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXDEFPIS_MASK                  (0x10000U)
#define ENET_MMC_TX_INTERRUPT_TXDEFPIS_SHIFT                 (16U)
#define ENET_MMC_TX_INTERRUPT_TXDEFPIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXDEFPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXDEFPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXMCOLGPIS_MASK                (0x8000U)
#define ENET_MMC_TX_INTERRUPT_TXMCOLGPIS_SHIFT               (15U)
#define ENET_MMC_TX_INTERRUPT_TXMCOLGPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXMCOLGPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXMCOLGPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXSCOLGPIS_MASK                (0x4000U)
#define ENET_MMC_TX_INTERRUPT_TXSCOLGPIS_SHIFT               (14U)
#define ENET_MMC_TX_INTERRUPT_TXSCOLGPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXSCOLGPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXSCOLGPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXUFLOWERPIS_MASK              (0x2000U)
#define ENET_MMC_TX_INTERRUPT_TXUFLOWERPIS_SHIFT             (13U)
#define ENET_MMC_TX_INTERRUPT_TXUFLOWERPIS(x)                (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXUFLOWERPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXUFLOWERPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXBCGBPIS_MASK                 (0x1000U)
#define ENET_MMC_TX_INTERRUPT_TXBCGBPIS_SHIFT                (12U)
#define ENET_MMC_TX_INTERRUPT_TXBCGBPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXBCGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXBCGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXMCGBPIS_MASK                 (0x800U)
#define ENET_MMC_TX_INTERRUPT_TXMCGBPIS_SHIFT                (11U)
#define ENET_MMC_TX_INTERRUPT_TXMCGBPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXMCGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXMCGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXUCGBPIS_MASK                 (0x400U)
#define ENET_MMC_TX_INTERRUPT_TXUCGBPIS_SHIFT                (10U)
#define ENET_MMC_TX_INTERRUPT_TXUCGBPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXUCGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXUCGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TX1024TMAXOCTGBPIS_MASK           (0x200U)
#define ENET_MMC_TX_INTERRUPT_TX1024TMAXOCTGBPIS_SHIFT          (9U)
#define ENET_MMC_TX_INTERRUPT_TX1024TMAXOCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TX1024TMAXOCTGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TX1024TMAXOCTGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TX512T1023OCTGBPIS_MASK           (0x100U)
#define ENET_MMC_TX_INTERRUPT_TX512T1023OCTGBPIS_SHIFT          (8U)
#define ENET_MMC_TX_INTERRUPT_TX512T1023OCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TX512T1023OCTGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TX512T1023OCTGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TX256T511OCTGBPIS_MASK           (0x80U)
#define ENET_MMC_TX_INTERRUPT_TX256T511OCTGBPIS_SHIFT          (7U)
#define ENET_MMC_TX_INTERRUPT_TX256T511OCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TX256T511OCTGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TX256T511OCTGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TX128T255OCTGBPIS_MASK           (0x40U)
#define ENET_MMC_TX_INTERRUPT_TX128T255OCTGBPIS_SHIFT          (6U)
#define ENET_MMC_TX_INTERRUPT_TX128T255OCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TX128T255OCTGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TX128T255OCTGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TX65T127OCTGBPIS_MASK           (0x20U)
#define ENET_MMC_TX_INTERRUPT_TX65T127OCTGBPIS_SHIFT          (5U)
#define ENET_MMC_TX_INTERRUPT_TX65T127OCTGBPIS(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TX65T127OCTGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TX65T127OCTGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TX64OCTGBPIS_MASK              (0x10U)
#define ENET_MMC_TX_INTERRUPT_TX64OCTGBPIS_SHIFT             (4U)
#define ENET_MMC_TX_INTERRUPT_TX64OCTGBPIS(x)                (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TX64OCTGBPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TX64OCTGBPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXMCGPIS_MASK                  (0x8U)
#define ENET_MMC_TX_INTERRUPT_TXMCGPIS_SHIFT                 (3U)
#define ENET_MMC_TX_INTERRUPT_TXMCGPIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXMCGPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXMCGPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXBCGPIS_MASK                  (0x4U)
#define ENET_MMC_TX_INTERRUPT_TXBCGPIS_SHIFT                 (2U)
#define ENET_MMC_TX_INTERRUPT_TXBCGPIS(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXBCGPIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXBCGPIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXGBPKTIS_MASK                 (0x2U)
#define ENET_MMC_TX_INTERRUPT_TXGBPKTIS_SHIFT                (1U)
#define ENET_MMC_TX_INTERRUPT_TXGBPKTIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXGBPKTIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXGBPKTIS_MASK)
#define ENET_MMC_TX_INTERRUPT_TXGBOCTIS_MASK                 (0x1U)
#define ENET_MMC_TX_INTERRUPT_TXGBOCTIS_SHIFT                (0U)
#define ENET_MMC_TX_INTERRUPT_TXGBOCTIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_TXGBOCTIS_SHIFT)) & ENET_MMC_TX_INTERRUPT_TXGBOCTIS_MASK)
/* ENET_MMC_RX_INTERRUPT_MASK Register */
#define ENET_MMC_RX_INTERRUPT_MASK_RXLPITRCIM_MASK                (0x8000000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXLPITRCIM_SHIFT               (27U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXLPITRCIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXLPITRCIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXLPITRCIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXLPIUSCIM_MASK                (0x4000000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXLPIUSCIM_SHIFT               (26U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXLPIUSCIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXLPIUSCIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXLPIUSCIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXCTRLPIM_MASK                 (0x2000000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXCTRLPIM_SHIFT                (25U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXCTRLPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXCTRLPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXCTRLPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXRCVERRPIM_MASK               (0x1000000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXRCVERRPIM_SHIFT              (24U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXRCVERRPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXRCVERRPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXRCVERRPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXWDOGPIM_MASK                 (0x800000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXWDOGPIM_SHIFT                (23U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXWDOGPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXWDOGPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXWDOGPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXVLANGBPIM_MASK               (0x400000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXVLANGBPIM_SHIFT              (22U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXVLANGBPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXVLANGBPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXVLANGBPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXFOVPIM_MASK                  (0x200000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXFOVPIM_SHIFT                 (21U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXFOVPIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXFOVPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXFOVPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXPAUSPIM_MASK                 (0x100000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXPAUSPIM_SHIFT                (20U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXPAUSPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXPAUSPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXPAUSPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXORANGEPIM_MASK               (0x80000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXORANGEPIM_SHIFT              (19U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXORANGEPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXORANGEPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXORANGEPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXLENERPIM_MASK                (0x40000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXLENERPIM_SHIFT               (18U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXLENERPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXLENERPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXLENERPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXUCGPIM_MASK                  (0x20000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXUCGPIM_SHIFT                 (17U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXUCGPIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXUCGPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXUCGPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RX1024TMAXOCTGBPIM_MASK           (0x10000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX1024TMAXOCTGBPIM_SHIFT          (16U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX1024TMAXOCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RX1024TMAXOCTGBPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RX1024TMAXOCTGBPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RX512T1023OCTGBPIM_MASK           (0x8000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX512T1023OCTGBPIM_SHIFT          (15U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX512T1023OCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RX512T1023OCTGBPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RX512T1023OCTGBPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RX256T511OCTGBPIM_MASK           (0x4000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX256T511OCTGBPIM_SHIFT          (14U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX256T511OCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RX256T511OCTGBPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RX256T511OCTGBPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RX128T255OCTGBPIM_MASK           (0x2000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX128T255OCTGBPIM_SHIFT          (13U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX128T255OCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RX128T255OCTGBPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RX128T255OCTGBPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RX65T127OCTGBPIM_MASK           (0x1000U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX65T127OCTGBPIM_SHIFT          (12U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX65T127OCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RX65T127OCTGBPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RX65T127OCTGBPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RX64OCTGBPIM_MASK              (0x800U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX64OCTGBPIM_SHIFT             (11U)
#define ENET_MMC_RX_INTERRUPT_MASK_RX64OCTGBPIM(x)                (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RX64OCTGBPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RX64OCTGBPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXOSIZEGPIM_MASK               (0x400U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXOSIZEGPIM_SHIFT              (10U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXOSIZEGPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXOSIZEGPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXOSIZEGPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXUSIZEGPIM_MASK               (0x200U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXUSIZEGPIM_SHIFT              (9U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXUSIZEGPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXUSIZEGPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXUSIZEGPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXJABERPIM_MASK                (0x100U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXJABERPIM_SHIFT               (8U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXJABERPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXJABERPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXJABERPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXRUNTPIM_MASK                 (0x80U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXRUNTPIM_SHIFT                (7U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXRUNTPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXRUNTPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXRUNTPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXALGNERPIM_MASK               (0x40U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXALGNERPIM_SHIFT              (6U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXALGNERPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXALGNERPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXALGNERPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXCRCERPIM_MASK                (0x20U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXCRCERPIM_SHIFT               (5U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXCRCERPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXCRCERPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXCRCERPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXMCGPIM_MASK                  (0x10U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXMCGPIM_SHIFT                 (4U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXMCGPIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXMCGPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXMCGPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXBCGPIM_MASK                  (0x8U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXBCGPIM_SHIFT                 (3U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXBCGPIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXBCGPIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXBCGPIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGOCTIM_MASK                  (0x4U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGOCTIM_SHIFT                 (2U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGOCTIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXGOCTIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXGOCTIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGBOCTIM_MASK                 (0x2U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGBOCTIM_SHIFT                (1U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGBOCTIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXGBOCTIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXGBOCTIM_MASK)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGBPKTIM_MASK                 (0x1U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGBPKTIM_SHIFT                (0U)
#define ENET_MMC_RX_INTERRUPT_MASK_RXGBPKTIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_RX_INTERRUPT_MASK_RXGBPKTIM_SHIFT)) & ENET_MMC_RX_INTERRUPT_MASK_RXGBPKTIM_MASK)
/* ENET_MMC_TX_INTERRUPT_MASK Register */
#define ENET_MMC_TX_INTERRUPT_MASK_TXLPITRCIM_MASK                (0x8000000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXLPITRCIM_SHIFT               (27U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXLPITRCIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXLPITRCIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXLPITRCIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXLPIUSCIM_MASK                (0x4000000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXLPIUSCIM_SHIFT               (26U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXLPIUSCIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXLPIUSCIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXLPIUSCIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXOSIZEGPIM_MASK               (0x2000000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXOSIZEGPIM_SHIFT              (25U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXOSIZEGPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXOSIZEGPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXOSIZEGPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXVLANGPIM_MASK                (0x1000000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXVLANGPIM_SHIFT               (24U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXVLANGPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXVLANGPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXVLANGPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXPAUSPIM_MASK                 (0x800000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXPAUSPIM_SHIFT                (23U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXPAUSPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXPAUSPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXPAUSPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXEXDEFPIM_MASK                (0x400000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXEXDEFPIM_SHIFT               (22U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXEXDEFPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXEXDEFPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXEXDEFPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGPKTIM_MASK                  (0x200000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGPKTIM_SHIFT                 (21U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGPKTIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXGPKTIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXGPKTIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGOCTIM_MASK                  (0x100000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGOCTIM_SHIFT                 (20U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGOCTIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXGOCTIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXGOCTIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXCARERPIM_MASK                (0x80000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXCARERPIM_SHIFT               (19U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXCARERPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXCARERPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXCARERPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXEXCOLPIM_MASK                (0x40000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXEXCOLPIM_SHIFT               (18U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXEXCOLPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXEXCOLPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXEXCOLPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXLATCOLPIM_MASK               (0x20000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXLATCOLPIM_SHIFT              (17U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXLATCOLPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXLATCOLPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXLATCOLPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXDEFPIM_MASK                  (0x10000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXDEFPIM_SHIFT                 (16U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXDEFPIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXDEFPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXDEFPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCOLGPIM_MASK                (0x8000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCOLGPIM_SHIFT               (15U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCOLGPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXMCOLGPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXMCOLGPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXSCOLGPIM_MASK                (0x4000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXSCOLGPIM_SHIFT               (14U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXSCOLGPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXSCOLGPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXSCOLGPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXUFLOWERPIM_MASK              (0x2000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXUFLOWERPIM_SHIFT             (13U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXUFLOWERPIM(x)                (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXUFLOWERPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXUFLOWERPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXBCGBPIM_MASK                 (0x1000U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXBCGBPIM_SHIFT                (12U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXBCGBPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXBCGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXBCGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCGBPIM_MASK                 (0x800U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCGBPIM_SHIFT                (11U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCGBPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXMCGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXMCGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXUCGBPIM_MASK                 (0x400U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXUCGBPIM_SHIFT                (10U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXUCGBPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXUCGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXUCGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TX1024TMAXOCTGBPIM_MASK           (0x200U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX1024TMAXOCTGBPIM_SHIFT          (9U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX1024TMAXOCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TX1024TMAXOCTGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TX1024TMAXOCTGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TX512T1023OCTGBPIM_MASK           (0x100U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX512T1023OCTGBPIM_SHIFT          (8U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX512T1023OCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TX512T1023OCTGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TX512T1023OCTGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TX256T511OCTGBPIM_MASK           (0x80U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX256T511OCTGBPIM_SHIFT          (7U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX256T511OCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TX256T511OCTGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TX256T511OCTGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TX128T255OCTGBPIM_MASK           (0x40U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX128T255OCTGBPIM_SHIFT          (6U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX128T255OCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TX128T255OCTGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TX128T255OCTGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TX65T127OCTGBPIM_MASK           (0x20U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX65T127OCTGBPIM_SHIFT          (5U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX65T127OCTGBPIM(x)             (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TX65T127OCTGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TX65T127OCTGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TX64OCTGBPIM_MASK              (0x10U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX64OCTGBPIM_SHIFT             (4U)
#define ENET_MMC_TX_INTERRUPT_MASK_TX64OCTGBPIM(x)                (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TX64OCTGBPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TX64OCTGBPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCGPIM_MASK                  (0x8U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCGPIM_SHIFT                 (3U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXMCGPIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXMCGPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXMCGPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXBCGPIM_MASK                  (0x4U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXBCGPIM_SHIFT                 (2U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXBCGPIM(x)                    (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXBCGPIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXBCGPIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGBPKTIM_MASK                 (0x2U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGBPKTIM_SHIFT                (1U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGBPKTIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXGBPKTIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXGBPKTIM_MASK)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGBOCTIM_MASK                 (0x1U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGBOCTIM_SHIFT                (0U)
#define ENET_MMC_TX_INTERRUPT_MASK_TXGBOCTIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_TX_INTERRUPT_MASK_TXGBOCTIM_SHIFT)) & ENET_MMC_TX_INTERRUPT_MASK_TXGBOCTIM_MASK)
/* ENET_TX_OCTET_COUNT_GOOD_BAD Register */
#define ENET_TX_OCTET_COUNT_GOOD_BAD_TXOCTGB_MASK                   (0xFFFFFFFFU)
#define ENET_TX_OCTET_COUNT_GOOD_BAD_TXOCTGB_SHIFT                  (0U)
#define ENET_TX_OCTET_COUNT_GOOD_BAD_TXOCTGB(x)                     (((uint32)(((uint32)(x)) << ENET_TX_OCTET_COUNT_GOOD_BAD_TXOCTGB_SHIFT)) & ENET_TX_OCTET_COUNT_GOOD_BAD_TXOCTGB_MASK)
/* ENET_TX_PACKET_COUNT_GOOD_BAD Register */
#define ENET_TX_PACKET_COUNT_GOOD_BAD_TXPKTGB_MASK                   (0xFFFFFFFFU)
#define ENET_TX_PACKET_COUNT_GOOD_BAD_TXPKTGB_SHIFT                  (0U)
#define ENET_TX_PACKET_COUNT_GOOD_BAD_TXPKTGB(x)                     (((uint32)(((uint32)(x)) << ENET_TX_PACKET_COUNT_GOOD_BAD_TXPKTGB_SHIFT)) & ENET_TX_PACKET_COUNT_GOOD_BAD_TXPKTGB_MASK)
/* ENET_TX_BROADCAST_PACKETS_GOOD Register */
#define ENET_TX_BROADCAST_PACKETS_GOOD_TXBCASTG_MASK                  (0xFFFFU)
#define ENET_TX_BROADCAST_PACKETS_GOOD_TXBCASTG_SHIFT                 (0U)
#define ENET_TX_BROADCAST_PACKETS_GOOD_TXBCASTG(x)                    (((uint32)(((uint32)(x)) << ENET_TX_BROADCAST_PACKETS_GOOD_TXBCASTG_SHIFT)) & ENET_TX_BROADCAST_PACKETS_GOOD_TXBCASTG_MASK)
/* ENET_TX_MULTICAST_PACKETS_GOOD Register */
#define ENET_TX_MULTICAST_PACKETS_GOOD_TXMCASTG_MASK                  (0xFFFFU)
#define ENET_TX_MULTICAST_PACKETS_GOOD_TXMCASTG_SHIFT                 (0U)
#define ENET_TX_MULTICAST_PACKETS_GOOD_TXMCASTG(x)                    (((uint32)(((uint32)(x)) << ENET_TX_MULTICAST_PACKETS_GOOD_TXMCASTG_SHIFT)) & ENET_TX_MULTICAST_PACKETS_GOOD_TXMCASTG_MASK)
/* ENET_TX_64OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_TX_64OCTETS_PACKETS_GOOD_BAD_TX64OCTGB_MASK                 (0xFFFFU)
#define ENET_TX_64OCTETS_PACKETS_GOOD_BAD_TX64OCTGB_SHIFT                (0U)
#define ENET_TX_64OCTETS_PACKETS_GOOD_BAD_TX64OCTGB(x)                   (((uint32)(((uint32)(x)) << ENET_TX_64OCTETS_PACKETS_GOOD_BAD_TX64OCTGB_SHIFT)) & ENET_TX_64OCTETS_PACKETS_GOOD_BAD_TX64OCTGB_MASK)
/* ENET_TX_65TO127OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_TX_65TO127OCTETS_PACKETS_GOOD_BAD_TX65TO127OCTGB_MASK            (0xFFFFU)
#define ENET_TX_65TO127OCTETS_PACKETS_GOOD_BAD_TX65TO127OCTGB_SHIFT           (0U)
#define ENET_TX_65TO127OCTETS_PACKETS_GOOD_BAD_TX65TO127OCTGB(x)              (((uint32)(((uint32)(x)) << ENET_TX_65TO127OCTETS_PACKETS_GOOD_BAD_TX65TO127OCTGB_SHIFT)) & ENET_TX_65TO127OCTETS_PACKETS_GOOD_BAD_TX65TO127OCTGB_MASK)
/* ENET_TX_128TO255OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_TX_128TO255OCTETS_PACKETS_GOOD_BAD_TX128TO255OCTGB_MASK           (0xFFFFU)
#define ENET_TX_128TO255OCTETS_PACKETS_GOOD_BAD_TX128TO255OCTGB_SHIFT          (0U)
#define ENET_TX_128TO255OCTETS_PACKETS_GOOD_BAD_TX128TO255OCTGB(x)             (((uint32)(((uint32)(x)) << ENET_TX_128TO255OCTETS_PACKETS_GOOD_BAD_TX128TO255OCTGB_SHIFT)) & ENET_TX_128TO255OCTETS_PACKETS_GOOD_BAD_TX128TO255OCTGB_MASK)
/* ENET_TX_256TO511OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_TX_256TO511OCTETS_PACKETS_GOOD_BAD_TX256TO511OCTGB_MASK           (0xFFFFU)
#define ENET_TX_256TO511OCTETS_PACKETS_GOOD_BAD_TX256TO511OCTGB_SHIFT          (0U)
#define ENET_TX_256TO511OCTETS_PACKETS_GOOD_BAD_TX256TO511OCTGB(x)             (((uint32)(((uint32)(x)) << ENET_TX_256TO511OCTETS_PACKETS_GOOD_BAD_TX256TO511OCTGB_SHIFT)) & ENET_TX_256TO511OCTETS_PACKETS_GOOD_BAD_TX256TO511OCTGB_MASK)
/* ENET_TX_512TO1023OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_TX_512TO1023OCTETS_PACKETS_GOOD_BAD_TX512TO1023OCTGB_MASK           (0xFFFFU)
#define ENET_TX_512TO1023OCTETS_PACKETS_GOOD_BAD_TX512TO1023OCTGB_SHIFT          (0U)
#define ENET_TX_512TO1023OCTETS_PACKETS_GOOD_BAD_TX512TO1023OCTGB(x)             (((uint32)(((uint32)(x)) << ENET_TX_512TO1023OCTETS_PACKETS_GOOD_BAD_TX512TO1023OCTGB_SHIFT)) & ENET_TX_512TO1023OCTETS_PACKETS_GOOD_BAD_TX512TO1023OCTGB_MASK)
/* ENET_TX_1024TOMAXOCTETS_PACKETS_GOOD_BAD Register */
#define ENET_TX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_TX1024TOMAXOCTGB_MASK           (0xFFFFU)
#define ENET_TX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_TX1024TOMAXOCTGB_SHIFT          (0U)
#define ENET_TX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_TX1024TOMAXOCTGB(x)             (((uint32)(((uint32)(x)) << ENET_TX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_TX1024TOMAXOCTGB_SHIFT)) & ENET_TX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_TX1024TOMAXOCTGB_MASK)
/* ENET_TX_UNICAST_PACKETS_GOOD_BAD Register */
#define ENET_TX_UNICAST_PACKETS_GOOD_BAD_TXUCASTGB_MASK                 (0xFFFFU)
#define ENET_TX_UNICAST_PACKETS_GOOD_BAD_TXUCASTGB_SHIFT                (0U)
#define ENET_TX_UNICAST_PACKETS_GOOD_BAD_TXUCASTGB(x)                   (((uint32)(((uint32)(x)) << ENET_TX_UNICAST_PACKETS_GOOD_BAD_TXUCASTGB_SHIFT)) & ENET_TX_UNICAST_PACKETS_GOOD_BAD_TXUCASTGB_MASK)
/* ENET_TX_MULTICAST_PACKETS_GOOD_BAD Register */
#define ENET_TX_MULTICAST_PACKETS_GOOD_BAD_TXMCASTGB_MASK                 (0xFFFFU)
#define ENET_TX_MULTICAST_PACKETS_GOOD_BAD_TXMCASTGB_SHIFT                (0U)
#define ENET_TX_MULTICAST_PACKETS_GOOD_BAD_TXMCASTGB(x)                   (((uint32)(((uint32)(x)) << ENET_TX_MULTICAST_PACKETS_GOOD_BAD_TXMCASTGB_SHIFT)) & ENET_TX_MULTICAST_PACKETS_GOOD_BAD_TXMCASTGB_MASK)
/* ENET_TX_BROADCAST_PACKETS_GOOD_BAD Register */
#define ENET_TX_BROADCAST_PACKETS_GOOD_BAD_TXBCASTGB_MASK                 (0xFFFFU)
#define ENET_TX_BROADCAST_PACKETS_GOOD_BAD_TXBCASTGB_SHIFT                (0U)
#define ENET_TX_BROADCAST_PACKETS_GOOD_BAD_TXBCASTGB(x)                   (((uint32)(((uint32)(x)) << ENET_TX_BROADCAST_PACKETS_GOOD_BAD_TXBCASTGB_SHIFT)) & ENET_TX_BROADCAST_PACKETS_GOOD_BAD_TXBCASTGB_MASK)
/* ENET_TX_UNDERFLOW_ERROR_PACKETS Register */
#define ENET_TX_UNDERFLOW_ERROR_PACKETS_TXUNDRFLW_MASK                 (0xFFFFU)
#define ENET_TX_UNDERFLOW_ERROR_PACKETS_TXUNDRFLW_SHIFT                (0U)
#define ENET_TX_UNDERFLOW_ERROR_PACKETS_TXUNDRFLW(x)                   (((uint32)(((uint32)(x)) << ENET_TX_UNDERFLOW_ERROR_PACKETS_TXUNDRFLW_SHIFT)) & ENET_TX_UNDERFLOW_ERROR_PACKETS_TXUNDRFLW_MASK)
/* ENET_TX_SINGLE_COLLISION_GOOD_PACKETS Register */
#define ENET_TX_SINGLE_COLLISION_GOOD_PACKETS_TXSNGLCOLG_MASK                (0xFFFFU)
#define ENET_TX_SINGLE_COLLISION_GOOD_PACKETS_TXSNGLCOLG_SHIFT               (0U)
#define ENET_TX_SINGLE_COLLISION_GOOD_PACKETS_TXSNGLCOLG(x)                  (((uint32)(((uint32)(x)) << ENET_TX_SINGLE_COLLISION_GOOD_PACKETS_TXSNGLCOLG_SHIFT)) & ENET_TX_SINGLE_COLLISION_GOOD_PACKETS_TXSNGLCOLG_MASK)
/* ENET_TX_MULTIPLE_COLLISION_GOOD_PACKETS Register */
#define ENET_TX_MULTIPLE_COLLISION_GOOD_PACKETS_TXMULTCOLG_MASK                (0xFFFFU)
#define ENET_TX_MULTIPLE_COLLISION_GOOD_PACKETS_TXMULTCOLG_SHIFT               (0U)
#define ENET_TX_MULTIPLE_COLLISION_GOOD_PACKETS_TXMULTCOLG(x)                  (((uint32)(((uint32)(x)) << ENET_TX_MULTIPLE_COLLISION_GOOD_PACKETS_TXMULTCOLG_SHIFT)) & ENET_TX_MULTIPLE_COLLISION_GOOD_PACKETS_TXMULTCOLG_MASK)
/* ENET_TX_DEFERRED_PACKETS Register */
#define ENET_TX_DEFERRED_PACKETS_TXDEFRD_MASK                   (0xFFFFU)
#define ENET_TX_DEFERRED_PACKETS_TXDEFRD_SHIFT                  (0U)
#define ENET_TX_DEFERRED_PACKETS_TXDEFRD(x)                     (((uint32)(((uint32)(x)) << ENET_TX_DEFERRED_PACKETS_TXDEFRD_SHIFT)) & ENET_TX_DEFERRED_PACKETS_TXDEFRD_MASK)
/* ENET_TX_LATE_COLLISION_PACKETS Register */
#define ENET_TX_LATE_COLLISION_PACKETS_TXLATECOL_MASK                 (0xFFFFU)
#define ENET_TX_LATE_COLLISION_PACKETS_TXLATECOL_SHIFT                (0U)
#define ENET_TX_LATE_COLLISION_PACKETS_TXLATECOL(x)                   (((uint32)(((uint32)(x)) << ENET_TX_LATE_COLLISION_PACKETS_TXLATECOL_SHIFT)) & ENET_TX_LATE_COLLISION_PACKETS_TXLATECOL_MASK)
/* ENET_TX_EXCESSIVE_COLLISION_PACKETS Register */
#define ENET_TX_EXCESSIVE_COLLISION_PACKETS_TXEXSCOL_MASK                  (0xFFFFU)
#define ENET_TX_EXCESSIVE_COLLISION_PACKETS_TXEXSCOL_SHIFT                 (0U)
#define ENET_TX_EXCESSIVE_COLLISION_PACKETS_TXEXSCOL(x)                    (((uint32)(((uint32)(x)) << ENET_TX_EXCESSIVE_COLLISION_PACKETS_TXEXSCOL_SHIFT)) & ENET_TX_EXCESSIVE_COLLISION_PACKETS_TXEXSCOL_MASK)
/* ENET_TX_CARRIER_ERROR_PACKETS Register */
#define ENET_TX_CARRIER_ERROR_PACKETS_TXCARR_MASK                    (0xFFFFU)
#define ENET_TX_CARRIER_ERROR_PACKETS_TXCARR_SHIFT                   (0U)
#define ENET_TX_CARRIER_ERROR_PACKETS_TXCARR(x)                      (((uint32)(((uint32)(x)) << ENET_TX_CARRIER_ERROR_PACKETS_TXCARR_SHIFT)) & ENET_TX_CARRIER_ERROR_PACKETS_TXCARR_MASK)
/* ENET_TX_OCTET_COUNT_GOOD Register */
#define ENET_TX_OCTET_COUNT_GOOD_TXOCTG_MASK                    (0xFFFFU)
#define ENET_TX_OCTET_COUNT_GOOD_TXOCTG_SHIFT                   (0U)
#define ENET_TX_OCTET_COUNT_GOOD_TXOCTG(x)                      (((uint32)(((uint32)(x)) << ENET_TX_OCTET_COUNT_GOOD_TXOCTG_SHIFT)) & ENET_TX_OCTET_COUNT_GOOD_TXOCTG_MASK)
/* ENET_TX_PACKET_COUNT_GOOD Register */
#define ENET_TX_PACKET_COUNT_GOOD_TXPKTG_MASK                    (0xFFFFU)
#define ENET_TX_PACKET_COUNT_GOOD_TXPKTG_SHIFT                   (0U)
#define ENET_TX_PACKET_COUNT_GOOD_TXPKTG(x)                      (((uint32)(((uint32)(x)) << ENET_TX_PACKET_COUNT_GOOD_TXPKTG_SHIFT)) & ENET_TX_PACKET_COUNT_GOOD_TXPKTG_MASK)
/* ENET_TX_EXCESSIVE_DEFERRAL_ERROR Register */
#define ENET_TX_EXCESSIVE_DEFERRAL_ERROR_TXEXSDEF_MASK                  (0xFFFFU)
#define ENET_TX_EXCESSIVE_DEFERRAL_ERROR_TXEXSDEF_SHIFT                 (0U)
#define ENET_TX_EXCESSIVE_DEFERRAL_ERROR_TXEXSDEF(x)                    (((uint32)(((uint32)(x)) << ENET_TX_EXCESSIVE_DEFERRAL_ERROR_TXEXSDEF_SHIFT)) & ENET_TX_EXCESSIVE_DEFERRAL_ERROR_TXEXSDEF_MASK)
/* ENET_TX_PAUSE_PACKETS Register */
#define ENET_TX_PAUSE_PACKETS_TXPAUSE_MASK                   (0xFFFFU)
#define ENET_TX_PAUSE_PACKETS_TXPAUSE_SHIFT                  (0U)
#define ENET_TX_PAUSE_PACKETS_TXPAUSE(x)                     (((uint32)(((uint32)(x)) << ENET_TX_PAUSE_PACKETS_TXPAUSE_SHIFT)) & ENET_TX_PAUSE_PACKETS_TXPAUSE_MASK)
/* ENET_TX_VLAN_PACKETS_GOOD Register */
#define ENET_TX_VLAN_PACKETS_GOOD_TXVLANG_MASK                   (0xFFFFU)
#define ENET_TX_VLAN_PACKETS_GOOD_TXVLANG_SHIFT                  (0U)
#define ENET_TX_VLAN_PACKETS_GOOD_TXVLANG(x)                     (((uint32)(((uint32)(x)) << ENET_TX_VLAN_PACKETS_GOOD_TXVLANG_SHIFT)) & ENET_TX_VLAN_PACKETS_GOOD_TXVLANG_MASK)
/* ENET_TX_OSIZE_PACKETS_GOOD Register */
#define ENET_TX_OSIZE_PACKETS_GOOD_TXOSIZG_MASK                   (0xFFFFU)
#define ENET_TX_OSIZE_PACKETS_GOOD_TXOSIZG_SHIFT                  (0U)
#define ENET_TX_OSIZE_PACKETS_GOOD_TXOSIZG(x)                     (((uint32)(((uint32)(x)) << ENET_TX_OSIZE_PACKETS_GOOD_TXOSIZG_SHIFT)) & ENET_TX_OSIZE_PACKETS_GOOD_TXOSIZG_MASK)
/* ENET_RX_PACKETS_COUNT_GOOD_BAD Register */
#define ENET_RX_PACKETS_COUNT_GOOD_BAD_RXPKTGB_MASK                   (0xFFFFU)
#define ENET_RX_PACKETS_COUNT_GOOD_BAD_RXPKTGB_SHIFT                  (0U)
#define ENET_RX_PACKETS_COUNT_GOOD_BAD_RXPKTGB(x)                     (((uint32)(((uint32)(x)) << ENET_RX_PACKETS_COUNT_GOOD_BAD_RXPKTGB_SHIFT)) & ENET_RX_PACKETS_COUNT_GOOD_BAD_RXPKTGB_MASK)
/* ENET_RX_OCTET_COUNT_GOOD_BAD Register */
#define ENET_RX_OCTET_COUNT_GOOD_BAD_RXOCTGB_MASK                   (0xFFFFU)
#define ENET_RX_OCTET_COUNT_GOOD_BAD_RXOCTGB_SHIFT                  (0U)
#define ENET_RX_OCTET_COUNT_GOOD_BAD_RXOCTGB(x)                     (((uint32)(((uint32)(x)) << ENET_RX_OCTET_COUNT_GOOD_BAD_RXOCTGB_SHIFT)) & ENET_RX_OCTET_COUNT_GOOD_BAD_RXOCTGB_MASK)
/* ENET_RX_OCTET_COUNT_GOOD Register */
#define ENET_RX_OCTET_COUNT_GOOD_RXOCTG_MASK                    (0xFFFFU)
#define ENET_RX_OCTET_COUNT_GOOD_RXOCTG_SHIFT                   (0U)
#define ENET_RX_OCTET_COUNT_GOOD_RXOCTG(x)                      (((uint32)(((uint32)(x)) << ENET_RX_OCTET_COUNT_GOOD_RXOCTG_SHIFT)) & ENET_RX_OCTET_COUNT_GOOD_RXOCTG_MASK)
/* ENET_RX_BROADCAST_PACKETS_GOOD Register */
#define ENET_RX_BROADCAST_PACKETS_GOOD_RXBCASTG_MASK                  (0xFFFFU)
#define ENET_RX_BROADCAST_PACKETS_GOOD_RXBCASTG_SHIFT                 (0U)
#define ENET_RX_BROADCAST_PACKETS_GOOD_RXBCASTG(x)                    (((uint32)(((uint32)(x)) << ENET_RX_BROADCAST_PACKETS_GOOD_RXBCASTG_SHIFT)) & ENET_RX_BROADCAST_PACKETS_GOOD_RXBCASTG_MASK)
/* ENET_RX_MULTICAST_PACKETS_GOOD Register */
#define ENET_RX_MULTICAST_PACKETS_GOOD_RXMCASTG_MASK                  (0xFFFFU)
#define ENET_RX_MULTICAST_PACKETS_GOOD_RXMCASTG_SHIFT                 (0U)
#define ENET_RX_MULTICAST_PACKETS_GOOD_RXMCASTG(x)                    (((uint32)(((uint32)(x)) << ENET_RX_MULTICAST_PACKETS_GOOD_RXMCASTG_SHIFT)) & ENET_RX_MULTICAST_PACKETS_GOOD_RXMCASTG_MASK)
/* ENET_RX_CRC_ERROR_PACKETS Register */
#define ENET_RX_CRC_ERROR_PACKETS_RXCRCERR_MASK                  (0xFFFFU)
#define ENET_RX_CRC_ERROR_PACKETS_RXCRCERR_SHIFT                 (0U)
#define ENET_RX_CRC_ERROR_PACKETS_RXCRCERR(x)                    (((uint32)(((uint32)(x)) << ENET_RX_CRC_ERROR_PACKETS_RXCRCERR_SHIFT)) & ENET_RX_CRC_ERROR_PACKETS_RXCRCERR_MASK)
/* ENET_RX_ALIGNMENT_ERROR_PACKETS Register */
#define ENET_RX_ALIGNMENT_ERROR_PACKETS_RXALGNERR_MASK                 (0xFFFFU)
#define ENET_RX_ALIGNMENT_ERROR_PACKETS_RXALGNERR_SHIFT                (0U)
#define ENET_RX_ALIGNMENT_ERROR_PACKETS_RXALGNERR(x)                   (((uint32)(((uint32)(x)) << ENET_RX_ALIGNMENT_ERROR_PACKETS_RXALGNERR_SHIFT)) & ENET_RX_ALIGNMENT_ERROR_PACKETS_RXALGNERR_MASK)
/* ENET_RX_RUNT_ERROR_PACKETS Register */
#define ENET_RX_RUNT_ERROR_PACKETS_RXRUNTERR_MASK                 (0xFFFFU)
#define ENET_RX_RUNT_ERROR_PACKETS_RXRUNTERR_SHIFT                (0U)
#define ENET_RX_RUNT_ERROR_PACKETS_RXRUNTERR(x)                   (((uint32)(((uint32)(x)) << ENET_RX_RUNT_ERROR_PACKETS_RXRUNTERR_SHIFT)) & ENET_RX_RUNT_ERROR_PACKETS_RXRUNTERR_MASK)
/* ENET_RX_JABBER_ERROR_PACKETS Register */
#define ENET_RX_JABBER_ERROR_PACKETS_RXJABERR_MASK                  (0xFFFFU)
#define ENET_RX_JABBER_ERROR_PACKETS_RXJABERR_SHIFT                 (0U)
#define ENET_RX_JABBER_ERROR_PACKETS_RXJABERR(x)                    (((uint32)(((uint32)(x)) << ENET_RX_JABBER_ERROR_PACKETS_RXJABERR_SHIFT)) & ENET_RX_JABBER_ERROR_PACKETS_RXJABERR_MASK)
/* ENET_RX_UNDERSIZE_PACKETS_GOOD Register */
#define ENET_RX_UNDERSIZE_PACKETS_GOOD_RXUNDERSZG_MASK                (0xFFFFU)
#define ENET_RX_UNDERSIZE_PACKETS_GOOD_RXUNDERSZG_SHIFT               (0U)
#define ENET_RX_UNDERSIZE_PACKETS_GOOD_RXUNDERSZG(x)                  (((uint32)(((uint32)(x)) << ENET_RX_UNDERSIZE_PACKETS_GOOD_RXUNDERSZG_SHIFT)) & ENET_RX_UNDERSIZE_PACKETS_GOOD_RXUNDERSZG_MASK)
/* ENET_RX_OVERSIZE_PACKETS_GOOD Register */
#define ENET_RX_OVERSIZE_PACKETS_GOOD_RXOVERSZG_MASK                 (0xFFFFU)
#define ENET_RX_OVERSIZE_PACKETS_GOOD_RXOVERSZG_SHIFT                (0U)
#define ENET_RX_OVERSIZE_PACKETS_GOOD_RXOVERSZG(x)                   (((uint32)(((uint32)(x)) << ENET_RX_OVERSIZE_PACKETS_GOOD_RXOVERSZG_SHIFT)) & ENET_RX_OVERSIZE_PACKETS_GOOD_RXOVERSZG_MASK)
/* ENET_RX_64OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_RX_64OCTETS_PACKETS_GOOD_BAD_RX64OCTGB_MASK                 (0xFFFFU)
#define ENET_RX_64OCTETS_PACKETS_GOOD_BAD_RX64OCTGB_SHIFT                (0U)
#define ENET_RX_64OCTETS_PACKETS_GOOD_BAD_RX64OCTGB(x)                   (((uint32)(((uint32)(x)) << ENET_RX_64OCTETS_PACKETS_GOOD_BAD_RX64OCTGB_SHIFT)) & ENET_RX_64OCTETS_PACKETS_GOOD_BAD_RX64OCTGB_MASK)
/* ENET_RX_65TO127OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_RX_65TO127OCTETS_PACKETS_GOOD_BAD_RX65TO127OCTGB_MASK            (0xFFFFU)
#define ENET_RX_65TO127OCTETS_PACKETS_GOOD_BAD_RX65TO127OCTGB_SHIFT           (0U)
#define ENET_RX_65TO127OCTETS_PACKETS_GOOD_BAD_RX65TO127OCTGB(x)              (((uint32)(((uint32)(x)) << ENET_RX_65TO127OCTETS_PACKETS_GOOD_BAD_RX65TO127OCTGB_SHIFT)) & ENET_RX_65TO127OCTETS_PACKETS_GOOD_BAD_RX65TO127OCTGB_MASK)
/* ENET_RX_128TO255OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_RX_128TO255OCTETS_PACKETS_GOOD_BAD_RX128TO255OCTGB_MASK           (0xFFFFU)
#define ENET_RX_128TO255OCTETS_PACKETS_GOOD_BAD_RX128TO255OCTGB_SHIFT          (0U)
#define ENET_RX_128TO255OCTETS_PACKETS_GOOD_BAD_RX128TO255OCTGB(x)             (((uint32)(((uint32)(x)) << ENET_RX_128TO255OCTETS_PACKETS_GOOD_BAD_RX128TO255OCTGB_SHIFT)) & ENET_RX_128TO255OCTETS_PACKETS_GOOD_BAD_RX128TO255OCTGB_MASK)
/* ENET_RX_256TO511OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_RX_256TO511OCTETS_PACKETS_GOOD_BAD_RX256TO511OCTGB_MASK           (0xFFFFU)
#define ENET_RX_256TO511OCTETS_PACKETS_GOOD_BAD_RX256TO511OCTGB_SHIFT          (0U)
#define ENET_RX_256TO511OCTETS_PACKETS_GOOD_BAD_RX256TO511OCTGB(x)             (((uint32)(((uint32)(x)) << ENET_RX_256TO511OCTETS_PACKETS_GOOD_BAD_RX256TO511OCTGB_SHIFT)) & ENET_RX_256TO511OCTETS_PACKETS_GOOD_BAD_RX256TO511OCTGB_MASK)
/* ENET_RX_512TO1023OCTETS_PACKETS_GOOD_BAD Register */
#define ENET_RX_512TO1023OCTETS_PACKETS_GOOD_BAD_RX512TO1023OCTGB_MASK           (0xFFFFU)
#define ENET_RX_512TO1023OCTETS_PACKETS_GOOD_BAD_RX512TO1023OCTGB_SHIFT          (0U)
#define ENET_RX_512TO1023OCTETS_PACKETS_GOOD_BAD_RX512TO1023OCTGB(x)             (((uint32)(((uint32)(x)) << ENET_RX_512TO1023OCTETS_PACKETS_GOOD_BAD_RX512TO1023OCTGB_SHIFT)) & ENET_RX_512TO1023OCTETS_PACKETS_GOOD_BAD_RX512TO1023OCTGB_MASK)
/* ENET_RX_1024TOMAXOCTETS_PACKETS_GOOD_BAD Register */
#define ENET_RX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_RX1024TOMAXOCTGB_MASK           (0xFFFFU)
#define ENET_RX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_RX1024TOMAXOCTGB_SHIFT          (0U)
#define ENET_RX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_RX1024TOMAXOCTGB(x)             (((uint32)(((uint32)(x)) << ENET_RX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_RX1024TOMAXOCTGB_SHIFT)) & ENET_RX_1024TOMAXOCTETS_PACKETS_GOOD_BAD_RX1024TOMAXOCTGB_MASK)
/* ENET_RX_UNICAST_PACKETS_GOOD Register */
#define ENET_RX_UNICAST_PACKETS_GOOD_RXUCASTG_MASK                  (0xFFFFU)
#define ENET_RX_UNICAST_PACKETS_GOOD_RXUCASTG_SHIFT                 (0U)
#define ENET_RX_UNICAST_PACKETS_GOOD_RXUCASTG(x)                    (((uint32)(((uint32)(x)) << ENET_RX_UNICAST_PACKETS_GOOD_RXUCASTG_SHIFT)) & ENET_RX_UNICAST_PACKETS_GOOD_RXUCASTG_MASK)
/* ENET_RX_LENGTH_ERROR_PACKETS Register */
#define ENET_RX_LENGTH_ERROR_PACKETS_RXLENERR_MASK                  (0xFFFFU)
#define ENET_RX_LENGTH_ERROR_PACKETS_RXLENERR_SHIFT                 (0U)
#define ENET_RX_LENGTH_ERROR_PACKETS_RXLENERR(x)                    (((uint32)(((uint32)(x)) << ENET_RX_LENGTH_ERROR_PACKETS_RXLENERR_SHIFT)) & ENET_RX_LENGTH_ERROR_PACKETS_RXLENERR_MASK)
/* ENET_RX_OUT_OF_RANGE_TYPE_PACKETS Register */
#define ENET_RX_OUT_OF_RANGE_TYPE_PACKETS_RXOUTOFRNG_MASK                (0xFFFFU)
#define ENET_RX_OUT_OF_RANGE_TYPE_PACKETS_RXOUTOFRNG_SHIFT               (0U)
#define ENET_RX_OUT_OF_RANGE_TYPE_PACKETS_RXOUTOFRNG(x)                  (((uint32)(((uint32)(x)) << ENET_RX_OUT_OF_RANGE_TYPE_PACKETS_RXOUTOFRNG_SHIFT)) & ENET_RX_OUT_OF_RANGE_TYPE_PACKETS_RXOUTOFRNG_MASK)
/* ENET_RX_PAUSE_PACKETS Register */
#define ENET_RX_PAUSE_PACKETS_RXPAUSEPKT_MASK                (0xFFFFU)
#define ENET_RX_PAUSE_PACKETS_RXPAUSEPKT_SHIFT               (0U)
#define ENET_RX_PAUSE_PACKETS_RXPAUSEPKT(x)                  (((uint32)(((uint32)(x)) << ENET_RX_PAUSE_PACKETS_RXPAUSEPKT_SHIFT)) & ENET_RX_PAUSE_PACKETS_RXPAUSEPKT_MASK)
/* ENET_RX_FIFO_OVERFLOW_PACKETS Register */
#define ENET_RX_FIFO_OVERFLOW_PACKETS_RXFIFOOVFL_MASK                (0xFFFFU)
#define ENET_RX_FIFO_OVERFLOW_PACKETS_RXFIFOOVFL_SHIFT               (0U)
#define ENET_RX_FIFO_OVERFLOW_PACKETS_RXFIFOOVFL(x)                  (((uint32)(((uint32)(x)) << ENET_RX_FIFO_OVERFLOW_PACKETS_RXFIFOOVFL_SHIFT)) & ENET_RX_FIFO_OVERFLOW_PACKETS_RXFIFOOVFL_MASK)
/* ENET_RX_VLAN_PACKETS_GOOD_BAD Register */
#define ENET_RX_VLAN_PACKETS_GOOD_BAD_RXVLANPKTGB_MASK               (0xFFFFU)
#define ENET_RX_VLAN_PACKETS_GOOD_BAD_RXVLANPKTGB_SHIFT              (0U)
#define ENET_RX_VLAN_PACKETS_GOOD_BAD_RXVLANPKTGB(x)                 (((uint32)(((uint32)(x)) << ENET_RX_VLAN_PACKETS_GOOD_BAD_RXVLANPKTGB_SHIFT)) & ENET_RX_VLAN_PACKETS_GOOD_BAD_RXVLANPKTGB_MASK)
/* ENET_RX_WATCHDOG_ERROR_PACKETS Register */
#define ENET_RX_WATCHDOG_ERROR_PACKETS_RXWDGERR_MASK                  (0xFFFFU)
#define ENET_RX_WATCHDOG_ERROR_PACKETS_RXWDGERR_SHIFT                 (0U)
#define ENET_RX_WATCHDOG_ERROR_PACKETS_RXWDGERR(x)                    (((uint32)(((uint32)(x)) << ENET_RX_WATCHDOG_ERROR_PACKETS_RXWDGERR_SHIFT)) & ENET_RX_WATCHDOG_ERROR_PACKETS_RXWDGERR_MASK)
/* ENET_RX_RECEIVE_ERROR_PACKETS Register */
#define ENET_RX_RECEIVE_ERROR_PACKETS_RXRCVERR_MASK                  (0xFFFFU)
#define ENET_RX_RECEIVE_ERROR_PACKETS_RXRCVERR_SHIFT                 (0U)
#define ENET_RX_RECEIVE_ERROR_PACKETS_RXRCVERR(x)                    (((uint32)(((uint32)(x)) << ENET_RX_RECEIVE_ERROR_PACKETS_RXRCVERR_SHIFT)) & ENET_RX_RECEIVE_ERROR_PACKETS_RXRCVERR_MASK)
/* ENET_RX_CONTROL_PACKETS_GOOD Register */
#define ENET_RX_CONTROL_PACKETS_GOOD_RXCTRLG_MASK                   (0xFFFFU)
#define ENET_RX_CONTROL_PACKETS_GOOD_RXCTRLG_SHIFT                  (0U)
#define ENET_RX_CONTROL_PACKETS_GOOD_RXCTRLG(x)                     (((uint32)(((uint32)(x)) << ENET_RX_CONTROL_PACKETS_GOOD_RXCTRLG_SHIFT)) & ENET_RX_CONTROL_PACKETS_GOOD_RXCTRLG_MASK)
/* ENET_TX_LPI_USEC_CNTR Register */
#define ENET_TX_LPI_USEC_CNTR_TXLPIUSC_MASK                  (0xFFFFU)
#define ENET_TX_LPI_USEC_CNTR_TXLPIUSC_SHIFT                 (0U)
#define ENET_TX_LPI_USEC_CNTR_TXLPIUSC(x)                    (((uint32)(((uint32)(x)) << ENET_TX_LPI_USEC_CNTR_TXLPIUSC_SHIFT)) & ENET_TX_LPI_USEC_CNTR_TXLPIUSC_MASK)
/* ENET_TX_LPI_TRAN_CNTR Register */
#define ENET_TX_LPI_TRAN_CNTR_TXLPITRC_MASK                  (0xFFFFU)
#define ENET_TX_LPI_TRAN_CNTR_TXLPITRC_SHIFT                 (0U)
#define ENET_TX_LPI_TRAN_CNTR_TXLPITRC(x)                    (((uint32)(((uint32)(x)) << ENET_TX_LPI_TRAN_CNTR_TXLPITRC_SHIFT)) & ENET_TX_LPI_TRAN_CNTR_TXLPITRC_MASK)
/* ENET_RX_LPI_USEC_CNTR Register */
#define ENET_RX_LPI_USEC_CNTR_RXLPIUSC_MASK                  (0xFFFFU)
#define ENET_RX_LPI_USEC_CNTR_RXLPIUSC_SHIFT                 (0U)
#define ENET_RX_LPI_USEC_CNTR_RXLPIUSC(x)                    (((uint32)(((uint32)(x)) << ENET_RX_LPI_USEC_CNTR_RXLPIUSC_SHIFT)) & ENET_RX_LPI_USEC_CNTR_RXLPIUSC_MASK)
/* ENET_RX_LPI_TRAN_CNTR Register */
#define ENET_RX_LPI_TRAN_CNTR_RXLPITRC_MASK                  (0xFFFFU)
#define ENET_RX_LPI_TRAN_CNTR_RXLPITRC_SHIFT                 (0U)
#define ENET_RX_LPI_TRAN_CNTR_RXLPITRC(x)                    (((uint32)(((uint32)(x)) << ENET_RX_LPI_TRAN_CNTR_RXLPITRC_SHIFT)) & ENET_RX_LPI_TRAN_CNTR_RXLPITRC_MASK)
/* ENET_MMC_IPC_RX_INTERRUPT_MASK Register */
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPEROIM_MASK               (0x20000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPEROIM_SHIFT              (29U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPEROIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPEROIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPEROIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGOIM_MASK                (0x10000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGOIM_SHIFT               (28U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGOIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPEROIM_MASK                (0x8000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPEROIM_SHIFT               (27U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPEROIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPEROIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPEROIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGOIM_MASK                 (0x4000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGOIM_SHIFT                (26U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGOIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPEROIM_MASK                (0x2000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPEROIM_SHIFT               (25U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPEROIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPEROIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPEROIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGOIM_MASK                 (0x1000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGOIM_SHIFT                (24U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGOIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYOIM_MASK            (0x800000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYOIM_SHIFT           (23U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYOIM(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HEROIM_MASK              (0x400000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HEROIM_SHIFT             (22U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HEROIM(x)                (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HEROIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HEROIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GOIM_MASK                (0x200000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GOIM_SHIFT               (21U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GOIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLOIM_MASK            (0x100000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLOIM_SHIFT           (20U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLOIM(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGOIM_MASK             (0x80000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGOIM_SHIFT            (19U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGOIM(x)               (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYOIM_MASK            (0x40000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYOIM_SHIFT           (18U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYOIM(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HEROIM_MASK              (0x20000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HEROIM_SHIFT             (17U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HEROIM(x)                (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HEROIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HEROIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GOIM_MASK                (0x10000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GOIM_SHIFT               (16U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GOIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GOIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GOIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPERPIM_MASK               (0x2000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPERPIM_SHIFT              (13U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPERPIM(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPERPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPERPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGPIM_MASK                (0x1000U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGPIM_SHIFT               (12U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXICMPGPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPERPIM_MASK                (0x800U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPERPIM_SHIFT               (11U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPERPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPERPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPERPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGPIM_MASK                 (0x400U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGPIM_SHIFT                (10U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXTCPGPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPERPIM_MASK                (0x200U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPERPIM_SHIFT               (9U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPERPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPERPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPERPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGPIM_MASK                 (0x100U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGPIM_SHIFT                (8U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGPIM(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXUDPGPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYPIM_MASK            (0x80U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYPIM_SHIFT           (7U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYPIM(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6NOPAYPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HERPIM_MASK              (0x40U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HERPIM_SHIFT             (6U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HERPIM(x)                (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HERPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6HERPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GPIM_MASK                (0x20U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GPIM_SHIFT               (5U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV6GPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLPIM_MASK            (0x10U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLPIM_SHIFT           (4U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLPIM(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4UDSBLPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGPIM_MASK             (0x8U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGPIM_SHIFT            (3U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGPIM(x)               (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4FRAGPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYPIM_MASK            (0x4U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYPIM_SHIFT           (2U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYPIM(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4NOPAYPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HERPIM_MASK              (0x2U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HERPIM_SHIFT             (1U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HERPIM(x)                (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HERPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4HERPIM_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GPIM_MASK                (0x1U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GPIM_SHIFT               (0U)
#define ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GPIM(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GPIM_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_MASK_RXIPV4GPIM_MASK)
/* ENET_MMC_IPC_RX_INTERRUPT Register */
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPEROIS_MASK               (0x20000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPEROIS_SHIFT              (29U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPEROIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXICMPEROIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXICMPEROIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPGOIS_MASK                (0x10000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPGOIS_SHIFT               (28U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPGOIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXICMPGOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXICMPGOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPEROIS_MASK                (0x8000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPEROIS_SHIFT               (27U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPEROIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXTCPEROIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXTCPEROIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPGOIS_MASK                 (0x4000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPGOIS_SHIFT                (26U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPGOIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXTCPGOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXTCPGOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPEROIS_MASK                (0x2000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPEROIS_SHIFT               (25U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPEROIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXUDPEROIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXUDPEROIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPGOIS_MASK                 (0x1000000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPGOIS_SHIFT                (24U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPGOIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXUDPGOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXUDPGOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYOIS_MASK            (0x800000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYOIS_SHIFT           (23U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYOIS(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HEROIS_MASK              (0x400000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HEROIS_SHIFT             (22U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HEROIS(x)                (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HEROIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HEROIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GOIS_MASK                (0x200000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GOIS_SHIFT               (21U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GOIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLOIS_MASK            (0x100000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLOIS_SHIFT           (20U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLOIS(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGOIS_MASK             (0x80000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGOIS_SHIFT            (19U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGOIS(x)               (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYOIS_MASK            (0x40000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYOIS_SHIFT           (18U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYOIS(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HEROIS_MASK              (0x20000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HEROIS_SHIFT             (17U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HEROIS(x)                (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HEROIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HEROIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GOIS_MASK                (0x10000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GOIS_SHIFT               (16U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GOIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GOIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GOIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPERPIS_MASK               (0x2000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPERPIS_SHIFT              (13U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPERPIS(x)                 (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXICMPERPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXICMPERPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPGPIS_MASK                (0x1000U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPGPIS_SHIFT               (12U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXICMPGPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXICMPGPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXICMPGPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPERPIS_MASK                (0x800U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPERPIS_SHIFT               (11U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPERPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXTCPERPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXTCPERPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPGPIS_MASK                 (0x400U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPGPIS_SHIFT                (10U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXTCPGPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXTCPGPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXTCPGPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPERPIS_MASK                (0x200U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPERPIS_SHIFT               (9U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPERPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXUDPERPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXUDPERPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPGPIS_MASK                 (0x100U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPGPIS_SHIFT                (8U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXUDPGPIS(x)                   (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXUDPGPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXUDPGPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYPIS_MASK            (0x80U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYPIS_SHIFT           (7U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYPIS(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV6NOPAYPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HERPIS_MASK              (0x40U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HERPIS_SHIFT             (6U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HERPIS(x)                (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HERPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV6HERPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GPIS_MASK                (0x20U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GPIS_SHIFT               (5U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV6GPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLPIS_MASK            (0x10U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLPIS_SHIFT           (4U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLPIS(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4UDSBLPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGPIS_MASK             (0x8U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGPIS_SHIFT            (3U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGPIS(x)               (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4FRAGPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYPIS_MASK            (0x4U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYPIS_SHIFT           (2U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYPIS(x)              (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4NOPAYPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HERPIS_MASK              (0x2U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HERPIS_SHIFT             (1U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HERPIS(x)                (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HERPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4HERPIS_MASK)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GPIS_MASK                (0x1U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GPIS_SHIFT               (0U)
#define ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GPIS(x)                  (((uint32)(((uint32)(x)) << ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GPIS_SHIFT)) & ENET_MMC_IPC_RX_INTERRUPT_RXIPV4GPIS_MASK)
/* ENET_RXIPV4_GOOD_PACKETS Register */
#define ENET_RXIPV4_GOOD_PACKETS_RXIPV4GDPKT_MASK               (0xFFFFU)
#define ENET_RXIPV4_GOOD_PACKETS_RXIPV4GDPKT_SHIFT              (0U)
#define ENET_RXIPV4_GOOD_PACKETS_RXIPV4GDPKT(x)                 (((uint32)(((uint32)(x)) << ENET_RXIPV4_GOOD_PACKETS_RXIPV4GDPKT_SHIFT)) & ENET_RXIPV4_GOOD_PACKETS_RXIPV4GDPKT_MASK)
/* ENET_RXIPV4_HEADER_ERROR_PACKETS Register */
#define ENET_RXIPV4_HEADER_ERROR_PACKETS_RXIPV4HDRERRPKT_MASK           (0xFFFFU)
#define ENET_RXIPV4_HEADER_ERROR_PACKETS_RXIPV4HDRERRPKT_SHIFT          (0U)
#define ENET_RXIPV4_HEADER_ERROR_PACKETS_RXIPV4HDRERRPKT(x)             (((uint32)(((uint32)(x)) << ENET_RXIPV4_HEADER_ERROR_PACKETS_RXIPV4HDRERRPKT_SHIFT)) & ENET_RXIPV4_HEADER_ERROR_PACKETS_RXIPV4HDRERRPKT_MASK)
/* ENET_RXIPV4_NO_PAYLOAD_PACKETS Register */
#define ENET_RXIPV4_NO_PAYLOAD_PACKETS_RXIPV4NOPAYPKT_MASK            (0xFFFFU)
#define ENET_RXIPV4_NO_PAYLOAD_PACKETS_RXIPV4NOPAYPKT_SHIFT           (0U)
#define ENET_RXIPV4_NO_PAYLOAD_PACKETS_RXIPV4NOPAYPKT(x)              (((uint32)(((uint32)(x)) << ENET_RXIPV4_NO_PAYLOAD_PACKETS_RXIPV4NOPAYPKT_SHIFT)) & ENET_RXIPV4_NO_PAYLOAD_PACKETS_RXIPV4NOPAYPKT_MASK)
/* ENET_RXIPV4_FRAGMENTED_PACKETS Register */
#define ENET_RXIPV4_FRAGMENTED_PACKETS_RXIPV4FRAGPKT_MASK             (0xFFFFU)
#define ENET_RXIPV4_FRAGMENTED_PACKETS_RXIPV4FRAGPKT_SHIFT            (0U)
#define ENET_RXIPV4_FRAGMENTED_PACKETS_RXIPV4FRAGPKT(x)               (((uint32)(((uint32)(x)) << ENET_RXIPV4_FRAGMENTED_PACKETS_RXIPV4FRAGPKT_SHIFT)) & ENET_RXIPV4_FRAGMENTED_PACKETS_RXIPV4FRAGPKT_MASK)
/* ENET_RXIPV4_UDP_CHECKSUM_DISABLED_PACKETS Register */
#define ENET_RXIPV4_UDP_CHECKSUM_DISABLED_PACKETS_RXIPV4UDSBLPKT_MASK            (0xFFFFU)
#define ENET_RXIPV4_UDP_CHECKSUM_DISABLED_PACKETS_RXIPV4UDSBLPKT_SHIFT           (0U)
#define ENET_RXIPV4_UDP_CHECKSUM_DISABLED_PACKETS_RXIPV4UDSBLPKT(x)              (((uint32)(((uint32)(x)) << ENET_RXIPV4_UDP_CHECKSUM_DISABLED_PACKETS_RXIPV4UDSBLPKT_SHIFT)) & ENET_RXIPV4_UDP_CHECKSUM_DISABLED_PACKETS_RXIPV4UDSBLPKT_MASK)
/* ENET_RXIPV6_GOOD_PACKETS Register */
#define ENET_RXIPV6_GOOD_PACKETS_RXIPV6GDPKT_MASK               (0xFFFFU)
#define ENET_RXIPV6_GOOD_PACKETS_RXIPV6GDPKT_SHIFT              (0U)
#define ENET_RXIPV6_GOOD_PACKETS_RXIPV6GDPKT(x)                 (((uint32)(((uint32)(x)) << ENET_RXIPV6_GOOD_PACKETS_RXIPV6GDPKT_SHIFT)) & ENET_RXIPV6_GOOD_PACKETS_RXIPV6GDPKT_MASK)
/* ENET_RXIPV6_HEADER_ERROR_PACKETS Register */
#define ENET_RXIPV6_HEADER_ERROR_PACKETS_RXIPV6HDRERRPKT_MASK           (0xFFFFU)
#define ENET_RXIPV6_HEADER_ERROR_PACKETS_RXIPV6HDRERRPKT_SHIFT          (0U)
#define ENET_RXIPV6_HEADER_ERROR_PACKETS_RXIPV6HDRERRPKT(x)             (((uint32)(((uint32)(x)) << ENET_RXIPV6_HEADER_ERROR_PACKETS_RXIPV6HDRERRPKT_SHIFT)) & ENET_RXIPV6_HEADER_ERROR_PACKETS_RXIPV6HDRERRPKT_MASK)
/* ENET_RXIPV6_NO_PAYLOAD_PACKETS Register */
#define ENET_RXIPV6_NO_PAYLOAD_PACKETS_RXIPV6NOPAYPKT_MASK            (0xFFFFU)
#define ENET_RXIPV6_NO_PAYLOAD_PACKETS_RXIPV6NOPAYPKT_SHIFT           (0U)
#define ENET_RXIPV6_NO_PAYLOAD_PACKETS_RXIPV6NOPAYPKT(x)              (((uint32)(((uint32)(x)) << ENET_RXIPV6_NO_PAYLOAD_PACKETS_RXIPV6NOPAYPKT_SHIFT)) & ENET_RXIPV6_NO_PAYLOAD_PACKETS_RXIPV6NOPAYPKT_MASK)
/* ENET_RXUDP_GOOD_PACKETS Register */
#define ENET_RXUDP_GOOD_PACKETS_RXUDPGDPKT_MASK                (0xFFFFU)
#define ENET_RXUDP_GOOD_PACKETS_RXUDPGDPKT_SHIFT               (0U)
#define ENET_RXUDP_GOOD_PACKETS_RXUDPGDPKT(x)                  (((uint32)(((uint32)(x)) << ENET_RXUDP_GOOD_PACKETS_RXUDPGDPKT_SHIFT)) & ENET_RXUDP_GOOD_PACKETS_RXUDPGDPKT_MASK)
/* ENET_RXUDP_ERROR_PACKETS Register */
#define ENET_RXUDP_ERROR_PACKETS_RXUDPERRPKT_MASK               (0xFFFFU)
#define ENET_RXUDP_ERROR_PACKETS_RXUDPERRPKT_SHIFT              (0U)
#define ENET_RXUDP_ERROR_PACKETS_RXUDPERRPKT(x)                 (((uint32)(((uint32)(x)) << ENET_RXUDP_ERROR_PACKETS_RXUDPERRPKT_SHIFT)) & ENET_RXUDP_ERROR_PACKETS_RXUDPERRPKT_MASK)
/* ENET_RXTCP_GOOD_PACKETS Register */
#define ENET_RXTCP_GOOD_PACKETS_RXTCPGDPKT_MASK                (0xFFFFU)
#define ENET_RXTCP_GOOD_PACKETS_RXTCPGDPKT_SHIFT               (0U)
#define ENET_RXTCP_GOOD_PACKETS_RXTCPGDPKT(x)                  (((uint32)(((uint32)(x)) << ENET_RXTCP_GOOD_PACKETS_RXTCPGDPKT_SHIFT)) & ENET_RXTCP_GOOD_PACKETS_RXTCPGDPKT_MASK)
/* ENET_RXTCP_ERROR_PACKETS Register */
#define ENET_RXTCP_ERROR_PACKETS_RXTCPERRPKT_MASK               (0xFFFFU)
#define ENET_RXTCP_ERROR_PACKETS_RXTCPERRPKT_SHIFT              (0U)
#define ENET_RXTCP_ERROR_PACKETS_RXTCPERRPKT(x)                 (((uint32)(((uint32)(x)) << ENET_RXTCP_ERROR_PACKETS_RXTCPERRPKT_SHIFT)) & ENET_RXTCP_ERROR_PACKETS_RXTCPERRPKT_MASK)
/* ENET_RXICMP_GOOD_PACKETS Register */
#define ENET_RXICMP_GOOD_PACKETS_RXICMPGDPKT_MASK               (0xFFFFU)
#define ENET_RXICMP_GOOD_PACKETS_RXICMPGDPKT_SHIFT              (0U)
#define ENET_RXICMP_GOOD_PACKETS_RXICMPGDPKT(x)                 (((uint32)(((uint32)(x)) << ENET_RXICMP_GOOD_PACKETS_RXICMPGDPKT_SHIFT)) & ENET_RXICMP_GOOD_PACKETS_RXICMPGDPKT_MASK)
/* ENET_RXICMP_ERROR_PACKETS Register */
#define ENET_RXICMP_ERROR_PACKETS_RXICMPERRPKT_MASK              (0xFFFFU)
#define ENET_RXICMP_ERROR_PACKETS_RXICMPERRPKT_SHIFT             (0U)
#define ENET_RXICMP_ERROR_PACKETS_RXICMPERRPKT(x)                (((uint32)(((uint32)(x)) << ENET_RXICMP_ERROR_PACKETS_RXICMPERRPKT_SHIFT)) & ENET_RXICMP_ERROR_PACKETS_RXICMPERRPKT_MASK)
/* ENET_RXIPV4_GOOD_OCTETS Register */
#define ENET_RXIPV4_GOOD_OCTETS_RXIPV4GDOCT_MASK               (0xFFFFFFFFU)
#define ENET_RXIPV4_GOOD_OCTETS_RXIPV4GDOCT_SHIFT              (0U)
#define ENET_RXIPV4_GOOD_OCTETS_RXIPV4GDOCT(x)                 (((uint32)(((uint32)(x)) << ENET_RXIPV4_GOOD_OCTETS_RXIPV4GDOCT_SHIFT)) & ENET_RXIPV4_GOOD_OCTETS_RXIPV4GDOCT_MASK)
/* ENET_RXIPV4_HEADER_ERROR_OCTETS Register */
#define ENET_RXIPV4_HEADER_ERROR_OCTETS_RXIPV4HDRERROCT_MASK           (0xFFFFFFFFU)
#define ENET_RXIPV4_HEADER_ERROR_OCTETS_RXIPV4HDRERROCT_SHIFT          (0U)
#define ENET_RXIPV4_HEADER_ERROR_OCTETS_RXIPV4HDRERROCT(x)             (((uint32)(((uint32)(x)) << ENET_RXIPV4_HEADER_ERROR_OCTETS_RXIPV4HDRERROCT_SHIFT)) & ENET_RXIPV4_HEADER_ERROR_OCTETS_RXIPV4HDRERROCT_MASK)
/* ENET_RXIPV4_NOPAYLOAD_OCTETS Register */
#define ENET_RXIPV4_NOPAYLOAD_OCTETS_RXIPV4NOPAYOCT_MASK            (0xFFFFFFFFU)
#define ENET_RXIPV4_NOPAYLOAD_OCTETS_RXIPV4NOPAYOCT_SHIFT           (0U)
#define ENET_RXIPV4_NOPAYLOAD_OCTETS_RXIPV4NOPAYOCT(x)              (((uint32)(((uint32)(x)) << ENET_RXIPV4_NOPAYLOAD_OCTETS_RXIPV4NOPAYOCT_SHIFT)) & ENET_RXIPV4_NOPAYLOAD_OCTETS_RXIPV4NOPAYOCT_MASK)
/* ENET_RXIPV4_FRAGMENTED_OCTETS Register */
#define ENET_RXIPV4_FRAGMENTED_OCTETS_RXIPV4FRAGOCT_MASK             (0xFFFFFFFFU)
#define ENET_RXIPV4_FRAGMENTED_OCTETS_RXIPV4FRAGOCT_SHIFT            (0U)
#define ENET_RXIPV4_FRAGMENTED_OCTETS_RXIPV4FRAGOCT(x)               (((uint32)(((uint32)(x)) << ENET_RXIPV4_FRAGMENTED_OCTETS_RXIPV4FRAGOCT_SHIFT)) & ENET_RXIPV4_FRAGMENTED_OCTETS_RXIPV4FRAGOCT_MASK)
/* ENET_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS Register */
#define ENET_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_RXIPV4UDSBLOCT_MASK            (0xFFFFFFFFU)
#define ENET_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_RXIPV4UDSBLOCT_SHIFT           (0U)
#define ENET_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_RXIPV4UDSBLOCT(x)              (((uint32)(((uint32)(x)) << ENET_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_RXIPV4UDSBLOCT_SHIFT)) & ENET_RXIPV4_UDP_CHECKSUM_DISABLE_OCTETS_RXIPV4UDSBLOCT_MASK)
/* ENET_RXIPV6_GOOD_OCTETS Register */
#define ENET_RXIPV6_GOOD_OCTETS_RXIPV6GDOCT_MASK               (0xFFFFFFFFU)
#define ENET_RXIPV6_GOOD_OCTETS_RXIPV6GDOCT_SHIFT              (0U)
#define ENET_RXIPV6_GOOD_OCTETS_RXIPV6GDOCT(x)                 (((uint32)(((uint32)(x)) << ENET_RXIPV6_GOOD_OCTETS_RXIPV6GDOCT_SHIFT)) & ENET_RXIPV6_GOOD_OCTETS_RXIPV6GDOCT_MASK)
/* ENET_RXIPV6_HEADER_ERROR_OCTETS Register */
#define ENET_RXIPV6_HEADER_ERROR_OCTETS_RXIPV6HDRERROCT_MASK           (0xFFFFFFFFU)
#define ENET_RXIPV6_HEADER_ERROR_OCTETS_RXIPV6HDRERROCT_SHIFT          (0U)
#define ENET_RXIPV6_HEADER_ERROR_OCTETS_RXIPV6HDRERROCT(x)             (((uint32)(((uint32)(x)) << ENET_RXIPV6_HEADER_ERROR_OCTETS_RXIPV6HDRERROCT_SHIFT)) & ENET_RXIPV6_HEADER_ERROR_OCTETS_RXIPV6HDRERROCT_MASK)
/* ENET_RXIPV6_NO_PAYLOAD_OCTETS Register */
#define ENET_RXIPV6_NO_PAYLOAD_OCTETS_RXIPV6NOPAYOCT_MASK            (0xFFFFFFFFU)
#define ENET_RXIPV6_NO_PAYLOAD_OCTETS_RXIPV6NOPAYOCT_SHIFT           (0U)
#define ENET_RXIPV6_NO_PAYLOAD_OCTETS_RXIPV6NOPAYOCT(x)              (((uint32)(((uint32)(x)) << ENET_RXIPV6_NO_PAYLOAD_OCTETS_RXIPV6NOPAYOCT_SHIFT)) & ENET_RXIPV6_NO_PAYLOAD_OCTETS_RXIPV6NOPAYOCT_MASK)
/* ENET_RXUDP_GOOD_OCTETS Register */
#define ENET_RXUDP_GOOD_OCTETS_RXUDPGDOCT_MASK                (0xFFFFFFFFU)
#define ENET_RXUDP_GOOD_OCTETS_RXUDPGDOCT_SHIFT               (0U)
#define ENET_RXUDP_GOOD_OCTETS_RXUDPGDOCT(x)                  (((uint32)(((uint32)(x)) << ENET_RXUDP_GOOD_OCTETS_RXUDPGDOCT_SHIFT)) & ENET_RXUDP_GOOD_OCTETS_RXUDPGDOCT_MASK)
/* ENET_RXUDP_ERROR_OCTETS Register */
#define ENET_RXUDP_ERROR_OCTETS_RXUDPERROCT_MASK               (0xFFFFFFFFU)
#define ENET_RXUDP_ERROR_OCTETS_RXUDPERROCT_SHIFT              (0U)
#define ENET_RXUDP_ERROR_OCTETS_RXUDPERROCT(x)                 (((uint32)(((uint32)(x)) << ENET_RXUDP_ERROR_OCTETS_RXUDPERROCT_SHIFT)) & ENET_RXUDP_ERROR_OCTETS_RXUDPERROCT_MASK)
/* ENET_RXTCP_GOOD_OCTETS Register */
#define ENET_RXTCP_GOOD_OCTETS_RXTCPGDOCT_MASK                (0xFFFFFFFFU)
#define ENET_RXTCP_GOOD_OCTETS_RXTCPGDOCT_SHIFT               (0U)
#define ENET_RXTCP_GOOD_OCTETS_RXTCPGDOCT(x)                  (((uint32)(((uint32)(x)) << ENET_RXTCP_GOOD_OCTETS_RXTCPGDOCT_SHIFT)) & ENET_RXTCP_GOOD_OCTETS_RXTCPGDOCT_MASK)
/* ENET_RXTCP_ERROR_OCTETS Register */
#define ENET_RXTCP_ERROR_OCTETS_RXTCPERROCT_MASK               (0xFFFFFFFFU)
#define ENET_RXTCP_ERROR_OCTETS_RXTCPERROCT_SHIFT              (0U)
#define ENET_RXTCP_ERROR_OCTETS_RXTCPERROCT(x)                 (((uint32)(((uint32)(x)) << ENET_RXTCP_ERROR_OCTETS_RXTCPERROCT_SHIFT)) & ENET_RXTCP_ERROR_OCTETS_RXTCPERROCT_MASK)
/* ENET_RXICMP_GOOD_OCTETS Register */
#define ENET_RXICMP_GOOD_OCTETS_RXICMPGDOCT_MASK               (0xFFFFFFFFU)
#define ENET_RXICMP_GOOD_OCTETS_RXICMPGDOCT_SHIFT              (0U)
#define ENET_RXICMP_GOOD_OCTETS_RXICMPGDOCT(x)                 (((uint32)(((uint32)(x)) << ENET_RXICMP_GOOD_OCTETS_RXICMPGDOCT_SHIFT)) & ENET_RXICMP_GOOD_OCTETS_RXICMPGDOCT_MASK)
/* ENET_RXICMP_ERROR_OCTETS Register */
#define ENET_RXICMP_ERROR_OCTETS_RXICMPERROCT_MASK              (0xFFFFFFFFU)
#define ENET_RXICMP_ERROR_OCTETS_RXICMPERROCT_SHIFT             (0U)
#define ENET_RXICMP_ERROR_OCTETS_RXICMPERROCT(x)                (((uint32)(((uint32)(x)) << ENET_RXICMP_ERROR_OCTETS_RXICMPERROCT_SHIFT)) & ENET_RXICMP_ERROR_OCTETS_RXICMPERROCT_MASK)
/* ENET_MMC_FPE_TX_INTERRUPT Register */
#define ENET_MMC_FPE_TX_INTERRUPT_HRCIS_MASK                     (0x2U)
#define ENET_MMC_FPE_TX_INTERRUPT_HRCIS_SHIFT                    (1U)
#define ENET_MMC_FPE_TX_INTERRUPT_HRCIS(x)                       (((uint32)(((uint32)(x)) << ENET_MMC_FPE_TX_INTERRUPT_HRCIS_SHIFT)) & ENET_MMC_FPE_TX_INTERRUPT_HRCIS_MASK)
#define ENET_MMC_FPE_TX_INTERRUPT_FCIS_MASK                      (0x1U)
#define ENET_MMC_FPE_TX_INTERRUPT_FCIS_SHIFT                     (0U)
#define ENET_MMC_FPE_TX_INTERRUPT_FCIS(x)                        (((uint32)(((uint32)(x)) << ENET_MMC_FPE_TX_INTERRUPT_FCIS_SHIFT)) & ENET_MMC_FPE_TX_INTERRUPT_FCIS_MASK)
/* ENET_MMC_FPE_TX_INTERRUPT_MASK Register */
#define ENET_MMC_FPE_TX_INTERRUPT_MASK_HRCIM_MASK                     (0x2U)
#define ENET_MMC_FPE_TX_INTERRUPT_MASK_HRCIM_SHIFT                    (1U)
#define ENET_MMC_FPE_TX_INTERRUPT_MASK_HRCIM(x)                       (((uint32)(((uint32)(x)) << ENET_MMC_FPE_TX_INTERRUPT_MASK_HRCIM_SHIFT)) & ENET_MMC_FPE_TX_INTERRUPT_MASK_HRCIM_MASK)
#define ENET_MMC_FPE_TX_INTERRUPT_MASK_FCIM_MASK                      (0x1U)
#define ENET_MMC_FPE_TX_INTERRUPT_MASK_FCIM_SHIFT                     (0U)
#define ENET_MMC_FPE_TX_INTERRUPT_MASK_FCIM(x)                        (((uint32)(((uint32)(x)) << ENET_MMC_FPE_TX_INTERRUPT_MASK_FCIM_SHIFT)) & ENET_MMC_FPE_TX_INTERRUPT_MASK_FCIM_MASK)
/* ENET_MMC_TX_FPE_FRAGMENT_CNTR Register */
#define ENET_MMC_TX_FPE_FRAGMENT_CNTR_TXFFC_MASK                     (0xFFFFFFFFU)
#define ENET_MMC_TX_FPE_FRAGMENT_CNTR_TXFFC_SHIFT                    (0U)
#define ENET_MMC_TX_FPE_FRAGMENT_CNTR_TXFFC(x)                       (((uint32)(((uint32)(x)) << ENET_MMC_TX_FPE_FRAGMENT_CNTR_TXFFC_SHIFT)) & ENET_MMC_TX_FPE_FRAGMENT_CNTR_TXFFC_MASK)
/* ENET_MMC_TX_HOLD_REQ_CNTR Register */
#define ENET_MMC_TX_HOLD_REQ_CNTR_TXHRC_MASK                     (0xFFFFFFFFU)
#define ENET_MMC_TX_HOLD_REQ_CNTR_TXHRC_SHIFT                    (0U)
#define ENET_MMC_TX_HOLD_REQ_CNTR_TXHRC(x)                       (((uint32)(((uint32)(x)) << ENET_MMC_TX_HOLD_REQ_CNTR_TXHRC_SHIFT)) & ENET_MMC_TX_HOLD_REQ_CNTR_TXHRC_MASK)
/* ENET_MMC_FPE_RX_INTERRUPT Register */
#define ENET_MMC_FPE_RX_INTERRUPT_FCIS_MASK                      (0x8U)
#define ENET_MMC_FPE_RX_INTERRUPT_FCIS_SHIFT                     (3U)
#define ENET_MMC_FPE_RX_INTERRUPT_FCIS(x)                        (((uint32)(((uint32)(x)) << ENET_MMC_FPE_RX_INTERRUPT_FCIS_SHIFT)) & ENET_MMC_FPE_RX_INTERRUPT_FCIS_MASK)
#define ENET_MMC_FPE_RX_INTERRUPT_PAOCIS_MASK                    (0x4U)
#define ENET_MMC_FPE_RX_INTERRUPT_PAOCIS_SHIFT                   (2U)
#define ENET_MMC_FPE_RX_INTERRUPT_PAOCIS(x)                      (((uint32)(((uint32)(x)) << ENET_MMC_FPE_RX_INTERRUPT_PAOCIS_SHIFT)) & ENET_MMC_FPE_RX_INTERRUPT_PAOCIS_MASK)
#define ENET_MMC_FPE_RX_INTERRUPT_PSECIS_MASK                    (0x2U)
#define ENET_MMC_FPE_RX_INTERRUPT_PSECIS_SHIFT                   (1U)
#define ENET_MMC_FPE_RX_INTERRUPT_PSECIS(x)                      (((uint32)(((uint32)(x)) << ENET_MMC_FPE_RX_INTERRUPT_PSECIS_SHIFT)) & ENET_MMC_FPE_RX_INTERRUPT_PSECIS_MASK)
#define ENET_MMC_FPE_RX_INTERRUPT_PAECIS_MASK                    (0x1U)
#define ENET_MMC_FPE_RX_INTERRUPT_PAECIS_SHIFT                   (0U)
#define ENET_MMC_FPE_RX_INTERRUPT_PAECIS(x)                      (((uint32)(((uint32)(x)) << ENET_MMC_FPE_RX_INTERRUPT_PAECIS_SHIFT)) & ENET_MMC_FPE_RX_INTERRUPT_PAECIS_MASK)
/* ENET_MMC_FPE_RX_INTERRUPT_MASK Register */
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_FCIM_MASK                      (0x8U)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_FCIM_SHIFT                     (3U)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_FCIM(x)                        (((uint32)(((uint32)(x)) << ENET_MMC_FPE_RX_INTERRUPT_MASK_FCIM_SHIFT)) & ENET_MMC_FPE_RX_INTERRUPT_MASK_FCIM_MASK)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PAOCIM_MASK                    (0x4U)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PAOCIM_SHIFT                   (2U)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PAOCIM(x)                      (((uint32)(((uint32)(x)) << ENET_MMC_FPE_RX_INTERRUPT_MASK_PAOCIM_SHIFT)) & ENET_MMC_FPE_RX_INTERRUPT_MASK_PAOCIM_MASK)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PSECIM_MASK                    (0x2U)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PSECIM_SHIFT                   (1U)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PSECIM(x)                      (((uint32)(((uint32)(x)) << ENET_MMC_FPE_RX_INTERRUPT_MASK_PSECIM_SHIFT)) & ENET_MMC_FPE_RX_INTERRUPT_MASK_PSECIM_MASK)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PAECIM_MASK                    (0x1U)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PAECIM_SHIFT                   (0U)
#define ENET_MMC_FPE_RX_INTERRUPT_MASK_PAECIM(x)                      (((uint32)(((uint32)(x)) << ENET_MMC_FPE_RX_INTERRUPT_MASK_PAECIM_SHIFT)) & ENET_MMC_FPE_RX_INTERRUPT_MASK_PAECIM_MASK)
/* ENET_MMC_RX_PACKET_ASSEMBLY_ERR_CNTR Register */
#define ENET_MMC_RX_PACKET_ASSEMBLY_ERR_CNTR_PAEC_MASK                      (0xFFFFFFFFU)
#define ENET_MMC_RX_PACKET_ASSEMBLY_ERR_CNTR_PAEC_SHIFT                     (0U)
#define ENET_MMC_RX_PACKET_ASSEMBLY_ERR_CNTR_PAEC(x)                        (((uint32)(((uint32)(x)) << ENET_MMC_RX_PACKET_ASSEMBLY_ERR_CNTR_PAEC_SHIFT)) & ENET_MMC_RX_PACKET_ASSEMBLY_ERR_CNTR_PAEC_MASK)
/* ENET_MMC_RX_PACKET_SMD_ERR_CNTR Register */
#define ENET_MMC_RX_PACKET_SMD_ERR_CNTR_PSEC_MASK                      (0xFFFFFFFFU)
#define ENET_MMC_RX_PACKET_SMD_ERR_CNTR_PSEC_SHIFT                     (0U)
#define ENET_MMC_RX_PACKET_SMD_ERR_CNTR_PSEC(x)                        (((uint32)(((uint32)(x)) << ENET_MMC_RX_PACKET_SMD_ERR_CNTR_PSEC_SHIFT)) & ENET_MMC_RX_PACKET_SMD_ERR_CNTR_PSEC_MASK)
/* ENET_MMC_RX_PACKET_ASSEMBLY_OK_CNTR Register */
#define ENET_MMC_RX_PACKET_ASSEMBLY_OK_CNTR_PAOC_MASK                      (0xFFFFFFFFU)
#define ENET_MMC_RX_PACKET_ASSEMBLY_OK_CNTR_PAOC_SHIFT                     (0U)
#define ENET_MMC_RX_PACKET_ASSEMBLY_OK_CNTR_PAOC(x)                        (((uint32)(((uint32)(x)) << ENET_MMC_RX_PACKET_ASSEMBLY_OK_CNTR_PAOC_SHIFT)) & ENET_MMC_RX_PACKET_ASSEMBLY_OK_CNTR_PAOC_MASK)
/* ENET_MMC_RX_FPE_FRAGMENT_CNTR Register */
#define ENET_MMC_RX_FPE_FRAGMENT_CNTR_FFC_MASK                       (0xFFFFFFFFU)
#define ENET_MMC_RX_FPE_FRAGMENT_CNTR_FFC_SHIFT                      (0U)
#define ENET_MMC_RX_FPE_FRAGMENT_CNTR_FFC(x)                         (((uint32)(((uint32)(x)) << ENET_MMC_RX_FPE_FRAGMENT_CNTR_FFC_SHIFT)) & ENET_MMC_RX_FPE_FRAGMENT_CNTR_FFC_MASK)
/* ENET_MAC_EQOS_L3_L4_FILTER_CONTROL Register */
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHEN0_MASK                           (0x10000000U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHEN0_SHIFT                          (28U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHEN0(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHEN0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHEN0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHN0_MASK                            (0x3000000U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHN0_SHIFT                           (24U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHN0(x)                              (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHN0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_DMCHN0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPIM0_MASK                           (0x200000U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPIM0_SHIFT                          (21U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPIM0(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPIM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPIM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPM0_MASK                            (0x100000U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPM0_SHIFT                           (20U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPM0(x)                              (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4DPM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPIM0_MASK                           (0x80000U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPIM0_SHIFT                          (19U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPIM0(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPIM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPIM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPM0_MASK                            (0x40000U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPM0_SHIFT                           (18U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPM0(x)                              (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4SPM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4PEN0_MASK                            (0x10000U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4PEN0_SHIFT                           (16U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4PEN0(x)                              (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4PEN0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L4PEN0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HDBM0_MASK                           (0xF800U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HDBM0_SHIFT                          (11U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HDBM0(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HDBM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HDBM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HSBM0_MASK                           (0x7C0U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HSBM0_SHIFT                          (6U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HSBM0(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HSBM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3HSBM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAIM0_MASK                           (0x20U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAIM0_SHIFT                          (5U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAIM0(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAIM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAIM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAM0_MASK                            (0x10U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAM0_SHIFT                           (4U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAM0(x)                              (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3DAM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAIM0_MASK                           (0x8U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAIM0_SHIFT                          (3U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAIM0(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAIM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAIM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAM0_MASK                            (0x4U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAM0_SHIFT                           (2U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAM0(x)                              (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAM0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3SAM0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3PEN0_MASK                            (0x1U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3PEN0_SHIFT                           (0U)
#define ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3PEN0(x)                              (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3PEN0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_CONTROL_L3PEN0_MASK)
/* ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS Register */
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4DP0_MASK                      (0xFFFF0000U)
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4DP0_SHIFT                     (16U)
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4DP0(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4DP0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4DP0_MASK)
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4SP0_MASK                      (0xFFFFU)
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4SP0_SHIFT                     (0U)
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4SP0(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4SP0_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_LAYER4_ADDRESS_L4SP0_MASK)
/* ENET_MAC_EQOS_L3_L4_FILTER_LAYER3_ADDR Register */
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER3_ADDR_L3A_MASK                           (0xFFFFFFFFU)
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER3_ADDR_L3A_SHIFT                          (0U)
#define ENET_MAC_EQOS_L3_L4_FILTER_LAYER3_ADDR_L3A(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_EQOS_L3_L4_FILTER_LAYER3_ADDR_L3A_SHIFT)) & ENET_MAC_EQOS_L3_L4_FILTER_LAYER3_ADDR_L3A_MASK)

/* ENET_MAC_INDIR_ACCESS_CTRL Register */
#define ENET_MAC_INDIR_ACCESS_CTRL_MSEL_MASK                      (0xF0000U)
#define ENET_MAC_INDIR_ACCESS_CTRL_MSEL_SHIFT                     (16U)
#define ENET_MAC_INDIR_ACCESS_CTRL_MSEL(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_INDIR_ACCESS_CTRL_MSEL_SHIFT)) & ENET_MAC_INDIR_ACCESS_CTRL_MSEL_MASK)
#define ENET_MAC_INDIR_ACCESS_CTRL_AOFF_MASK                      (0xFF00U)
#define ENET_MAC_INDIR_ACCESS_CTRL_AOFF_SHIFT                     (8U)
#define ENET_MAC_INDIR_ACCESS_CTRL_AOFF(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_INDIR_ACCESS_CTRL_AOFF_SHIFT)) & ENET_MAC_INDIR_ACCESS_CTRL_AOFF_MASK)
#define ENET_MAC_INDIR_ACCESS_CTRL_AUTO_MASK                      (0x20U)
#define ENET_MAC_INDIR_ACCESS_CTRL_AUTO_SHIFT                     (5U)
#define ENET_MAC_INDIR_ACCESS_CTRL_AUTO(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_INDIR_ACCESS_CTRL_AUTO_SHIFT)) & ENET_MAC_INDIR_ACCESS_CTRL_AUTO_MASK)
#define ENET_MAC_INDIR_ACCESS_CTRL_COM_MASK                       (0x2U)
#define ENET_MAC_INDIR_ACCESS_CTRL_COM_SHIFT                      (1U)
#define ENET_MAC_INDIR_ACCESS_CTRL_COM(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_INDIR_ACCESS_CTRL_COM_SHIFT)) & ENET_MAC_INDIR_ACCESS_CTRL_COM_MASK)
#define ENET_MAC_INDIR_ACCESS_CTRL_OB_MASK                        (0x1U)
#define ENET_MAC_INDIR_ACCESS_CTRL_OB_SHIFT                       (0U)
#define ENET_MAC_INDIR_ACCESS_CTRL_OB(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_INDIR_ACCESS_CTRL_OB_SHIFT)) & ENET_MAC_INDIR_ACCESS_CTRL_OB_MASK)
/* ENET_MAC_TMRQ_REGS Register */
#define ENET_MAC_TMRQ_REGS_PFEX_MASK                        (0x100000U)
#define ENET_MAC_TMRQ_REGS_PFEX_SHIFT                       (20U)
#define ENET_MAC_TMRQ_REGS_PFEX(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_TMRQ_REGS_PFEX_SHIFT)) & ENET_MAC_TMRQ_REGS_PFEX_MASK)
#define ENET_MAC_TMRQ_REGS_TMRQ_MASK                        (0x70000U)
#define ENET_MAC_TMRQ_REGS_TMRQ_SHIFT                       (16U)
#define ENET_MAC_TMRQ_REGS_TMRQ(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_TMRQ_REGS_TMRQ_SHIFT)) & ENET_MAC_TMRQ_REGS_TMRQ_MASK)
#define ENET_MAC_TMRQ_REGS_TYP_MASK                         (0xFFFFU)
#define ENET_MAC_TMRQ_REGS_TYP_SHIFT                        (0U)
#define ENET_MAC_TMRQ_REGS_TYP(x)                           (((uint32)(((uint32)(x)) << ENET_MAC_TMRQ_REGS_TYP_SHIFT)) & ENET_MAC_TMRQ_REGS_TYP_MASK)
/* ENET_MAC_TIMESTAMP_CONTROL Register */
#define ENET_MAC_TIMESTAMP_CONTROL_AV8021ASMEN_MASK               (0x10000000U)
#define ENET_MAC_TIMESTAMP_CONTROL_AV8021ASMEN_SHIFT              (28U)
#define ENET_MAC_TIMESTAMP_CONTROL_AV8021ASMEN(x)                 (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_AV8021ASMEN_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_AV8021ASMEN_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TXTSSTSM_MASK                  (0x1000000U)
#define ENET_MAC_TIMESTAMP_CONTROL_TXTSSTSM_SHIFT                 (24U)
#define ENET_MAC_TIMESTAMP_CONTROL_TXTSSTSM(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TXTSSTSM_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TXTSSTSM_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_ESTI_MASK                      (0x100000U)
#define ENET_MAC_TIMESTAMP_CONTROL_ESTI_SHIFT                     (20U)
#define ENET_MAC_TIMESTAMP_CONTROL_ESTI(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_ESTI_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_ESTI_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_CSC_MASK                       (0x80000U)
#define ENET_MAC_TIMESTAMP_CONTROL_CSC_SHIFT                      (19U)
#define ENET_MAC_TIMESTAMP_CONTROL_CSC(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_CSC_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_CSC_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENMACADDR_MASK               (0x40000U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENMACADDR_SHIFT              (18U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENMACADDR(x)                 (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSENMACADDR_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSENMACADDR_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_SNAPTYPSEL_MASK                (0x30000U)
#define ENET_MAC_TIMESTAMP_CONTROL_SNAPTYPSEL_SHIFT               (16U)
#define ENET_MAC_TIMESTAMP_CONTROL_SNAPTYPSEL(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_SNAPTYPSEL_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_SNAPTYPSEL_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSMSTRENA_MASK                 (0x8000U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSMSTRENA_SHIFT                (15U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSMSTRENA(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSMSTRENA_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSMSTRENA_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSEVNTENA_MASK                 (0x4000U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSEVNTENA_SHIFT                (14U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSEVNTENA(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSEVNTENA_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSEVNTENA_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPV4ENA_MASK                 (0x2000U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPV4ENA_SHIFT                (13U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPV4ENA(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSIPV4ENA_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSIPV4ENA_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPV6ENA_MASK                 (0x1000U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPV6ENA_SHIFT                (12U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPV6ENA(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSIPV6ENA_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSIPV6ENA_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPENA_MASK                   (0x800U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPENA_SHIFT                  (11U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSIPENA(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSIPENA_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSIPENA_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSVER2ENA_MASK                 (0x400U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSVER2ENA_SHIFT                (10U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSVER2ENA(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSVER2ENA_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSVER2ENA_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSCTRLSSR_MASK                 (0x200U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSCTRLSSR_SHIFT                (9U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSCTRLSSR(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSCTRLSSR_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSCTRLSSR_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENALL_MASK                   (0x100U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENALL_SHIFT                  (8U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENALL(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSENALL_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSENALL_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_PTGE_MASK                      (0x40U)
#define ENET_MAC_TIMESTAMP_CONTROL_PTGE_SHIFT                     (6U)
#define ENET_MAC_TIMESTAMP_CONTROL_PTGE(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_PTGE_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_PTGE_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSADDREG_MASK                  (0x20U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSADDREG_SHIFT                 (5U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSADDREG(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSADDREG_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSADDREG_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSTRIG_MASK                    (0x10U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSTRIG_SHIFT                   (4U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSTRIG(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSTRIG_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSTRIG_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSUPDT_MASK                    (0x8U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSUPDT_SHIFT                   (3U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSUPDT(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSUPDT_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSUPDT_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSINIT_MASK                    (0x4U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSINIT_SHIFT                   (2U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSINIT(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSINIT_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSINIT_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSCFUPDT_MASK                  (0x2U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSCFUPDT_SHIFT                 (1U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSCFUPDT(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSCFUPDT_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSCFUPDT_MASK)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENA_MASK                     (0x1U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENA_SHIFT                    (0U)
#define ENET_MAC_TIMESTAMP_CONTROL_TSENA(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_CONTROL_TSENA_SHIFT)) & ENET_MAC_TIMESTAMP_CONTROL_TSENA_MASK)
/* ENET_MAC_SUB_SECOND_INCREMENT Register */
#define ENET_MAC_SUB_SECOND_INCREMENT_SSINC_MASK                     (0xFF0000U)
#define ENET_MAC_SUB_SECOND_INCREMENT_SSINC_SHIFT                    (16U)
#define ENET_MAC_SUB_SECOND_INCREMENT_SSINC(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_SUB_SECOND_INCREMENT_SSINC_SHIFT)) & ENET_MAC_SUB_SECOND_INCREMENT_SSINC_MASK)
#define ENET_MAC_SUB_SECOND_INCREMENT_SNSINC_MASK                    (0xFF00U)
#define ENET_MAC_SUB_SECOND_INCREMENT_SNSINC_SHIFT                   (8U)
#define ENET_MAC_SUB_SECOND_INCREMENT_SNSINC(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_SUB_SECOND_INCREMENT_SNSINC_SHIFT)) & ENET_MAC_SUB_SECOND_INCREMENT_SNSINC_MASK)
/* ENET_MAC_SYSTEM_TIME_SECONDS Register */
#define ENET_MAC_SYSTEM_TIME_SECONDS_TSS_MASK                       (0xFFFFFFFFU)
#define ENET_MAC_SYSTEM_TIME_SECONDS_TSS_SHIFT                      (0U)
#define ENET_MAC_SYSTEM_TIME_SECONDS_TSS(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_SYSTEM_TIME_SECONDS_TSS_SHIFT)) & ENET_MAC_SYSTEM_TIME_SECONDS_TSS_MASK)
/* ENET_MAC_SYSTEM_TIME_NANOSECONDS Register */
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_TSSS_MASK                      (0x7FFFFFFFU)
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_TSSS_SHIFT                     (0U)
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_TSSS(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_SYSTEM_TIME_NANOSECONDS_TSSS_SHIFT)) & ENET_MAC_SYSTEM_TIME_NANOSECONDS_TSSS_MASK)
/* ENET_MAC_SYSTEM_TIME_SECONDS_UPDATE Register */
#define ENET_MAC_SYSTEM_TIME_SECONDS_UPDATE_TSS_MASK                       (0xFFFFFFFFU)
#define ENET_MAC_SYSTEM_TIME_SECONDS_UPDATE_TSS_SHIFT                      (0U)
#define ENET_MAC_SYSTEM_TIME_SECONDS_UPDATE_TSS(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_SYSTEM_TIME_SECONDS_UPDATE_TSS_SHIFT)) & ENET_MAC_SYSTEM_TIME_SECONDS_UPDATE_TSS_MASK)
/* ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE Register */
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_ADDSUB_MASK                    (0x80000000U)
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_ADDSUB_SHIFT                   (31U)
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_ADDSUB(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_ADDSUB_SHIFT)) & ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_ADDSUB_MASK)
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_TSSS_MASK                      (0x7FFFFFFFU)
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_TSSS_SHIFT                     (0U)
#define ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_TSSS(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_TSSS_SHIFT)) & ENET_MAC_SYSTEM_TIME_NANOSECONDS_UPDATE_TSSS_MASK)
/* ENET_MAC_TIMESTAMP_ADDEND Register */
#define ENET_MAC_TIMESTAMP_ADDEND_TSAR_MASK                      (0xFFFFFFFFU)
#define ENET_MAC_TIMESTAMP_ADDEND_TSAR_SHIFT                     (0U)
#define ENET_MAC_TIMESTAMP_ADDEND_TSAR(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_ADDEND_TSAR_SHIFT)) & ENET_MAC_TIMESTAMP_ADDEND_TSAR_MASK)
/* ENET_MAC_SYSTEM_TIME_HIGHER_WORD_SECONDS Register */
#define ENET_MAC_SYSTEM_TIME_HIGHER_WORD_SECONDS_TSHWR_MASK                     (0xFFFFU)
#define ENET_MAC_SYSTEM_TIME_HIGHER_WORD_SECONDS_TSHWR_SHIFT                    (0U)
#define ENET_MAC_SYSTEM_TIME_HIGHER_WORD_SECONDS_TSHWR(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_SYSTEM_TIME_HIGHER_WORD_SECONDS_TSHWR_SHIFT)) & ENET_MAC_SYSTEM_TIME_HIGHER_WORD_SECONDS_TSHWR_MASK)
/* ENET_MAC_TIMESTAMP_STATUS Register */
#define ENET_MAC_TIMESTAMP_STATUS_ATSNS_MASK                     (0x3E000000U)
#define ENET_MAC_TIMESTAMP_STATUS_ATSNS_SHIFT                    (25U)
#define ENET_MAC_TIMESTAMP_STATUS_ATSNS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_ATSNS_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_ATSNS_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_ATSSTM_MASK                    (0x1000000U)
#define ENET_MAC_TIMESTAMP_STATUS_ATSSTM_SHIFT                   (24U)
#define ENET_MAC_TIMESTAMP_STATUS_ATSSTM(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_ATSSTM_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_ATSSTM_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_ATSSTN_MASK                    (0xF0000U)
#define ENET_MAC_TIMESTAMP_STATUS_ATSSTN_SHIFT                   (16U)
#define ENET_MAC_TIMESTAMP_STATUS_ATSSTN(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_ATSSTN_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_ATSSTN_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TXTSSIS_MASK                   (0x8000U)
#define ENET_MAC_TIMESTAMP_STATUS_TXTSSIS_SHIFT                  (15U)
#define ENET_MAC_TIMESTAMP_STATUS_TXTSSIS(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TXTSSIS_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TXTSSIS_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR3_MASK                (0x200U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR3_SHIFT               (9U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR3(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR3_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR3_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT3_MASK                  (0x100U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT3_SHIFT                 (8U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT3(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSTARGT3_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSTARGT3_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR2_MASK                (0x80U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR2_SHIFT               (7U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR2(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR2_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR2_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT2_MASK                  (0x40U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT2_SHIFT                 (6U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT2(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSTARGT2_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSTARGT2_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR1_MASK                (0x20U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR1_SHIFT               (5U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR1(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR1_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR1_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT1_MASK                  (0x10U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT1_SHIFT                 (4U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT1(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSTARGT1_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSTARGT1_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR0_MASK                (0x8U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR0_SHIFT               (3U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR0(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR0_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSTRGTERR0_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_AUXTSTRIG_MASK                 (0x4U)
#define ENET_MAC_TIMESTAMP_STATUS_AUXTSTRIG_SHIFT                (2U)
#define ENET_MAC_TIMESTAMP_STATUS_AUXTSTRIG(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_AUXTSTRIG_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_AUXTSTRIG_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT0_MASK                  (0x2U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT0_SHIFT                 (1U)
#define ENET_MAC_TIMESTAMP_STATUS_TSTARGT0(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSTARGT0_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSTARGT0_MASK)
#define ENET_MAC_TIMESTAMP_STATUS_TSSOVF_MASK                    (0x1U)
#define ENET_MAC_TIMESTAMP_STATUS_TSSOVF_SHIFT                   (0U)
#define ENET_MAC_TIMESTAMP_STATUS_TSSOVF(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_STATUS_TSSOVF_SHIFT)) & ENET_MAC_TIMESTAMP_STATUS_TSSOVF_MASK)
/* ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS Register */
#define ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSMIS_MASK                  (0x80000000U)
#define ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSMIS_SHIFT                 (31U)
#define ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSMIS(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSMIS_SHIFT)) & ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSMIS_MASK)
#define ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSLO_MASK                   (0x7FFFFFFFU)
#define ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSLO_SHIFT                  (0U)
#define ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSLO(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSLO_SHIFT)) & ENET_MAC_TX_TIMESTAMP_STATUS_NANOSECONDS_TXTSSLO_MASK)
/* ENET_MAC_TX_TIMESTAMP_STATUS_SECONDS Register */
#define ENET_MAC_TX_TIMESTAMP_STATUS_SECONDS_TXTSSHI_MASK                   (0xFFFFFFFFU)
#define ENET_MAC_TX_TIMESTAMP_STATUS_SECONDS_TXTSSHI_SHIFT                  (0U)
#define ENET_MAC_TX_TIMESTAMP_STATUS_SECONDS_TXTSSHI(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_TX_TIMESTAMP_STATUS_SECONDS_TXTSSHI_SHIFT)) & ENET_MAC_TX_TIMESTAMP_STATUS_SECONDS_TXTSSHI_MASK)
/* ENET_MAC_AUXILIARY_CONTROL Register */
#define ENET_MAC_AUXILIARY_CONTROL_ATSENn_MASK                    (0xF0U)
#define ENET_MAC_AUXILIARY_CONTROL_ATSENn_SHIFT                   (4U)
#define ENET_MAC_AUXILIARY_CONTROL_ATSENn(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_AUXILIARY_CONTROL_ATSENn_SHIFT)) & ENET_MAC_AUXILIARY_CONTROL_ATSENn_MASK)
#define ENET_MAC_AUXILIARY_CONTROL_ATSFC_MASK                     (0x1U)
#define ENET_MAC_AUXILIARY_CONTROL_ATSFC_SHIFT                    (0U)
#define ENET_MAC_AUXILIARY_CONTROL_ATSFC(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_AUXILIARY_CONTROL_ATSFC_SHIFT)) & ENET_MAC_AUXILIARY_CONTROL_ATSFC_MASK)
/* ENET_MAC_AUXILIARY_TIMESTAMP_NANOSECONDS Register */
#define ENET_MAC_AUXILIARY_TIMESTAMP_NANOSECONDS_AUXTSLO_MASK                   (0x7FFFFFFFU)
#define ENET_MAC_AUXILIARY_TIMESTAMP_NANOSECONDS_AUXTSLO_SHIFT                  (0U)
#define ENET_MAC_AUXILIARY_TIMESTAMP_NANOSECONDS_AUXTSLO(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_AUXILIARY_TIMESTAMP_NANOSECONDS_AUXTSLO_SHIFT)) & ENET_MAC_AUXILIARY_TIMESTAMP_NANOSECONDS_AUXTSLO_MASK)
/* ENET_MAC_AUXILIARY_TIMESTAMP_SECONDS Register */
#define ENET_MAC_AUXILIARY_TIMESTAMP_SECONDS_AUXTSHI_MASK                   (0xFFFFFFFFU)
#define ENET_MAC_AUXILIARY_TIMESTAMP_SECONDS_AUXTSHI_SHIFT                  (0U)
#define ENET_MAC_AUXILIARY_TIMESTAMP_SECONDS_AUXTSHI(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_AUXILIARY_TIMESTAMP_SECONDS_AUXTSHI_SHIFT)) & ENET_MAC_AUXILIARY_TIMESTAMP_SECONDS_AUXTSHI_MASK)
/* ENET_MAC_TIMESTAMP_INGRESS_ASYM_CORR Register */
#define ENET_MAC_TIMESTAMP_INGRESS_ASYM_CORR_OSTIAC_MASK                    (0xFFFFFFFFU)
#define ENET_MAC_TIMESTAMP_INGRESS_ASYM_CORR_OSTIAC_SHIFT                   (0U)
#define ENET_MAC_TIMESTAMP_INGRESS_ASYM_CORR_OSTIAC(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_INGRESS_ASYM_CORR_OSTIAC_SHIFT)) & ENET_MAC_TIMESTAMP_INGRESS_ASYM_CORR_OSTIAC_MASK)
/* ENET_MAC_TIMESTAMP_EGRESS_ASYM_CORR Register */
#define ENET_MAC_TIMESTAMP_EGRESS_ASYM_CORR_OSTEAC_MASK                    (0xFFFFFFFFU)
#define ENET_MAC_TIMESTAMP_EGRESS_ASYM_CORR_OSTEAC_SHIFT                   (0U)
#define ENET_MAC_TIMESTAMP_EGRESS_ASYM_CORR_OSTEAC(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_EGRESS_ASYM_CORR_OSTEAC_SHIFT)) & ENET_MAC_TIMESTAMP_EGRESS_ASYM_CORR_OSTEAC_MASK)
/* ENET_MAC_TIMESTAMP_INGRESS_CORR_NANOSECOND Register */
#define ENET_MAC_TIMESTAMP_INGRESS_CORR_NANOSECOND_TSIC_MASK                      (0xFFFFFFFFU)
#define ENET_MAC_TIMESTAMP_INGRESS_CORR_NANOSECOND_TSIC_SHIFT                     (0U)
#define ENET_MAC_TIMESTAMP_INGRESS_CORR_NANOSECOND_TSIC(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_INGRESS_CORR_NANOSECOND_TSIC_SHIFT)) & ENET_MAC_TIMESTAMP_INGRESS_CORR_NANOSECOND_TSIC_MASK)
/* ENET_MAC_TIMESTAMP_EGRESS_CORR_NANOSECOND Register */
#define ENET_MAC_TIMESTAMP_EGRESS_CORR_NANOSECOND_TSEC_MASK                      (0xFFFFFFFFU)
#define ENET_MAC_TIMESTAMP_EGRESS_CORR_NANOSECOND_TSEC_SHIFT                     (0U)
#define ENET_MAC_TIMESTAMP_EGRESS_CORR_NANOSECOND_TSEC(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_EGRESS_CORR_NANOSECOND_TSEC_SHIFT)) & ENET_MAC_TIMESTAMP_EGRESS_CORR_NANOSECOND_TSEC_MASK)
/* ENET_MAC_TIMESTAMP_INGRESS_CORR_SUBNANOSEC Register */
#define ENET_MAC_TIMESTAMP_INGRESS_CORR_SUBNANOSEC_TSICSNS_MASK                   (0xFF00U)
#define ENET_MAC_TIMESTAMP_INGRESS_CORR_SUBNANOSEC_TSICSNS_SHIFT                  (8U)
#define ENET_MAC_TIMESTAMP_INGRESS_CORR_SUBNANOSEC_TSICSNS(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_INGRESS_CORR_SUBNANOSEC_TSICSNS_SHIFT)) & ENET_MAC_TIMESTAMP_INGRESS_CORR_SUBNANOSEC_TSICSNS_MASK)
/* ENET_MAC_TIMESTAMP_EGRESS_CORR_SUBNANOSEC Register */
#define ENET_MAC_TIMESTAMP_EGRESS_CORR_SUBNANOSEC_TSECSNS_MASK                   (0xFF00U)
#define ENET_MAC_TIMESTAMP_EGRESS_CORR_SUBNANOSEC_TSECSNS_SHIFT                  (8U)
#define ENET_MAC_TIMESTAMP_EGRESS_CORR_SUBNANOSEC_TSECSNS(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_EGRESS_CORR_SUBNANOSEC_TSECSNS_SHIFT)) & ENET_MAC_TIMESTAMP_EGRESS_CORR_SUBNANOSEC_TSECSNS_MASK)
/* ENET_MAC_TIMESTAMP_INGRESS_LATENCY Register */
#define ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLNS_MASK                     (0xFFF0000U)
#define ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLNS_SHIFT                    (16U)
#define ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLNS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLNS_SHIFT)) & ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLNS_MASK)
#define ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLSNS_MASK                    (0xFF00U)
#define ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLSNS_SHIFT                   (8U)
#define ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLSNS(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLSNS_SHIFT)) & ENET_MAC_TIMESTAMP_INGRESS_LATENCY_ITLSNS_MASK)
/* ENET_MAC_TIMESTAMP_EGRESS_LATENCY Register */
#define ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLNS_MASK                     (0xFFF0000U)
#define ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLNS_SHIFT                    (16U)
#define ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLNS(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLNS_SHIFT)) & ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLNS_MASK)
#define ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLSNS_MASK                    (0xFF00U)
#define ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLSNS_SHIFT                   (8U)
#define ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLSNS(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLSNS_SHIFT)) & ENET_MAC_TIMESTAMP_EGRESS_LATENCY_ETLSNS_MASK)
/* ENET_MAC_PPS_CONTROL Register */
#define ENET_MAC_PPS_CONTROL_MCGREN3_MASK                   (0x80000000U)
#define ENET_MAC_PPS_CONTROL_MCGREN3_SHIFT                  (31U)
#define ENET_MAC_PPS_CONTROL_MCGREN3(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_MCGREN3_SHIFT)) & ENET_MAC_PPS_CONTROL_MCGREN3_MASK)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL3_MASK               (0x60000000U)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL3_SHIFT              (29U)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL3(x)                 (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_TRGTMODSEL3_SHIFT)) & ENET_MAC_PPS_CONTROL_TRGTMODSEL3_MASK)
#define ENET_MAC_PPS_CONTROL_PPSCMD3_MASK                   (0xF000000U)
#define ENET_MAC_PPS_CONTROL_PPSCMD3_SHIFT                  (24U)
#define ENET_MAC_PPS_CONTROL_PPSCMD3(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_PPSCMD3_SHIFT)) & ENET_MAC_PPS_CONTROL_PPSCMD3_MASK)
#define ENET_MAC_PPS_CONTROL_MCGREN2_MASK                   (0x800000U)
#define ENET_MAC_PPS_CONTROL_MCGREN2_SHIFT                  (23U)
#define ENET_MAC_PPS_CONTROL_MCGREN2(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_MCGREN2_SHIFT)) & ENET_MAC_PPS_CONTROL_MCGREN2_MASK)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL2_MASK               (0x600000U)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL2_SHIFT              (21U)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL2(x)                 (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_TRGTMODSEL2_SHIFT)) & ENET_MAC_PPS_CONTROL_TRGTMODSEL2_MASK)
#define ENET_MAC_PPS_CONTROL_PPSCMD2_MASK                   (0xF0000U)
#define ENET_MAC_PPS_CONTROL_PPSCMD2_SHIFT                  (16U)
#define ENET_MAC_PPS_CONTROL_PPSCMD2(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_PPSCMD2_SHIFT)) & ENET_MAC_PPS_CONTROL_PPSCMD2_MASK)
#define ENET_MAC_PPS_CONTROL_MCGREN1_MASK                   (0x8000U)
#define ENET_MAC_PPS_CONTROL_MCGREN1_SHIFT                  (15U)
#define ENET_MAC_PPS_CONTROL_MCGREN1(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_MCGREN1_SHIFT)) & ENET_MAC_PPS_CONTROL_MCGREN1_MASK)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL1_MASK               (0x6000U)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL1_SHIFT              (13U)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL1(x)                 (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_TRGTMODSEL1_SHIFT)) & ENET_MAC_PPS_CONTROL_TRGTMODSEL1_MASK)
#define ENET_MAC_PPS_CONTROL_PPSCMD1_MASK                   (0xF00U)
#define ENET_MAC_PPS_CONTROL_PPSCMD1_SHIFT                  (8U)
#define ENET_MAC_PPS_CONTROL_PPSCMD1(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_PPSCMD1_SHIFT)) & ENET_MAC_PPS_CONTROL_PPSCMD1_MASK)
#define ENET_MAC_PPS_CONTROL_MCGREN0_MASK                   (0x80U)
#define ENET_MAC_PPS_CONTROL_MCGREN0_SHIFT                  (7U)
#define ENET_MAC_PPS_CONTROL_MCGREN0(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_MCGREN0_SHIFT)) & ENET_MAC_PPS_CONTROL_MCGREN0_MASK)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL0_MASK               (0x60U)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL0_SHIFT              (5U)
#define ENET_MAC_PPS_CONTROL_TRGTMODSEL0(x)                 (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_TRGTMODSEL0_SHIFT)) & ENET_MAC_PPS_CONTROL_TRGTMODSEL0_MASK)
#define ENET_MAC_PPS_CONTROL_PPSEN0_MASK                    (0x10U)
#define ENET_MAC_PPS_CONTROL_PPSEN0_SHIFT                   (4U)
#define ENET_MAC_PPS_CONTROL_PPSEN0(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_PPSEN0_SHIFT)) & ENET_MAC_PPS_CONTROL_PPSEN0_MASK)
#define ENET_MAC_PPS_CONTROL_PPSCTRL_PPSCMD_MASK            (0xFU)
#define ENET_MAC_PPS_CONTROL_PPSCTRL_PPSCMD_SHIFT           (0U)
#define ENET_MAC_PPS_CONTROL_PPSCTRL_PPSCMD(x)              (((uint32)(((uint32)(x)) << ENET_MAC_PPS_CONTROL_PPSCTRL_PPSCMD_SHIFT)) & ENET_MAC_PPS_CONTROL_PPSCTRL_PPSCMD_MASK)
/* ENET_MAC_PPS_TARGET_TIME_SECONDS Register */
#define ENET_MAC_PPS_TARGET_TIME_SECONDS_TSTRH_MASK                     (0xFFFFFFFFU)
#define ENET_MAC_PPS_TARGET_TIME_SECONDS_TSTRH_SHIFT                    (0U)
#define ENET_MAC_PPS_TARGET_TIME_SECONDS_TSTRH(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_PPS_TARGET_TIME_SECONDS_TSTRH_SHIFT)) & ENET_MAC_PPS_TARGET_TIME_SECONDS_TSTRH_MASK)
/* ENET_MAC_PPS_TARGET_TIME_NANOSECONDS Register */
#define ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TRGTBUSY_MASK                  (0x80000000U)
#define ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TRGTBUSY_SHIFT                 (31U)
#define ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TRGTBUSY(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TRGTBUSY_SHIFT)) & ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TRGTBUSY_MASK)
#define ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TTSL_MASK                      (0x7FFFFFFFU)
#define ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TTSL_SHIFT                     (0U)
#define ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TTSL(x)                        (((uint32)(((uint32)(x)) << ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TTSL_SHIFT)) & ENET_MAC_PPS_TARGET_TIME_NANOSECONDS_TTSL_MASK)
/* ENET_MAC_PPS_INTERVAL Register */
#define ENET_MAC_PPS_INTERVAL_PPSINT_MASK                           (0xFFFFFFFFU)
#define ENET_MAC_PPS_INTERVAL_PPSINT_SHIFT                          (0U)
#define ENET_MAC_PPS_INTERVAL_PPSINT(x)                             (((uint32)(((uint32)(x)) << ENET_MAC_PPS_INTERVAL_PPSINT_SHIFT)) & ENET_MAC_PPS_INTERVAL_PPSINT_MASK)
/* ENET_MAC_PPS_WIDTH Register */
#define ENET_MAC_PPS_WIDTH_PPSWIDTH_MASK                            (0xFFFFFFFFU)
#define ENET_MAC_PPS_WIDTH_PPSWIDTH_SHIFT                           (0U)
#define ENET_MAC_PPS_WIDTH_PPSWIDTH(x)                              (((uint32)(((uint32)(x)) << ENET_MAC_PPS_WIDTH_PPSWIDTH_SHIFT)) & ENET_MAC_PPS_WIDTH_PPSWIDTH_MASK)

/* ENET_MAC_PTO_CONTROL Register */
#define ENET_MAC_PTO_CONTROL_DN_MASK                        (0xFF00U)
#define ENET_MAC_PTO_CONTROL_DN_SHIFT                       (8U)
#define ENET_MAC_PTO_CONTROL_DN(x)                          (((uint32)(((uint32)(x)) << ENET_MAC_PTO_CONTROL_DN_SHIFT)) & ENET_MAC_PTO_CONTROL_DN_MASK)
#define ENET_MAC_PTO_CONTROL_PDRDIS_MASK                    (0x80U)
#define ENET_MAC_PTO_CONTROL_PDRDIS_SHIFT                   (7U)
#define ENET_MAC_PTO_CONTROL_PDRDIS(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PTO_CONTROL_PDRDIS_SHIFT)) & ENET_MAC_PTO_CONTROL_PDRDIS_MASK)
#define ENET_MAC_PTO_CONTROL_DRRDIS_MASK                    (0x40U)
#define ENET_MAC_PTO_CONTROL_DRRDIS_SHIFT                   (6U)
#define ENET_MAC_PTO_CONTROL_DRRDIS(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_PTO_CONTROL_DRRDIS_SHIFT)) & ENET_MAC_PTO_CONTROL_DRRDIS_MASK)
#define ENET_MAC_PTO_CONTROL_APDREQTRIG_MASK                (0x20U)
#define ENET_MAC_PTO_CONTROL_APDREQTRIG_SHIFT               (5U)
#define ENET_MAC_PTO_CONTROL_APDREQTRIG(x)                  (((uint32)(((uint32)(x)) << ENET_MAC_PTO_CONTROL_APDREQTRIG_SHIFT)) & ENET_MAC_PTO_CONTROL_APDREQTRIG_MASK)
#define ENET_MAC_PTO_CONTROL_ASYNCTRIG_MASK                 (0x10U)
#define ENET_MAC_PTO_CONTROL_ASYNCTRIG_SHIFT                (4U)
#define ENET_MAC_PTO_CONTROL_ASYNCTRIG(x)                   (((uint32)(((uint32)(x)) << ENET_MAC_PTO_CONTROL_ASYNCTRIG_SHIFT)) & ENET_MAC_PTO_CONTROL_ASYNCTRIG_MASK)
#define ENET_MAC_PTO_CONTROL_APDREQEN_MASK                  (0x4U)
#define ENET_MAC_PTO_CONTROL_APDREQEN_SHIFT                 (2U)
#define ENET_MAC_PTO_CONTROL_APDREQEN(x)                    (((uint32)(((uint32)(x)) << ENET_MAC_PTO_CONTROL_APDREQEN_SHIFT)) & ENET_MAC_PTO_CONTROL_APDREQEN_MASK)
#define ENET_MAC_PTO_CONTROL_ASYNCEN_MASK                   (0x2U)
#define ENET_MAC_PTO_CONTROL_ASYNCEN_SHIFT                  (1U)
#define ENET_MAC_PTO_CONTROL_ASYNCEN(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_PTO_CONTROL_ASYNCEN_SHIFT)) & ENET_MAC_PTO_CONTROL_ASYNCEN_MASK)
#define ENET_MAC_PTO_CONTROL_PTOEN_MASK                     (0x1U)
#define ENET_MAC_PTO_CONTROL_PTOEN_SHIFT                    (0U)
#define ENET_MAC_PTO_CONTROL_PTOEN(x)                       (((uint32)(((uint32)(x)) << ENET_MAC_PTO_CONTROL_PTOEN_SHIFT)) & ENET_MAC_PTO_CONTROL_PTOEN_MASK)
/* ENET_MAC_SOURCE_PORT_IDENTITY Register */
#define ENET_MAC_SOURCE_PORT_IDENTITY_SPI_MASK                       (0xFFFFFFFFU)
#define ENET_MAC_SOURCE_PORT_IDENTITY_SPI_SHIFT                      (0U)
#define ENET_MAC_SOURCE_PORT_IDENTITY_SPI(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_SOURCE_PORT_IDENTITY_SPI_SHIFT)) & ENET_MAC_SOURCE_PORT_IDENTITY_SPI_MASK)
/* ENET_MAC_LOG_MESSAGE_INTERVAL Register */
#define ENET_MAC_LOG_MESSAGE_INTERVAL_LMPDRI_MASK                    (0xFF000000U)
#define ENET_MAC_LOG_MESSAGE_INTERVAL_LMPDRI_SHIFT                   (24U)
#define ENET_MAC_LOG_MESSAGE_INTERVAL_LMPDRI(x)                      (((uint32)(((uint32)(x)) << ENET_MAC_LOG_MESSAGE_INTERVAL_LMPDRI_SHIFT)) & ENET_MAC_LOG_MESSAGE_INTERVAL_LMPDRI_MASK)
#define ENET_MAC_LOG_MESSAGE_INTERVAL_DRSYNCR_MASK                   (0x700U)
#define ENET_MAC_LOG_MESSAGE_INTERVAL_DRSYNCR_SHIFT                  (8U)
#define ENET_MAC_LOG_MESSAGE_INTERVAL_DRSYNCR(x)                     (((uint32)(((uint32)(x)) << ENET_MAC_LOG_MESSAGE_INTERVAL_DRSYNCR_SHIFT)) & ENET_MAC_LOG_MESSAGE_INTERVAL_DRSYNCR_MASK)
#define ENET_MAC_LOG_MESSAGE_INTERVAL_LSI_MASK                       (0xFFU)
#define ENET_MAC_LOG_MESSAGE_INTERVAL_LSI_SHIFT                      (0U)
#define ENET_MAC_LOG_MESSAGE_INTERVAL_LSI(x)                         (((uint32)(((uint32)(x)) << ENET_MAC_LOG_MESSAGE_INTERVAL_LSI_SHIFT)) & ENET_MAC_LOG_MESSAGE_INTERVAL_LSI_MASK)
/* ENET_MTL_OPERATION_MODE Register */
#define ENET_MTL_OPERATION_MODE_FRPE_MASK                      (0x8000U)
#define ENET_MTL_OPERATION_MODE_FRPE_SHIFT                     (15U)
#define ENET_MTL_OPERATION_MODE_FRPE(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_OPERATION_MODE_FRPE_SHIFT)) & ENET_MTL_OPERATION_MODE_FRPE_MASK)
#define ENET_MTL_OPERATION_MODE_CNTCLR_MASK                    (0x200U)
#define ENET_MTL_OPERATION_MODE_CNTCLR_SHIFT                   (9U)
#define ENET_MTL_OPERATION_MODE_CNTCLR(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_OPERATION_MODE_CNTCLR_SHIFT)) & ENET_MTL_OPERATION_MODE_CNTCLR_MASK)
#define ENET_MTL_OPERATION_MODE_CNTPRST_MASK                   (0x100U)
#define ENET_MTL_OPERATION_MODE_CNTPRST_SHIFT                  (8U)
#define ENET_MTL_OPERATION_MODE_CNTPRST(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_OPERATION_MODE_CNTPRST_SHIFT)) & ENET_MTL_OPERATION_MODE_CNTPRST_MASK)
#define ENET_MTL_OPERATION_MODE_SCHALG_MASK                    (0x60U)
#define ENET_MTL_OPERATION_MODE_SCHALG_SHIFT                   (5U)
#define ENET_MTL_OPERATION_MODE_SCHALG(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_OPERATION_MODE_SCHALG_SHIFT)) & ENET_MTL_OPERATION_MODE_SCHALG_MASK)
#define ENET_MTL_OPERATION_MODE_RAA_MASK                       (0x4U)
#define ENET_MTL_OPERATION_MODE_RAA_SHIFT                      (2U)
#define ENET_MTL_OPERATION_MODE_RAA(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_OPERATION_MODE_RAA_SHIFT)) & ENET_MTL_OPERATION_MODE_RAA_MASK)
#define ENET_MTL_OPERATION_MODE_DTXSTS_MASK                    (0x2U)
#define ENET_MTL_OPERATION_MODE_DTXSTS_SHIFT                   (1U)
#define ENET_MTL_OPERATION_MODE_DTXSTS(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_OPERATION_MODE_DTXSTS_SHIFT)) & ENET_MTL_OPERATION_MODE_DTXSTS_MASK)
/* ENET_MTL_DBG_CTL Register */
#define ENET_MTL_DBG_CTL_EIEC_MASK                          (0x60000U)
#define ENET_MTL_DBG_CTL_EIEC_SHIFT                         (17U)
#define ENET_MTL_DBG_CTL_EIEC(x)                            (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_EIEC_SHIFT)) & ENET_MTL_DBG_CTL_EIEC_MASK)
#define ENET_MTL_DBG_CTL_EIEE_MASK                          (0x10000U)
#define ENET_MTL_DBG_CTL_EIEE_SHIFT                         (16U)
#define ENET_MTL_DBG_CTL_EIEE(x)                            (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_EIEE_SHIFT)) & ENET_MTL_DBG_CTL_EIEE_MASK)
#define ENET_MTL_DBG_CTL_STSIE_MASK                         (0x8000U)
#define ENET_MTL_DBG_CTL_STSIE_SHIFT                        (15U)
#define ENET_MTL_DBG_CTL_STSIE(x)                           (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_STSIE_SHIFT)) & ENET_MTL_DBG_CTL_STSIE_MASK)
#define ENET_MTL_DBG_CTL_PKTIE_MASK                         (0x4000U)
#define ENET_MTL_DBG_CTL_PKTIE_SHIFT                        (14U)
#define ENET_MTL_DBG_CTL_PKTIE(x)                           (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_PKTIE_SHIFT)) & ENET_MTL_DBG_CTL_PKTIE_MASK)
#define ENET_MTL_DBG_CTL_FIFOSEL_MASK                       (0x3000U)
#define ENET_MTL_DBG_CTL_FIFOSEL_SHIFT                      (12U)
#define ENET_MTL_DBG_CTL_FIFOSEL(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_FIFOSEL_SHIFT)) & ENET_MTL_DBG_CTL_FIFOSEL_MASK)
#define ENET_MTL_DBG_CTL_FIFOWREN_MASK                      (0x800U)
#define ENET_MTL_DBG_CTL_FIFOWREN_SHIFT                     (11U)
#define ENET_MTL_DBG_CTL_FIFOWREN(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_FIFOWREN_SHIFT)) & ENET_MTL_DBG_CTL_FIFOWREN_MASK)
#define ENET_MTL_DBG_CTL_FIFORDEN_MASK                      (0x400U)
#define ENET_MTL_DBG_CTL_FIFORDEN_SHIFT                     (10U)
#define ENET_MTL_DBG_CTL_FIFORDEN(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_FIFORDEN_SHIFT)) & ENET_MTL_DBG_CTL_FIFORDEN_MASK)
#define ENET_MTL_DBG_CTL_RSTSEL_MASK                        (0x200U)
#define ENET_MTL_DBG_CTL_RSTSEL_SHIFT                       (9U)
#define ENET_MTL_DBG_CTL_RSTSEL(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_RSTSEL_SHIFT)) & ENET_MTL_DBG_CTL_RSTSEL_MASK)
#define ENET_MTL_DBG_CTL_RSTALL_MASK                        (0x100U)
#define ENET_MTL_DBG_CTL_RSTALL_SHIFT                       (8U)
#define ENET_MTL_DBG_CTL_RSTALL(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_RSTALL_SHIFT)) & ENET_MTL_DBG_CTL_RSTALL_MASK)
#define ENET_MTL_DBG_CTL_PKTSTATE_MASK                      (0x60U)
#define ENET_MTL_DBG_CTL_PKTSTATE_SHIFT                     (5U)
#define ENET_MTL_DBG_CTL_PKTSTATE(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_PKTSTATE_SHIFT)) & ENET_MTL_DBG_CTL_PKTSTATE_MASK)
#define ENET_MTL_DBG_CTL_BYTEEN_MASK                        (0xCU)
#define ENET_MTL_DBG_CTL_BYTEEN_SHIFT                       (2U)
#define ENET_MTL_DBG_CTL_BYTEEN(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_BYTEEN_SHIFT)) & ENET_MTL_DBG_CTL_BYTEEN_MASK)
#define ENET_MTL_DBG_CTL_DBGMOD_MASK                        (0x2U)
#define ENET_MTL_DBG_CTL_DBGMOD_SHIFT                       (1U)
#define ENET_MTL_DBG_CTL_DBGMOD(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_DBGMOD_SHIFT)) & ENET_MTL_DBG_CTL_DBGMOD_MASK)
#define ENET_MTL_DBG_CTL_FDBGEN_MASK                        (0x1U)
#define ENET_MTL_DBG_CTL_FDBGEN_SHIFT                       (0U)
#define ENET_MTL_DBG_CTL_FDBGEN(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_DBG_CTL_FDBGEN_SHIFT)) & ENET_MTL_DBG_CTL_FDBGEN_MASK)
/* ENET_MTL_DBG_STS Register */
#define ENET_MTL_DBG_STS_LOCR_MASK                          (0xFFFF8000U)
#define ENET_MTL_DBG_STS_LOCR_SHIFT                         (15U)
#define ENET_MTL_DBG_STS_LOCR(x)                            (((uint32)(((uint32)(x)) << ENET_MTL_DBG_STS_LOCR_SHIFT)) & ENET_MTL_DBG_STS_LOCR_MASK)
#define ENET_MTL_DBG_STS_STSI_MASK                          (0x200U)
#define ENET_MTL_DBG_STS_STSI_SHIFT                         (9U)
#define ENET_MTL_DBG_STS_STSI(x)                            (((uint32)(((uint32)(x)) << ENET_MTL_DBG_STS_STSI_SHIFT)) & ENET_MTL_DBG_STS_STSI_MASK)
#define ENET_MTL_DBG_STS_PKTI_MASK                          (0x100U)
#define ENET_MTL_DBG_STS_PKTI_SHIFT                         (8U)
#define ENET_MTL_DBG_STS_PKTI(x)                            (((uint32)(((uint32)(x)) << ENET_MTL_DBG_STS_PKTI_SHIFT)) & ENET_MTL_DBG_STS_PKTI_MASK)
#define ENET_MTL_DBG_STS_BYTEEN_MASK                        (0x18U)
#define ENET_MTL_DBG_STS_BYTEEN_SHIFT                       (3U)
#define ENET_MTL_DBG_STS_BYTEEN(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_DBG_STS_BYTEEN_SHIFT)) & ENET_MTL_DBG_STS_BYTEEN_MASK)
#define ENET_MTL_DBG_STS_PKTSTATE_MASK                      (0x6U)
#define ENET_MTL_DBG_STS_PKTSTATE_SHIFT                     (1U)
#define ENET_MTL_DBG_STS_PKTSTATE(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_DBG_STS_PKTSTATE_SHIFT)) & ENET_MTL_DBG_STS_PKTSTATE_MASK)
#define ENET_MTL_DBG_STS_FIFOBUSY_MASK                      (0x1U)
#define ENET_MTL_DBG_STS_FIFOBUSY_SHIFT                     (0U)
#define ENET_MTL_DBG_STS_FIFOBUSY(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_DBG_STS_FIFOBUSY_SHIFT)) & ENET_MTL_DBG_STS_FIFOBUSY_MASK)
/* ENET_MTL_DBG_DATA Register */
#define ENET_MTL_DBG_DATA_FDBGDATA_MASK                     (0xFFFFFFFFU)
#define ENET_MTL_DBG_DATA_FDBGDATA_SHIFT                    (0U)
#define ENET_MTL_DBG_DATA_FDBGDATA(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_DBG_DATA_FDBGDATA_SHIFT)) & ENET_MTL_DBG_DATA_FDBGDATA_MASK)
/* ENET_MTL_INTERRUPT_STATUS Register */
#define ENET_MTL_INTERRUPT_STATUS_MTLPIS_MASK                    (0x800000U)
#define ENET_MTL_INTERRUPT_STATUS_MTLPIS_SHIFT                   (23U)
#define ENET_MTL_INTERRUPT_STATUS_MTLPIS(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_INTERRUPT_STATUS_MTLPIS_SHIFT)) & ENET_MTL_INTERRUPT_STATUS_MTLPIS_MASK)
#define ENET_MTL_INTERRUPT_STATUS_ESTIS_MASK                     (0x400000U)
#define ENET_MTL_INTERRUPT_STATUS_ESTIS_SHIFT                    (22U)
#define ENET_MTL_INTERRUPT_STATUS_ESTIS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_INTERRUPT_STATUS_ESTIS_SHIFT)) & ENET_MTL_INTERRUPT_STATUS_ESTIS_MASK)
#define ENET_MTL_INTERRUPT_STATUS_DBGIS_MASK                     (0x20000U)
#define ENET_MTL_INTERRUPT_STATUS_DBGIS_SHIFT                    (17U)
#define ENET_MTL_INTERRUPT_STATUS_DBGIS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_INTERRUPT_STATUS_DBGIS_SHIFT)) & ENET_MTL_INTERRUPT_STATUS_DBGIS_MASK)
#define ENET_MTL_INTERRUPT_STATUS_MACIS_MASK                     (0x10000U)
#define ENET_MTL_INTERRUPT_STATUS_MACIS_SHIFT                    (16U)
#define ENET_MTL_INTERRUPT_STATUS_MACIS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_INTERRUPT_STATUS_MACIS_SHIFT)) & ENET_MTL_INTERRUPT_STATUS_MACIS_MASK)
#define ENET_MTL_INTERRUPT_STATUS_QnIS_MASK                      (0xFFU)
#define ENET_MTL_INTERRUPT_STATUS_QnIS_SHIFT                     (0U)
#define ENET_MTL_INTERRUPT_STATUS_QnIS(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_INTERRUPT_STATUS_QnIS_SHIFT)) & ENET_MTL_INTERRUPT_STATUS_QnIS_MASK)
/* ENET_MTL_RXQ_DMA_MAP0 Register */
#define ENET_MTL_RXQ_DMA_MAP0_Q3DDMACH_MASK                  (0x10000000U)
#define ENET_MTL_RXQ_DMA_MAP0_Q3DDMACH_SHIFT                 (28U)
#define ENET_MTL_RXQ_DMA_MAP0_Q3DDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP0_Q3DDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP0_Q3DDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP0_Q3MDMACH_MASK                  (0x7000000U)
#define ENET_MTL_RXQ_DMA_MAP0_Q3MDMACH_SHIFT                 (24U)
#define ENET_MTL_RXQ_DMA_MAP0_Q3MDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP0_Q3MDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP0_Q3MDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP0_Q2DDMACH_MASK                  (0x100000U)
#define ENET_MTL_RXQ_DMA_MAP0_Q2DDMACH_SHIFT                 (20U)
#define ENET_MTL_RXQ_DMA_MAP0_Q2DDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP0_Q2DDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP0_Q2DDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP0_Q2MDMACH_MASK                  (0x70000U)
#define ENET_MTL_RXQ_DMA_MAP0_Q2MDMACH_SHIFT                 (16U)
#define ENET_MTL_RXQ_DMA_MAP0_Q2MDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP0_Q2MDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP0_Q2MDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP0_Q1DDMACH_MASK                  (0x1000U)
#define ENET_MTL_RXQ_DMA_MAP0_Q1DDMACH_SHIFT                 (12U)
#define ENET_MTL_RXQ_DMA_MAP0_Q1DDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP0_Q1DDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP0_Q1DDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP0_Q1MDMACH_MASK                  (0x700U)
#define ENET_MTL_RXQ_DMA_MAP0_Q1MDMACH_SHIFT                 (8U)
#define ENET_MTL_RXQ_DMA_MAP0_Q1MDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP0_Q1MDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP0_Q1MDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP0_Q0DDMACH_MASK                  (0x10U)
#define ENET_MTL_RXQ_DMA_MAP0_Q0DDMACH_SHIFT                 (4U)
#define ENET_MTL_RXQ_DMA_MAP0_Q0DDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP0_Q0DDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP0_Q0DDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP0_Q0MDMACH_MASK                  (0x7U)
#define ENET_MTL_RXQ_DMA_MAP0_Q0MDMACH_SHIFT                 (0U)
#define ENET_MTL_RXQ_DMA_MAP0_Q0MDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP0_Q0MDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP0_Q0MDMACH_MASK)
/* ENET_MTL_RXQ_DMA_MAP1 Register */
#define ENET_MTL_RXQ_DMA_MAP1_Q7DDMACH_MASK                  (0x10000000U)
#define ENET_MTL_RXQ_DMA_MAP1_Q7DDMACH_SHIFT                 (28U)
#define ENET_MTL_RXQ_DMA_MAP1_Q7DDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP1_Q7DDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP1_Q7DDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP1_Q7MDMACH_MASK                  (0x7000000U)
#define ENET_MTL_RXQ_DMA_MAP1_Q7MDMACH_SHIFT                 (24U)
#define ENET_MTL_RXQ_DMA_MAP1_Q7MDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP1_Q7MDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP1_Q7MDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP1_Q6DDMACH_MASK                  (0x100000U)
#define ENET_MTL_RXQ_DMA_MAP1_Q6DDMACH_SHIFT                 (20U)
#define ENET_MTL_RXQ_DMA_MAP1_Q6DDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP1_Q6DDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP1_Q6DDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP1_Q6MDMACH_MASK                  (0x70000U)
#define ENET_MTL_RXQ_DMA_MAP1_Q6MDMACH_SHIFT                 (16U)
#define ENET_MTL_RXQ_DMA_MAP1_Q6MDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP1_Q6MDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP1_Q6MDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP1_Q5DDMACH_MASK                  (0x1000U)
#define ENET_MTL_RXQ_DMA_MAP1_Q5DDMACH_SHIFT                 (12U)
#define ENET_MTL_RXQ_DMA_MAP1_Q5DDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP1_Q5DDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP1_Q5DDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP1_Q5MDMACH_MASK                  (0x700U)
#define ENET_MTL_RXQ_DMA_MAP1_Q5MDMACH_SHIFT                 (8U)
#define ENET_MTL_RXQ_DMA_MAP1_Q5MDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP1_Q5MDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP1_Q5MDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP1_Q4DDMACH_MASK                  (0x10U)
#define ENET_MTL_RXQ_DMA_MAP1_Q4DDMACH_SHIFT                 (4U)
#define ENET_MTL_RXQ_DMA_MAP1_Q4DDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP1_Q4DDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP1_Q4DDMACH_MASK)
#define ENET_MTL_RXQ_DMA_MAP1_Q4MDMACH_MASK                  (0x7U)
#define ENET_MTL_RXQ_DMA_MAP1_Q4MDMACH_SHIFT                 (0U)
#define ENET_MTL_RXQ_DMA_MAP1_Q4MDMACH(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXQ_DMA_MAP1_Q4MDMACH_SHIFT)) & ENET_MTL_RXQ_DMA_MAP1_Q4MDMACH_MASK)
/* ENET_MTL_TBS_CTRL Register */
#define ENET_MTL_TBS_CTRL_LEOS_MASK                         (0xFFFFFF00U)
#define ENET_MTL_TBS_CTRL_LEOS_SHIFT                        (8U)
#define ENET_MTL_TBS_CTRL_LEOS(x)                           (((uint32)(((uint32)(x)) << ENET_MTL_TBS_CTRL_LEOS_SHIFT)) & ENET_MTL_TBS_CTRL_LEOS_MASK)
#define ENET_MTL_TBS_CTRL_LEGOS_MASK                        (0x70U)
#define ENET_MTL_TBS_CTRL_LEGOS_SHIFT                       (4U)
#define ENET_MTL_TBS_CTRL_LEGOS(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_TBS_CTRL_LEGOS_SHIFT)) & ENET_MTL_TBS_CTRL_LEGOS_MASK)
#define ENET_MTL_TBS_CTRL_LEOV_MASK                         (0x2U)
#define ENET_MTL_TBS_CTRL_LEOV_SHIFT                        (1U)
#define ENET_MTL_TBS_CTRL_LEOV(x)                           (((uint32)(((uint32)(x)) << ENET_MTL_TBS_CTRL_LEOV_SHIFT)) & ENET_MTL_TBS_CTRL_LEOV_MASK)
#define ENET_MTL_TBS_CTRL_ESTM_MASK                         (0x1U)
#define ENET_MTL_TBS_CTRL_ESTM_SHIFT                        (0U)
#define ENET_MTL_TBS_CTRL_ESTM(x)                           (((uint32)(((uint32)(x)) << ENET_MTL_TBS_CTRL_ESTM_SHIFT)) & ENET_MTL_TBS_CTRL_ESTM_MASK)
/* ENET_MTL_EST_CONTROL Register */
#define ENET_MTL_EST_CONTROL_PTOV_MASK                      (0xFF000000U)
#define ENET_MTL_EST_CONTROL_PTOV_SHIFT                     (24U)
#define ENET_MTL_EST_CONTROL_PTOV(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_CONTROL_PTOV_SHIFT)) & ENET_MTL_EST_CONTROL_PTOV_MASK)
#define ENET_MTL_EST_CONTROL_CTOV_MASK                      (0xFFF000U)
#define ENET_MTL_EST_CONTROL_CTOV_SHIFT                     (12U)
#define ENET_MTL_EST_CONTROL_CTOV(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_CONTROL_CTOV_SHIFT)) & ENET_MTL_EST_CONTROL_CTOV_MASK)
#define ENET_MTL_EST_CONTROL_TILS_MASK                      (0x700U)
#define ENET_MTL_EST_CONTROL_TILS_SHIFT                     (8U)
#define ENET_MTL_EST_CONTROL_TILS(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_CONTROL_TILS_SHIFT)) & ENET_MTL_EST_CONTROL_TILS_MASK)
#define ENET_MTL_EST_CONTROL_LCSE_MASK                      (0xC0U)
#define ENET_MTL_EST_CONTROL_LCSE_SHIFT                     (6U)
#define ENET_MTL_EST_CONTROL_LCSE(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_CONTROL_LCSE_SHIFT)) & ENET_MTL_EST_CONTROL_LCSE_MASK)
#define ENET_MTL_EST_CONTROL_DFBS_MASK                      (0x20U)
#define ENET_MTL_EST_CONTROL_DFBS_SHIFT                     (5U)
#define ENET_MTL_EST_CONTROL_DFBS(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_CONTROL_DFBS_SHIFT)) & ENET_MTL_EST_CONTROL_DFBS_MASK)
#define ENET_MTL_EST_CONTROL_DDBF_MASK                      (0x10U)
#define ENET_MTL_EST_CONTROL_DDBF_SHIFT                     (4U)
#define ENET_MTL_EST_CONTROL_DDBF(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_CONTROL_DDBF_SHIFT)) & ENET_MTL_EST_CONTROL_DDBF_MASK)
#define ENET_MTL_EST_CONTROL_SSWL_MASK                      (0x2U)
#define ENET_MTL_EST_CONTROL_SSWL_SHIFT                     (1U)
#define ENET_MTL_EST_CONTROL_SSWL(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_CONTROL_SSWL_SHIFT)) & ENET_MTL_EST_CONTROL_SSWL_MASK)
#define ENET_MTL_EST_CONTROL_EEST_MASK                      (0x1U)
#define ENET_MTL_EST_CONTROL_EEST_SHIFT                     (0U)
#define ENET_MTL_EST_CONTROL_EEST(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_CONTROL_EEST_SHIFT)) & ENET_MTL_EST_CONTROL_EEST_MASK)
/* ENET_MTL_EST_EXT_CONTROL Register */
#define ENET_MTL_EST_EXT_CONTROL_OVHD_MASK                      (0x3FU)
#define ENET_MTL_EST_EXT_CONTROL_OVHD_SHIFT                     (0U)
#define ENET_MTL_EST_EXT_CONTROL_OVHD(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_EXT_CONTROL_OVHD_SHIFT)) & ENET_MTL_EST_EXT_CONTROL_OVHD_MASK)
/* ENET_MTL_EST_STATUS Register */
#define ENET_MTL_EST_STATUS_CGSN_MASK                       (0xF0000U)
#define ENET_MTL_EST_STATUS_CGSN_SHIFT                      (16U)
#define ENET_MTL_EST_STATUS_CGSN(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_STATUS_CGSN_SHIFT)) & ENET_MTL_EST_STATUS_CGSN_MASK)
#define ENET_MTL_EST_STATUS_BTRL_MASK                       (0xF00U)
#define ENET_MTL_EST_STATUS_BTRL_SHIFT                      (8U)
#define ENET_MTL_EST_STATUS_BTRL(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_STATUS_BTRL_SHIFT)) & ENET_MTL_EST_STATUS_BTRL_MASK)
#define ENET_MTL_EST_STATUS_SWOL_MASK                       (0x80U)
#define ENET_MTL_EST_STATUS_SWOL_SHIFT                      (7U)
#define ENET_MTL_EST_STATUS_SWOL(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_STATUS_SWOL_SHIFT)) & ENET_MTL_EST_STATUS_SWOL_MASK)
#define ENET_MTL_EST_STATUS_CGCE_MASK                       (0x10U)
#define ENET_MTL_EST_STATUS_CGCE_SHIFT                      (4U)
#define ENET_MTL_EST_STATUS_CGCE(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_STATUS_CGCE_SHIFT)) & ENET_MTL_EST_STATUS_CGCE_MASK)
#define ENET_MTL_EST_STATUS_HLBS_MASK                       (0x8U)
#define ENET_MTL_EST_STATUS_HLBS_SHIFT                      (3U)
#define ENET_MTL_EST_STATUS_HLBS(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_STATUS_HLBS_SHIFT)) & ENET_MTL_EST_STATUS_HLBS_MASK)
#define ENET_MTL_EST_STATUS_HLBF_MASK                       (0x4U)
#define ENET_MTL_EST_STATUS_HLBF_SHIFT                      (2U)
#define ENET_MTL_EST_STATUS_HLBF(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_STATUS_HLBF_SHIFT)) & ENET_MTL_EST_STATUS_HLBF_MASK)
#define ENET_MTL_EST_STATUS_BTRE_MASK                       (0x2U)
#define ENET_MTL_EST_STATUS_BTRE_SHIFT                      (1U)
#define ENET_MTL_EST_STATUS_BTRE(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_STATUS_BTRE_SHIFT)) & ENET_MTL_EST_STATUS_BTRE_MASK)
#define ENET_MTL_EST_STATUS_SWLC_MASK                       (0x1U)
#define ENET_MTL_EST_STATUS_SWLC_SHIFT                      (0U)
#define ENET_MTL_EST_STATUS_SWLC(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_STATUS_SWLC_SHIFT)) & ENET_MTL_EST_STATUS_SWLC_MASK)
/* ENET_MTL_EST_SCH_ERROR Register */
#define ENET_MTL_EST_SCH_ERROR_SEQN_MASK                      (0x7U)
#define ENET_MTL_EST_SCH_ERROR_SEQN_SHIFT                     (0U)
#define ENET_MTL_EST_SCH_ERROR_SEQN(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_SCH_ERROR_SEQN_SHIFT)) & ENET_MTL_EST_SCH_ERROR_SEQN_MASK)
/* ENET_MTL_EST_FRM_SIZE_ERROR Register */
#define ENET_MTL_EST_FRM_SIZE_ERROR_FEQN_MASK                      (0x7U)
#define ENET_MTL_EST_FRM_SIZE_ERROR_FEQN_SHIFT                     (0U)
#define ENET_MTL_EST_FRM_SIZE_ERROR_FEQN(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_FRM_SIZE_ERROR_FEQN_SHIFT)) & ENET_MTL_EST_FRM_SIZE_ERROR_FEQN_MASK)
/* ENET_MTL_EST_FRM_SIZE_CAPTURE Register */
#define ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFQ_MASK                      (0x70000U)
#define ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFQ_SHIFT                     (16U)
#define ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFQ(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFQ_SHIFT)) & ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFQ_MASK)
#define ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFS_MASK                      (0xFFFFU)
#define ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFS_SHIFT                     (0U)
#define ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFS(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFS_SHIFT)) & ENET_MTL_EST_FRM_SIZE_CAPTURE_HBFS_MASK)
/* ENET_MTL_EST_INTR_ENABLE Register */
#define ENET_MTL_EST_INTR_ENABLE_CGCE_MASK                      (0x10U)
#define ENET_MTL_EST_INTR_ENABLE_CGCE_SHIFT                     (4U)
#define ENET_MTL_EST_INTR_ENABLE_CGCE(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_INTR_ENABLE_CGCE_SHIFT)) & ENET_MTL_EST_INTR_ENABLE_CGCE_MASK)
#define ENET_MTL_EST_INTR_ENABLE_IEHS_MASK                      (0x8U)
#define ENET_MTL_EST_INTR_ENABLE_IEHS_SHIFT                     (3U)
#define ENET_MTL_EST_INTR_ENABLE_IEHS(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_INTR_ENABLE_IEHS_SHIFT)) & ENET_MTL_EST_INTR_ENABLE_IEHS_MASK)
#define ENET_MTL_EST_INTR_ENABLE_IEHF_MASK                      (0x4U)
#define ENET_MTL_EST_INTR_ENABLE_IEHF_SHIFT                     (2U)
#define ENET_MTL_EST_INTR_ENABLE_IEHF(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_INTR_ENABLE_IEHF_SHIFT)) & ENET_MTL_EST_INTR_ENABLE_IEHF_MASK)
#define ENET_MTL_EST_INTR_ENABLE_IEBE_MASK                      (0x2U)
#define ENET_MTL_EST_INTR_ENABLE_IEBE_SHIFT                     (1U)
#define ENET_MTL_EST_INTR_ENABLE_IEBE(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_INTR_ENABLE_IEBE_SHIFT)) & ENET_MTL_EST_INTR_ENABLE_IEBE_MASK)
#define ENET_MTL_EST_INTR_ENABLE_IECC_MASK                      (0x1U)
#define ENET_MTL_EST_INTR_ENABLE_IECC_SHIFT                     (0U)
#define ENET_MTL_EST_INTR_ENABLE_IECC(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_INTR_ENABLE_IECC_SHIFT)) & ENET_MTL_EST_INTR_ENABLE_IECC_MASK)
/* ENET_MTL_EST_GCL_CONTROL Register */
#define ENET_MTL_EST_GCL_CONTROL_ESTEIEC_MASK                   (0xC00000U)
#define ENET_MTL_EST_GCL_CONTROL_ESTEIEC_SHIFT                  (22U)
#define ENET_MTL_EST_GCL_CONTROL_ESTEIEC(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_ESTEIEC_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_ESTEIEC_MASK)
#define ENET_MTL_EST_GCL_CONTROL_ESTEIEE_MASK                   (0x200000U)
#define ENET_MTL_EST_GCL_CONTROL_ESTEIEE_SHIFT                  (21U)
#define ENET_MTL_EST_GCL_CONTROL_ESTEIEE(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_ESTEIEE_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_ESTEIEE_MASK)
#define ENET_MTL_EST_GCL_CONTROL_ERR0_MASK                      (0x100000U)
#define ENET_MTL_EST_GCL_CONTROL_ERR0_SHIFT                     (20U)
#define ENET_MTL_EST_GCL_CONTROL_ERR0(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_ERR0_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_ERR0_MASK)
#define ENET_MTL_EST_GCL_CONTROL_ADDR_MASK                      (0x1FF00U)
#define ENET_MTL_EST_GCL_CONTROL_ADDR_SHIFT                     (8U)
#define ENET_MTL_EST_GCL_CONTROL_ADDR(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_ADDR_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_ADDR_MASK)
#define ENET_MTL_EST_GCL_CONTROL_DBGB_MASK                      (0x20U)
#define ENET_MTL_EST_GCL_CONTROL_DBGB_SHIFT                     (5U)
#define ENET_MTL_EST_GCL_CONTROL_DBGB(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_DBGB_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_DBGB_MASK)
#define ENET_MTL_EST_GCL_CONTROL_DBGM_MASK                      (0x10U)
#define ENET_MTL_EST_GCL_CONTROL_DBGM_SHIFT                     (4U)
#define ENET_MTL_EST_GCL_CONTROL_DBGM(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_DBGM_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_DBGM_MASK)
#define ENET_MTL_EST_GCL_CONTROL_GCRR_MASK                      (0x4U)
#define ENET_MTL_EST_GCL_CONTROL_GCRR_SHIFT                     (2U)
#define ENET_MTL_EST_GCL_CONTROL_GCRR(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_GCRR_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_GCRR_MASK)
#define ENET_MTL_EST_GCL_CONTROL_R1W0_MASK                      (0x2U)
#define ENET_MTL_EST_GCL_CONTROL_R1W0_SHIFT                     (1U)
#define ENET_MTL_EST_GCL_CONTROL_R1W0(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_R1W0_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_R1W0_MASK)
#define ENET_MTL_EST_GCL_CONTROL_SRWO_MASK                      (0x1U)
#define ENET_MTL_EST_GCL_CONTROL_SRWO_SHIFT                     (0U)
#define ENET_MTL_EST_GCL_CONTROL_SRWO(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_CONTROL_SRWO_SHIFT)) & ENET_MTL_EST_GCL_CONTROL_SRWO_MASK)
/* ENET_MTL_EST_GCL_DATA Register */
#define ENET_MTL_EST_GCL_DATA_GCD_MASK                       (0xFFFFFFFFU)
#define ENET_MTL_EST_GCL_DATA_GCD_SHIFT                      (0U)
#define ENET_MTL_EST_GCL_DATA_GCD(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_EST_GCL_DATA_GCD_SHIFT)) & ENET_MTL_EST_GCL_DATA_GCD_MASK)
/* ENET_MTL_FPE_CTRL_STS Register */
#define ENET_MTL_FPE_CTRL_STS_HRS_MASK                       (0x10000000U)
#define ENET_MTL_FPE_CTRL_STS_HRS_SHIFT                      (28U)
#define ENET_MTL_FPE_CTRL_STS_HRS(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_FPE_CTRL_STS_HRS_SHIFT)) & ENET_MTL_FPE_CTRL_STS_HRS_MASK)
#define ENET_MTL_FPE_CTRL_STS_PEC_MASK                       (0x300U)
#define ENET_MTL_FPE_CTRL_STS_PEC_SHIFT                      (8U)
#define ENET_MTL_FPE_CTRL_STS_PEC(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_FPE_CTRL_STS_PEC_SHIFT)) & ENET_MTL_FPE_CTRL_STS_PEC_MASK)
#define ENET_MTL_FPE_CTRL_STS_AFSZ_MASK                      (0x3U)
#define ENET_MTL_FPE_CTRL_STS_AFSZ_SHIFT                     (0U)
#define ENET_MTL_FPE_CTRL_STS_AFSZ(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_FPE_CTRL_STS_AFSZ_SHIFT)) & ENET_MTL_FPE_CTRL_STS_AFSZ_MASK)
/* ENET_MTL_FPE_ADVANCE Register */
#define ENET_MTL_FPE_ADVANCE_RADV_MASK                      (0xFFFF0000U)
#define ENET_MTL_FPE_ADVANCE_RADV_SHIFT                     (16U)
#define ENET_MTL_FPE_ADVANCE_RADV(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_FPE_ADVANCE_RADV_SHIFT)) & ENET_MTL_FPE_ADVANCE_RADV_MASK)
#define ENET_MTL_FPE_ADVANCE_HADV_MASK                      (0xFFFFU)
#define ENET_MTL_FPE_ADVANCE_HADV_SHIFT                     (0U)
#define ENET_MTL_FPE_ADVANCE_HADV(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_FPE_ADVANCE_HADV_SHIFT)) & ENET_MTL_FPE_ADVANCE_HADV_MASK)
/* ENET_MTL_RXP_CONTROL_STATUS Register */
#define ENET_MTL_RXP_CONTROL_STATUS_RXPI_MASK                      (0x80000000U)
#define ENET_MTL_RXP_CONTROL_STATUS_RXPI_SHIFT                     (31U)
#define ENET_MTL_RXP_CONTROL_STATUS_RXPI(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_RXP_CONTROL_STATUS_RXPI_SHIFT)) & ENET_MTL_RXP_CONTROL_STATUS_RXPI_MASK)
#define ENET_MTL_RXP_CONTROL_STATUS_NPE_MASK                       (0xFF0000U)
#define ENET_MTL_RXP_CONTROL_STATUS_NPE_SHIFT                      (16U)
#define ENET_MTL_RXP_CONTROL_STATUS_NPE(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_RXP_CONTROL_STATUS_NPE_SHIFT)) & ENET_MTL_RXP_CONTROL_STATUS_NPE_MASK)
#define ENET_MTL_RXP_CONTROL_STATUS_NVE_MASK                       (0xFFU)
#define ENET_MTL_RXP_CONTROL_STATUS_NVE_SHIFT                      (0U)
#define ENET_MTL_RXP_CONTROL_STATUS_NVE(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_RXP_CONTROL_STATUS_NVE_SHIFT)) & ENET_MTL_RXP_CONTROL_STATUS_NVE_MASK)
/* ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS Register */
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIE_MASK                    (0x80000U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIE_SHIFT                   (19U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIE(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIE_SHIFT)) & ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIE_MASK)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIE_MASK                    (0x40000U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIE_SHIFT                   (18U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIE(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIE_SHIFT)) & ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIE_MASK)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIE_MASK                   (0x20000U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIE_SHIFT                  (17U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIE(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIE_SHIFT)) & ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIE_MASK)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIE_MASK                   (0x10000U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIE_SHIFT                  (16U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIE(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIE_SHIFT)) & ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIE_MASK)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIS_MASK                    (0x8U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIS_SHIFT                   (3U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIS(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIS_SHIFT)) & ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_PDRFIS_MASK)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIS_MASK                    (0x4U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIS_SHIFT                   (2U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIS(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIS_SHIFT)) & ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_FOOVIS_MASK)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIS_MASK                   (0x2U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIS_SHIFT                  (1U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIS(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIS_SHIFT)) & ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NPEOVIS_MASK)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIS_MASK                   (0x1U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIS_SHIFT                  (0U)
#define ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIS(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIS_SHIFT)) & ENET_MTL_RXP_INTERRUPT_CONTROL_STATUS_NVEOVIS_MASK)
/* ENET_MTL_RXP_DROP_CNT Register */
#define ENET_MTL_RXP_DROP_CNT_RXPDCOVF_MASK                  (0x80000000U)
#define ENET_MTL_RXP_DROP_CNT_RXPDCOVF_SHIFT                 (31U)
#define ENET_MTL_RXP_DROP_CNT_RXPDCOVF(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXP_DROP_CNT_RXPDCOVF_SHIFT)) & ENET_MTL_RXP_DROP_CNT_RXPDCOVF_MASK)
#define ENET_MTL_RXP_DROP_CNT_RXPDC_MASK                     (0x7FFFFFFFU)
#define ENET_MTL_RXP_DROP_CNT_RXPDC_SHIFT                    (0U)
#define ENET_MTL_RXP_DROP_CNT_RXPDC(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_RXP_DROP_CNT_RXPDC_SHIFT)) & ENET_MTL_RXP_DROP_CNT_RXPDC_MASK)
/* ENET_MTL_RXP_ERROR_CNT Register */
#define ENET_MTL_RXP_ERROR_CNT_RXPECOVF_MASK                  (0x80000000U)
#define ENET_MTL_RXP_ERROR_CNT_RXPECOVF_SHIFT                 (31U)
#define ENET_MTL_RXP_ERROR_CNT_RXPECOVF(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_RXP_ERROR_CNT_RXPECOVF_SHIFT)) & ENET_MTL_RXP_ERROR_CNT_RXPECOVF_MASK)
#define ENET_MTL_RXP_ERROR_CNT_RXPEC_MASK                     (0x7FFFFFFFU)
#define ENET_MTL_RXP_ERROR_CNT_RXPEC_SHIFT                    (0U)
#define ENET_MTL_RXP_ERROR_CNT_RXPEC(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_RXP_ERROR_CNT_RXPEC_SHIFT)) & ENET_MTL_RXP_ERROR_CNT_RXPEC_MASK)
/* ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS Register */
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_STARTBUSY_MASK                 (0x80000000U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_STARTBUSY_SHIFT                (31U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_STARTBUSY(x)                   (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_STARTBUSY_SHIFT)) & ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_STARTBUSY_MASK)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEC_MASK                   (0x600000U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEC_SHIFT                  (21U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEC(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEC_SHIFT)) & ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEC_MASK)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEE_MASK                   (0x100000U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEE_SHIFT                  (20U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEE(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEE_SHIFT)) & ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_RXPEIEE_MASK)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_WRRDN_MASK                     (0x10000U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_WRRDN_SHIFT                    (16U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_WRRDN(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_WRRDN_SHIFT)) & ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_WRRDN_MASK)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_ADDR_MASK                      (0xFFU)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_ADDR_SHIFT                     (0U)
#define ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_ADDR(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_ADDR_SHIFT)) & ENET_MTL_RXP_INDIRECT_ACC_CONTROL_STATUS_ADDR_MASK)
/* ENET_MTL_RXP_INDIRECT_ACC_DATA Register */
#define ENET_MTL_RXP_INDIRECT_ACC_DATA_DATA_MASK                      (0xFFFFFFFFU)
#define ENET_MTL_RXP_INDIRECT_ACC_DATA_DATA_SHIFT                     (0U)
#define ENET_MTL_RXP_INDIRECT_ACC_DATA_DATA(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_RXP_INDIRECT_ACC_DATA_DATA_SHIFT)) & ENET_MTL_RXP_INDIRECT_ACC_DATA_DATA_MASK)
/* ENET_MTL_RXP_BYPASS_CNT Register */
#define ENET_MTL_RXP_BYPASS_CNT_RXPBCOF_MASK                   (0x80000000U)
#define ENET_MTL_RXP_BYPASS_CNT_RXPBCOF_SHIFT                  (31U)
#define ENET_MTL_RXP_BYPASS_CNT_RXPBCOF(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_RXP_BYPASS_CNT_RXPBCOF_SHIFT)) & ENET_MTL_RXP_BYPASS_CNT_RXPBCOF_MASK)
#define ENET_MTL_RXP_BYPASS_CNT_RXPBC_MASK                     (0x7FFFFFFFU)
#define ENET_MTL_RXP_BYPASS_CNT_RXPBC_SHIFT                    (0U)
#define ENET_MTL_RXP_BYPASS_CNT_RXPBC(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_RXP_BYPASS_CNT_RXPBC_SHIFT)) & ENET_MTL_RXP_BYPASS_CNT_RXPBC_MASK)
/* ENET_MTL_ECC_CONTROL Register */
#define ENET_MTL_ECC_CONTROL_MEEAO_MASK                     (0x100U)
#define ENET_MTL_ECC_CONTROL_MEEAO_SHIFT                    (8U)
#define ENET_MTL_ECC_CONTROL_MEEAO(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_CONTROL_MEEAO_SHIFT)) & ENET_MTL_ECC_CONTROL_MEEAO_MASK)
#define ENET_MTL_ECC_CONTROL_TSOEE_MASK                     (0x10U)
#define ENET_MTL_ECC_CONTROL_TSOEE_SHIFT                    (4U)
#define ENET_MTL_ECC_CONTROL_TSOEE(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_CONTROL_TSOEE_SHIFT)) & ENET_MTL_ECC_CONTROL_TSOEE_MASK)
#define ENET_MTL_ECC_CONTROL_MRXPEE_MASK                    (0x8U)
#define ENET_MTL_ECC_CONTROL_MRXPEE_SHIFT                   (3U)
#define ENET_MTL_ECC_CONTROL_MRXPEE(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_ECC_CONTROL_MRXPEE_SHIFT)) & ENET_MTL_ECC_CONTROL_MRXPEE_MASK)
#define ENET_MTL_ECC_CONTROL_MESTEE_MASK                    (0x4U)
#define ENET_MTL_ECC_CONTROL_MESTEE_SHIFT                   (2U)
#define ENET_MTL_ECC_CONTROL_MESTEE(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_ECC_CONTROL_MESTEE_SHIFT)) & ENET_MTL_ECC_CONTROL_MESTEE_MASK)
#define ENET_MTL_ECC_CONTROL_MRXEE_MASK                     (0x2U)
#define ENET_MTL_ECC_CONTROL_MRXEE_SHIFT                    (1U)
#define ENET_MTL_ECC_CONTROL_MRXEE(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_CONTROL_MRXEE_SHIFT)) & ENET_MTL_ECC_CONTROL_MRXEE_MASK)
#define ENET_MTL_ECC_CONTROL_MTXEE_MASK                     (0x1U)
#define ENET_MTL_ECC_CONTROL_MTXEE_SHIFT                    (0U)
#define ENET_MTL_ECC_CONTROL_MTXEE(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_CONTROL_MTXEE_SHIFT)) & ENET_MTL_ECC_CONTROL_MTXEE_MASK)
/* ENET_MTL_SAFETY_INTERRUPT_STATUS Register */
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MCSIS_MASK                     (0x80000000U)
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MCSIS_SHIFT                    (31U)
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MCSIS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_SAFETY_INTERRUPT_STATUS_MCSIS_SHIFT)) & ENET_MTL_SAFETY_INTERRUPT_STATUS_MCSIS_MASK)
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MEUIS_MASK                     (0x2U)
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MEUIS_SHIFT                    (1U)
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MEUIS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_SAFETY_INTERRUPT_STATUS_MEUIS_SHIFT)) & ENET_MTL_SAFETY_INTERRUPT_STATUS_MEUIS_MASK)
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MECIS_MASK                     (0x1U)
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MECIS_SHIFT                    (0U)
#define ENET_MTL_SAFETY_INTERRUPT_STATUS_MECIS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_SAFETY_INTERRUPT_STATUS_MECIS_SHIFT)) & ENET_MTL_SAFETY_INTERRUPT_STATUS_MECIS_MASK)
/* ENET_MTL_ECC_INTERRUPT_ENABLE Register */
#define ENET_MTL_ECC_INTERRUPT_ENABLE_RPCEIE_MASK                    (0x1000U)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_RPCEIE_SHIFT                   (12U)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_RPCEIE(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_ENABLE_RPCEIE_SHIFT)) & ENET_MTL_ECC_INTERRUPT_ENABLE_RPCEIE_MASK)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_ECEIE_MASK                     (0x100U)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_ECEIE_SHIFT                    (8U)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_ECEIE(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_ENABLE_ECEIE_SHIFT)) & ENET_MTL_ECC_INTERRUPT_ENABLE_ECEIE_MASK)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_RXCEIE_MASK                    (0x10U)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_RXCEIE_SHIFT                   (4U)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_RXCEIE(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_ENABLE_RXCEIE_SHIFT)) & ENET_MTL_ECC_INTERRUPT_ENABLE_RXCEIE_MASK)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_TXCEIE_MASK                    (0x1U)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_TXCEIE_SHIFT                   (0U)
#define ENET_MTL_ECC_INTERRUPT_ENABLE_TXCEIE(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_ENABLE_TXCEIE_SHIFT)) & ENET_MTL_ECC_INTERRUPT_ENABLE_TXCEIE_MASK)
/* ENET_MTL_ECC_INTERRUPT_STATUS Register */
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPUES_MASK                     (0x4000U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPUES_SHIFT                    (14U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPUES(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_RPUES_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_RPUES_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPAMS_MASK                     (0x2000U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPAMS_SHIFT                    (13U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPAMS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_RPAMS_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_RPAMS_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPCES_MASK                     (0x1000U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPCES_SHIFT                    (12U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RPCES(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_RPCES_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_RPCES_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_EUES_MASK                      (0x400U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_EUES_SHIFT                     (10U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_EUES(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_EUES_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_EUES_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_EAMS_MASK                      (0x200U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_EAMS_SHIFT                     (9U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_EAMS(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_EAMS_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_EAMS_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_ECES_MASK                      (0x100U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_ECES_SHIFT                     (8U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_ECES(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_ECES_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_ECES_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXUES_MASK                     (0x40U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXUES_SHIFT                    (6U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXUES(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_RXUES_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_RXUES_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXAMS_MASK                     (0x20U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXAMS_SHIFT                    (5U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXAMS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_RXAMS_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_RXAMS_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXCES_MASK                     (0x10U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXCES_SHIFT                    (4U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_RXCES(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_RXCES_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_RXCES_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXUES_MASK                     (0x4U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXUES_SHIFT                    (2U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXUES(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_TXUES_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_TXUES_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXAMS_MASK                     (0x2U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXAMS_SHIFT                    (1U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXAMS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_TXAMS_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_TXAMS_MASK)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXCES_MASK                     (0x1U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXCES_SHIFT                    (0U)
#define ENET_MTL_ECC_INTERRUPT_STATUS_TXCES(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_INTERRUPT_STATUS_TXCES_SHIFT)) & ENET_MTL_ECC_INTERRUPT_STATUS_TXCES_MASK)
/* ENET_MTL_ECC_ERR_STS_RCTL Register */
#define ENET_MTL_ECC_ERR_STS_RCTL_CUES_MASK                      (0x20U)
#define ENET_MTL_ECC_ERR_STS_RCTL_CUES_SHIFT                     (5U)
#define ENET_MTL_ECC_ERR_STS_RCTL_CUES(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_ECC_ERR_STS_RCTL_CUES_SHIFT)) & ENET_MTL_ECC_ERR_STS_RCTL_CUES_MASK)
#define ENET_MTL_ECC_ERR_STS_RCTL_CCES_MASK                      (0x10U)
#define ENET_MTL_ECC_ERR_STS_RCTL_CCES_SHIFT                     (4U)
#define ENET_MTL_ECC_ERR_STS_RCTL_CCES(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_ECC_ERR_STS_RCTL_CCES_SHIFT)) & ENET_MTL_ECC_ERR_STS_RCTL_CCES_MASK)
#define ENET_MTL_ECC_ERR_STS_RCTL_EMS_MASK                       (0xEU)
#define ENET_MTL_ECC_ERR_STS_RCTL_EMS_SHIFT                      (1U)
#define ENET_MTL_ECC_ERR_STS_RCTL_EMS(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_ECC_ERR_STS_RCTL_EMS_SHIFT)) & ENET_MTL_ECC_ERR_STS_RCTL_EMS_MASK)
#define ENET_MTL_ECC_ERR_STS_RCTL_EESRE_MASK                     (0x1U)
#define ENET_MTL_ECC_ERR_STS_RCTL_EESRE_SHIFT                    (0U)
#define ENET_MTL_ECC_ERR_STS_RCTL_EESRE(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_ERR_STS_RCTL_EESRE_SHIFT)) & ENET_MTL_ECC_ERR_STS_RCTL_EESRE_MASK)
/* ENET_MTL_ECC_ERR_ADDR_STATUS Register */
#define ENET_MTL_ECC_ERR_ADDR_STATUS_EUEAS_MASK                     (0xFFFF0000U)
#define ENET_MTL_ECC_ERR_ADDR_STATUS_EUEAS_SHIFT                    (16U)
#define ENET_MTL_ECC_ERR_ADDR_STATUS_EUEAS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_ERR_ADDR_STATUS_EUEAS_SHIFT)) & ENET_MTL_ECC_ERR_ADDR_STATUS_EUEAS_MASK)
#define ENET_MTL_ECC_ERR_ADDR_STATUS_ECEAS_MASK                     (0xFFFFU)
#define ENET_MTL_ECC_ERR_ADDR_STATUS_ECEAS_SHIFT                    (0U)
#define ENET_MTL_ECC_ERR_ADDR_STATUS_ECEAS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_ERR_ADDR_STATUS_ECEAS_SHIFT)) & ENET_MTL_ECC_ERR_ADDR_STATUS_ECEAS_MASK)
/* ENET_MTL_ECC_ERR_CNTR_STATUS Register */
#define ENET_MTL_ECC_ERR_CNTR_STATUS_EUECS_MASK                     (0xF0000U)
#define ENET_MTL_ECC_ERR_CNTR_STATUS_EUECS_SHIFT                    (16U)
#define ENET_MTL_ECC_ERR_CNTR_STATUS_EUECS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_ERR_CNTR_STATUS_EUECS_SHIFT)) & ENET_MTL_ECC_ERR_CNTR_STATUS_EUECS_MASK)
#define ENET_MTL_ECC_ERR_CNTR_STATUS_ECECS_MASK                     (0xFFU)
#define ENET_MTL_ECC_ERR_CNTR_STATUS_ECECS_SHIFT                    (0U)
#define ENET_MTL_ECC_ERR_CNTR_STATUS_ECECS(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_ECC_ERR_CNTR_STATUS_ECECS_SHIFT)) & ENET_MTL_ECC_ERR_CNTR_STATUS_ECECS_MASK)
/* ENET_MTL_DPP_CONTROL Register */
#define ENET_MTL_DPP_CONTROL_IPECW_MASK                     (0x2000U)
#define ENET_MTL_DPP_CONTROL_IPECW_SHIFT                    (13U)
#define ENET_MTL_DPP_CONTROL_IPECW(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPECW_SHIFT)) & ENET_MTL_DPP_CONTROL_IPECW_MASK)
#define ENET_MTL_DPP_CONTROL_IPEASW_MASK                    (0x1000U)
#define ENET_MTL_DPP_CONTROL_IPEASW_SHIFT                   (12U)
#define ENET_MTL_DPP_CONTROL_IPEASW(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPEASW_SHIFT)) & ENET_MTL_DPP_CONTROL_IPEASW_MASK)
#define ENET_MTL_DPP_CONTROL_IPERD_MASK                     (0x800U)
#define ENET_MTL_DPP_CONTROL_IPERD_SHIFT                    (11U)
#define ENET_MTL_DPP_CONTROL_IPERD(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPERD_SHIFT)) & ENET_MTL_DPP_CONTROL_IPERD_MASK)
#define ENET_MTL_DPP_CONTROL_IPETD_MASK                     (0x400U)
#define ENET_MTL_DPP_CONTROL_IPETD_SHIFT                    (10U)
#define ENET_MTL_DPP_CONTROL_IPETD(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPETD_SHIFT)) & ENET_MTL_DPP_CONTROL_IPETD_MASK)
#define ENET_MTL_DPP_CONTROL_IPETSO_MASK                    (0x200U)
#define ENET_MTL_DPP_CONTROL_IPETSO_SHIFT                   (9U)
#define ENET_MTL_DPP_CONTROL_IPETSO(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPETSO_SHIFT)) & ENET_MTL_DPP_CONTROL_IPETSO_MASK)
#define ENET_MTL_DPP_CONTROL_IPEDDC_MASK                    (0x100U)
#define ENET_MTL_DPP_CONTROL_IPEDDC_SHIFT                   (8U)
#define ENET_MTL_DPP_CONTROL_IPEDDC(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPEDDC_SHIFT)) & ENET_MTL_DPP_CONTROL_IPEDDC_MASK)
#define ENET_MTL_DPP_CONTROL_IPEMRF_MASK                    (0x80U)
#define ENET_MTL_DPP_CONTROL_IPEMRF_SHIFT                   (7U)
#define ENET_MTL_DPP_CONTROL_IPEMRF(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPEMRF_SHIFT)) & ENET_MTL_DPP_CONTROL_IPEMRF_MASK)
#define ENET_MTL_DPP_CONTROL_IPEMTS_MASK                    (0x40U)
#define ENET_MTL_DPP_CONTROL_IPEMTS_SHIFT                   (6U)
#define ENET_MTL_DPP_CONTROL_IPEMTS(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPEMTS_SHIFT)) & ENET_MTL_DPP_CONTROL_IPEMTS_MASK)
#define ENET_MTL_DPP_CONTROL_IPEMC_MASK                     (0x20U)
#define ENET_MTL_DPP_CONTROL_IPEMC_SHIFT                    (5U)
#define ENET_MTL_DPP_CONTROL_IPEMC(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPEMC_SHIFT)) & ENET_MTL_DPP_CONTROL_IPEMC_MASK)
#define ENET_MTL_DPP_CONTROL_IPEID_MASK                     (0x10U)
#define ENET_MTL_DPP_CONTROL_IPEID_SHIFT                    (4U)
#define ENET_MTL_DPP_CONTROL_IPEID(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_IPEID_SHIFT)) & ENET_MTL_DPP_CONTROL_IPEID_MASK)
#define ENET_MTL_DPP_CONTROL_EPSI_MASK                      (0x4U)
#define ENET_MTL_DPP_CONTROL_EPSI_SHIFT                     (2U)
#define ENET_MTL_DPP_CONTROL_EPSI(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_EPSI_SHIFT)) & ENET_MTL_DPP_CONTROL_EPSI_MASK)
#define ENET_MTL_DPP_CONTROL_OPE_MASK                       (0x2U)
#define ENET_MTL_DPP_CONTROL_OPE_SHIFT                      (1U)
#define ENET_MTL_DPP_CONTROL_OPE(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_OPE_SHIFT)) & ENET_MTL_DPP_CONTROL_OPE_MASK)
#define ENET_MTL_DPP_CONTROL_EDPP_MASK                      (0x1U)
#define ENET_MTL_DPP_CONTROL_EDPP_SHIFT                     (0U)
#define ENET_MTL_DPP_CONTROL_EDPP(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_DPP_CONTROL_EDPP_SHIFT)) & ENET_MTL_DPP_CONTROL_EDPP_MASK)
/* ENET_MTL_DPP_ECC_EIC Register */
#define ENET_MTL_DPP_ECC_EIC_EIM_MASK                       (0x10000U)
#define ENET_MTL_DPP_ECC_EIC_EIM_SHIFT                      (16U)
#define ENET_MTL_DPP_ECC_EIC_EIM(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_DPP_ECC_EIC_EIM_SHIFT)) & ENET_MTL_DPP_ECC_EIC_EIM_MASK)
#define ENET_MTL_DPP_ECC_EIC_BLEI_MASK                      (0xFFU)
#define ENET_MTL_DPP_ECC_EIC_BLEI_SHIFT                     (0U)
#define ENET_MTL_DPP_ECC_EIC_BLEI(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_DPP_ECC_EIC_BLEI_SHIFT)) & ENET_MTL_DPP_ECC_EIC_BLEI_MASK)
/* ENET_MTL_QUEUE_TX_OPERATION_MODE Register */
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TQS_MASK                       (0x1F0000U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TQS_SHIFT                      (16U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TQS(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_OPERATION_MODE_TQS_SHIFT)) & ENET_MTL_QUEUE_TX_OPERATION_MODE_TQS_MASK)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TTC_MASK                       (0xC000U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TTC_SHIFT                      (14U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TTC(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_OPERATION_MODE_TTC_SHIFT)) & ENET_MTL_QUEUE_TX_OPERATION_MODE_TTC_MASK)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TXQEN_MASK                     (0xCU)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TXQEN_SHIFT                    (2U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TXQEN(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_OPERATION_MODE_TXQEN_SHIFT)) & ENET_MTL_QUEUE_TX_OPERATION_MODE_TXQEN_MASK)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TSF_MASK                       (0x2U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TSF_SHIFT                      (1U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_TSF(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_OPERATION_MODE_TSF_SHIFT)) & ENET_MTL_QUEUE_TX_OPERATION_MODE_TSF_MASK)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_FTQ_MASK                       (0x1U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_FTQ_SHIFT                      (0U)
#define ENET_MTL_QUEUE_TX_OPERATION_MODE_FTQ(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_OPERATION_MODE_FTQ_SHIFT)) & ENET_MTL_QUEUE_TX_OPERATION_MODE_FTQ_MASK)
/* ENET_MTL_QUEUE_TX_UNDERFLOW Register */
#define ENET_MTL_QUEUE_TX_UNDERFLOW_UFCNTOVF_MASK                     (0x800U)
#define ENET_MTL_QUEUE_TX_UNDERFLOW_UFCNTOVF_SHIFT                    (11U)
#define ENET_MTL_QUEUE_TX_UNDERFLOW_UFCNTOVF(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_UNDERFLOW_UFCNTOVF_SHIFT)) & ENET_MTL_QUEUE_TX_UNDERFLOW_UFCNTOVF_MASK)
#define ENET_MTL_QUEUE_TX_UNDERFLOW_UFFRMCNT_MASK                     (0x7FFU)
#define ENET_MTL_QUEUE_TX_UNDERFLOW_UFFRMCNT_SHIFT                    (0U)
#define ENET_MTL_QUEUE_TX_UNDERFLOW_UFFRMCNT(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_UNDERFLOW_UFFRMCNT_SHIFT)) & ENET_MTL_QUEUE_TX_UNDERFLOW_UFFRMCNT_MASK)
/* ENET_MTL_QUEUE_TX_DEBUG Register */
#define ENET_MTL_QUEUE_TX_DEBUG_STXSTSF_MASK                          (0x700000U)
#define ENET_MTL_QUEUE_TX_DEBUG_STXSTSF_SHIFT                         (20U)
#define ENET_MTL_QUEUE_TX_DEBUG_STXSTSF(x)                            (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_DEBUG_STXSTSF_SHIFT)) & ENET_MTL_QUEUE_TX_DEBUG_STXSTSF_MASK)
#define ENET_MTL_QUEUE_TX_DEBUG_PTXQ_MASK                             (0x70000U)
#define ENET_MTL_QUEUE_TX_DEBUG_PTXQ_SHIFT                            (16U)
#define ENET_MTL_QUEUE_TX_DEBUG_PTXQ(x)                               (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_DEBUG_PTXQ_SHIFT)) & ENET_MTL_QUEUE_TX_DEBUG_PTXQ_MASK)
#define ENET_MTL_QUEUE_TX_DEBUG_TXSTSFSTS_MASK                        (0x20U)
#define ENET_MTL_QUEUE_TX_DEBUG_TXSTSFSTS_SHIFT                       (5U)
#define ENET_MTL_QUEUE_TX_DEBUG_TXSTSFSTS(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_DEBUG_TXSTSFSTS_SHIFT)) & ENET_MTL_QUEUE_TX_DEBUG_TXSTSFSTS_MASK)
#define ENET_MTL_QUEUE_TX_DEBUG_TXQSTS_MASK                           (0x10U)
#define ENET_MTL_QUEUE_TX_DEBUG_TXQSTS_SHIFT                          (4U)
#define ENET_MTL_QUEUE_TX_DEBUG_TXQSTS(x)                             (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_DEBUG_TXQSTS_SHIFT)) & ENET_MTL_QUEUE_TX_DEBUG_TXQSTS_MASK)
#define ENET_MTL_QUEUE_TX_DEBUG_TWCSTS_MASK                           (0x8U)
#define ENET_MTL_QUEUE_TX_DEBUG_TWCSTS_SHIFT                          (3U)
#define ENET_MTL_QUEUE_TX_DEBUG_TWCSTS(x)                             (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_DEBUG_TWCSTS_SHIFT)) & ENET_MTL_QUEUE_TX_DEBUG_TWCSTS_MASK)
#define ENET_MTL_QUEUE_TX_DEBUG_TRCSTS_MASK                           (0x6U)
#define ENET_MTL_QUEUE_TX_DEBUG_TRCSTS_SHIFT                          (1U)
#define ENET_MTL_QUEUE_TX_DEBUG_TRCSTS(x)                             (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_DEBUG_TRCSTS_SHIFT)) & ENET_MTL_QUEUE_TX_DEBUG_TRCSTS_MASK)
#define ENET_MTL_QUEUE_TX_DEBUG_TXQPAUSED_MASK                        (0x1U)
#define ENET_MTL_QUEUE_TX_DEBUG_TXQPAUSED_SHIFT                       (0U)
#define ENET_MTL_QUEUE_TX_DEBUG_TXQPAUSED(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_DEBUG_TXQPAUSED_SHIFT)) & ENET_MTL_QUEUE_TX_DEBUG_TXQPAUSED_MASK)
/* ENET_MTL_QUEUE_TX_ETS_CONTROL Register */
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_SLC_MASK                        (0x70U)
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_SLC_SHIFT                       (4U)
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_SLC(x)                          (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_ETS_CONTROL_SLC_SHIFT)) & ENET_MTL_QUEUE_TX_ETS_CONTROL_SLC_MASK)
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_CC_MASK                         (0x8U)
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_CC_SHIFT                        (3U)
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_CC(x)                           (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_ETS_CONTROL_CC_SHIFT)) & ENET_MTL_QUEUE_TX_ETS_CONTROL_CC_MASK)
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_AVALG_MASK                      (0x4U)
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_AVALG_SHIFT                     (2U)
#define ENET_MTL_QUEUE_TX_ETS_CONTROL_AVALG(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_ETS_CONTROL_AVALG_SHIFT)) & ENET_MTL_QUEUE_TX_ETS_CONTROL_AVALG_MASK)
/* ENET_MTL_QUEUE_TX_ETS_STATUS Register */
#define ENET_MTL_QUEUE_TX_ETS_STATUS_ABS_MASK                         (0xFFFFFFU)
#define ENET_MTL_QUEUE_TX_ETS_STATUS_ABS_SHIFT                        (0U)
#define ENET_MTL_QUEUE_TX_ETS_STATUS_ABS(x)                           (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_ETS_STATUS_ABS_SHIFT)) & ENET_MTL_QUEUE_TX_ETS_STATUS_ABS_MASK)
/* ENET_MTL_QUEUE_TX_QUANTUM_WEIGHT Register */
#define ENET_MTL_QUEUE_TX_QUANTUM_WEIGHT_ISCQW_MASK                     (0x1FFFFFU)
#define ENET_MTL_QUEUE_TX_QUANTUM_WEIGHT_ISCQW_SHIFT                    (0U)
#define ENET_MTL_QUEUE_TX_QUANTUM_WEIGHT_ISCQW(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_TX_QUANTUM_WEIGHT_ISCQW_SHIFT)) & ENET_MTL_QUEUE_TX_QUANTUM_WEIGHT_ISCQW_MASK)
/* ENET_MTL_QUEUE_SENDSLOPECREDIT Register */
#define ENET_MTL_QUEUE_SENDSLOPECREDIT_SSC_MASK                       (0x3FFFU)
#define ENET_MTL_QUEUE_SENDSLOPECREDIT_SSC_SHIFT                      (0U)
#define ENET_MTL_QUEUE_SENDSLOPECREDIT_SSC(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_SENDSLOPECREDIT_SSC_SHIFT)) & ENET_MTL_QUEUE_SENDSLOPECREDIT_SSC_MASK)
/* ENET_MTL_QUEUE_HICREDIT Register */
#define ENET_MTL_QUEUE_HICREDIT_HC_MASK                               (0x1FFFFFFFU)
#define ENET_MTL_QUEUE_HICREDIT_HC_SHIFT                              (0U)
#define ENET_MTL_QUEUE_HICREDIT_HC(x)                                 (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_HICREDIT_HC_SHIFT)) & ENET_MTL_QUEUE_HICREDIT_HC_MASK)
/* ENET_MTL_QUEUE_LOCREDIT Register */
#define ENET_MTL_QUEUE_LOCREDIT_LC_MASK                               (0x1FFFFFFFU)
#define ENET_MTL_QUEUE_LOCREDIT_LC_SHIFT                              (0U)
#define ENET_MTL_QUEUE_LOCREDIT_LC(x)                                 (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_LOCREDIT_LC_SHIFT)) & ENET_MTL_QUEUE_LOCREDIT_LC_MASK)
/* ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS Register */
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOIE_MASK                     (0x1000000U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOIE_SHIFT                    (24U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOIE(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOIE_SHIFT)) & ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOIE_MASK)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOVFIS_MASK                   (0x10000U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOVFIS_SHIFT                  (16U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOVFIS(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOVFIS_SHIFT)) & ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_RXOVFIS_MASK)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIE_MASK                    (0x200U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIE_SHIFT                   (9U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIE(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIE_SHIFT)) & ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIE_MASK)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUIE_MASK                     (0x100U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUIE_SHIFT                    (8U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUIE(x)                       (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUIE_SHIFT)) & ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUIE_MASK)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIS_MASK                    (0x2U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIS_SHIFT                   (1U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIS(x)                      (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIS_SHIFT)) & ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_ABPSIS_MASK)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUNFIS_MASK                   (0x1U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUNFIS_SHIFT                  (0U)
#define ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUNFIS(x)                     (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUNFIS_SHIFT)) & ENET_MTL_QUEUE_INTERRUPT_CONTROL_STATUS_TXUNFIS_MASK)
/* ENET_MTL_QUEUE_RX_OPERATION_MODE Register */
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RQS_MASK                       (0x1F00000U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RQS_SHIFT                      (20U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RQS(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_RQS_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_RQS_MASK)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RFD_MASK                       (0x3C000U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RFD_SHIFT                      (14U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RFD(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_RFD_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_RFD_MASK)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RFA_MASK                       (0xF00U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RFA_SHIFT                      (8U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RFA(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_RFA_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_RFA_MASK)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_EHFC_MASK                      (0x80U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_EHFC_SHIFT                     (7U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_EHFC(x)                        (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_EHFC_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_EHFC_MASK)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_DIS_TCP_EF_MASK                (0x40U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_DIS_TCP_EF_SHIFT               (6U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_DIS_TCP_EF(x)                  (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_DIS_TCP_EF_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_DIS_TCP_EF_MASK)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RSF_MASK                       (0x20U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RSF_SHIFT                      (5U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RSF(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_RSF_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_RSF_MASK)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_FEP_MASK                       (0x10U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_FEP_SHIFT                      (4U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_FEP(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_FEP_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_FEP_MASK)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_FUP_MASK                       (0x8U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_FUP_SHIFT                      (3U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_FUP(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_FUP_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_FUP_MASK)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RTC_MASK                       (0x3U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RTC_SHIFT                      (0U)
#define ENET_MTL_QUEUE_RX_OPERATION_MODE_RTC(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_OPERATION_MODE_RTC_SHIFT)) & ENET_MTL_QUEUE_RX_OPERATION_MODE_RTC_MASK)
/* ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT Register */
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISCNTOVF_MASK                 (0x8000000U)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISCNTOVF_SHIFT                (27U)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISCNTOVF(x)                   (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISCNTOVF_SHIFT)) & ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISCNTOVF_MASK)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISPKTCNT_MASK                 (0x7FF0000U)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISPKTCNT_SHIFT                (16U)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISPKTCNT(x)                   (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISPKTCNT_SHIFT)) & ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_MISPKTCNT_MASK)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFCNTOVF_MASK                 (0x800U)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFCNTOVF_SHIFT                (11U)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFCNTOVF(x)                   (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFCNTOVF_SHIFT)) & ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFCNTOVF_MASK)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFPKTCNT_MASK                 (0x7FFU)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFPKTCNT_SHIFT                (0U)
#define ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFPKTCNT(x)                   (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFPKTCNT_SHIFT)) & ENET_MTL_QUEUE_RX_MISSED_PACKET_OVERFLOW_CNT_OVFPKTCNT_MASK)
/* ENET_MTL_QUEUE_RX_DEBUG Register */
#define ENET_MTL_QUEUE_RX_DEBUG_PRXQ_MASK                             (0x3FFF0000U)
#define ENET_MTL_QUEUE_RX_DEBUG_PRXQ_SHIFT                            (16U)
#define ENET_MTL_QUEUE_RX_DEBUG_PRXQ(x)                               (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_DEBUG_PRXQ_SHIFT)) & ENET_MTL_QUEUE_RX_DEBUG_PRXQ_MASK)
#define ENET_MTL_QUEUE_RX_DEBUG_RXQSTS_MASK                           (0x30U)
#define ENET_MTL_QUEUE_RX_DEBUG_RXQSTS_SHIFT                          (4U)
#define ENET_MTL_QUEUE_RX_DEBUG_RXQSTS(x)                             (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_DEBUG_RXQSTS_SHIFT)) & ENET_MTL_QUEUE_RX_DEBUG_RXQSTS_MASK)
#define ENET_MTL_QUEUE_RX_DEBUG_RRCSTS_MASK                           (0x6U)
#define ENET_MTL_QUEUE_RX_DEBUG_RRCSTS_SHIFT                          (1U)
#define ENET_MTL_QUEUE_RX_DEBUG_RRCSTS(x)                             (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_DEBUG_RRCSTS_SHIFT)) & ENET_MTL_QUEUE_RX_DEBUG_RRCSTS_MASK)
#define ENET_MTL_QUEUE_RX_DEBUG_RWCSTS_MASK                           (0x1U)
#define ENET_MTL_QUEUE_RX_DEBUG_RWCSTS_SHIFT                          (0U)
#define ENET_MTL_QUEUE_RX_DEBUG_RWCSTS(x)                             (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_DEBUG_RWCSTS_SHIFT)) & ENET_MTL_QUEUE_RX_DEBUG_RWCSTS_MASK)
/* ENET_MTL_QUEUE_RX_CONTROL Register */
#define ENET_MTL_QUEUE_RX_CONTROL_RXQ_FRM_ARBIT_MASK                  (0x8U)
#define ENET_MTL_QUEUE_RX_CONTROL_RXQ_FRM_ARBIT_SHIFT                 (3U)
#define ENET_MTL_QUEUE_RX_CONTROL_RXQ_FRM_ARBIT(x)                    (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_CONTROL_RXQ_FRM_ARBIT_SHIFT)) & ENET_MTL_QUEUE_RX_CONTROL_RXQ_FRM_ARBIT_MASK)
#define ENET_MTL_QUEUE_RX_CONTROL_RXQ_WEGT_MASK                       (0x7U)
#define ENET_MTL_QUEUE_RX_CONTROL_RXQ_WEGT_SHIFT                      (0U)
#define ENET_MTL_QUEUE_RX_CONTROL_RXQ_WEGT(x)                         (((uint32)(((uint32)(x)) << ENET_MTL_QUEUE_RX_CONTROL_RXQ_WEGT_SHIFT)) & ENET_MTL_QUEUE_RX_CONTROL_RXQ_WEGT_MASK)

/* ENET_DMA_MODE Register */
#define ENET_DMA_MODE_INTM_MASK                             (0x30000U)
#define ENET_DMA_MODE_INTM_SHIFT                            (16U)
#define ENET_DMA_MODE_INTM(x)                               (((uint32)(((uint32)(x)) << ENET_DMA_MODE_INTM_SHIFT)) & ENET_DMA_MODE_INTM_MASK)
#define ENET_DMA_MODE_PR_MASK                               (0x7000U)
#define ENET_DMA_MODE_PR_SHIFT                              (12U)
#define ENET_DMA_MODE_PR(x)                                 (((uint32)(((uint32)(x)) << ENET_DMA_MODE_PR_SHIFT)) & ENET_DMA_MODE_PR_MASK)
#define ENET_DMA_MODE_TXPR_MASK                             (0x800U)
#define ENET_DMA_MODE_TXPR_SHIFT                            (11U)
#define ENET_DMA_MODE_TXPR(x)                               (((uint32)(((uint32)(x)) << ENET_DMA_MODE_TXPR_SHIFT)) & ENET_DMA_MODE_TXPR_MASK)
#define ENET_DMA_MODE_DSPW_MASK                             (0x100U)
#define ENET_DMA_MODE_DSPW_SHIFT                            (8U)
#define ENET_DMA_MODE_DSPW(x)                               (((uint32)(((uint32)(x)) << ENET_DMA_MODE_DSPW_SHIFT)) & ENET_DMA_MODE_DSPW_MASK)
#define ENET_DMA_MODE_TAA_MASK                              (0x1CU)
#define ENET_DMA_MODE_TAA_SHIFT                             (2U)
#define ENET_DMA_MODE_TAA(x)                                (((uint32)(((uint32)(x)) << ENET_DMA_MODE_TAA_SHIFT)) & ENET_DMA_MODE_TAA_MASK)
#define ENET_DMA_MODE_DA_MASK                               (0x2U)
#define ENET_DMA_MODE_DA_SHIFT                              (1U)
#define ENET_DMA_MODE_DA(x)                                 (((uint32)(((uint32)(x)) << ENET_DMA_MODE_DA_SHIFT)) & ENET_DMA_MODE_DA_MASK)
#define ENET_DMA_MODE_SWR_MASK                              (0x1U)
#define ENET_DMA_MODE_SWR_SHIFT                             (0U)
#define ENET_DMA_MODE_SWR(x)                                (((uint32)(((uint32)(x)) << ENET_DMA_MODE_SWR_SHIFT)) & ENET_DMA_MODE_SWR_MASK)
/* ENET_DMA_SYSBUS_MODE Register */
#define ENET_DMA_SYSBUS_MODE_EN_LPI_MASK                    (0x80000000U)
#define ENET_DMA_SYSBUS_MODE_EN_LPI_SHIFT                   (31U)
#define ENET_DMA_SYSBUS_MODE_EN_LPI(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_EN_LPI_SHIFT)) & ENET_DMA_SYSBUS_MODE_EN_LPI_MASK)
#define ENET_DMA_SYSBUS_MODE_LPI_XIT_PKT_MASK               (0x40000000U)
#define ENET_DMA_SYSBUS_MODE_LPI_XIT_PKT_SHIFT              (30U)
#define ENET_DMA_SYSBUS_MODE_LPI_XIT_PKT(x)                 (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_LPI_XIT_PKT_SHIFT)) & ENET_DMA_SYSBUS_MODE_LPI_XIT_PKT_MASK)
#define ENET_DMA_SYSBUS_MODE_WR_OSR_LMT_MASK                (0x3800000U)
#define ENET_DMA_SYSBUS_MODE_WR_OSR_LMT_SHIFT               (23U)
#define ENET_DMA_SYSBUS_MODE_WR_OSR_LMT(x)                  (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_WR_OSR_LMT_SHIFT)) & ENET_DMA_SYSBUS_MODE_WR_OSR_LMT_MASK)
#define ENET_DMA_SYSBUS_MODE_RD_OSR_LMT_MASK                (0x30000U)
#define ENET_DMA_SYSBUS_MODE_RD_OSR_LMT_SHIFT               (16U)
#define ENET_DMA_SYSBUS_MODE_RD_OSR_LMT(x)                  (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_RD_OSR_LMT_SHIFT)) & ENET_DMA_SYSBUS_MODE_RD_OSR_LMT_MASK)
#define ENET_DMA_SYSBUS_MODE_RB_MASK                        (0x8000U)
#define ENET_DMA_SYSBUS_MODE_RB_SHIFT                       (15U)
#define ENET_DMA_SYSBUS_MODE_RB(x)                          (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_RB_SHIFT)) & ENET_DMA_SYSBUS_MODE_RB_MASK)
#define ENET_DMA_SYSBUS_MODE_MB_MASK                        (0x4000U)
#define ENET_DMA_SYSBUS_MODE_MB_SHIFT                       (14U)
#define ENET_DMA_SYSBUS_MODE_MB(x)                          (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_MB_SHIFT)) & ENET_DMA_SYSBUS_MODE_MB_MASK)
#define ENET_DMA_SYSBUS_MODE_ONEKBBE_MASK                   (0x2000U)
#define ENET_DMA_SYSBUS_MODE_ONEKBBE_SHIFT                  (13U)
#define ENET_DMA_SYSBUS_MODE_ONEKBBE(x)                     (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_ONEKBBE_SHIFT)) & ENET_DMA_SYSBUS_MODE_ONEKBBE_MASK)
#define ENET_DMA_SYSBUS_MODE_AAL_MASK                       (0x1000U)
#define ENET_DMA_SYSBUS_MODE_AAL_SHIFT                      (12U)
#define ENET_DMA_SYSBUS_MODE_AAL(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_AAL_SHIFT)) & ENET_DMA_SYSBUS_MODE_AAL_MASK)
#define ENET_DMA_SYSBUS_MODE_EAME_MASK                      (0x800U)
#define ENET_DMA_SYSBUS_MODE_EAME_SHIFT                     (11U)
#define ENET_DMA_SYSBUS_MODE_EAME(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_EAME_SHIFT)) & ENET_DMA_SYSBUS_MODE_EAME_MASK)
#define ENET_DMA_SYSBUS_MODE_AALE_MASK                      (0x400U)
#define ENET_DMA_SYSBUS_MODE_AALE_SHIFT                     (10U)
#define ENET_DMA_SYSBUS_MODE_AALE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_AALE_SHIFT)) & ENET_DMA_SYSBUS_MODE_AALE_MASK)
#define ENET_DMA_SYSBUS_MODE_BLEN256_MASK                   (0x80U)
#define ENET_DMA_SYSBUS_MODE_BLEN256_SHIFT                  (7U)
#define ENET_DMA_SYSBUS_MODE_BLEN256(x)                     (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_BLEN256_SHIFT)) & ENET_DMA_SYSBUS_MODE_BLEN256_MASK)
#define ENET_DMA_SYSBUS_MODE_BLEN128_MASK                   (0x40U)
#define ENET_DMA_SYSBUS_MODE_BLEN128_SHIFT                  (6U)
#define ENET_DMA_SYSBUS_MODE_BLEN128(x)                     (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_BLEN128_SHIFT)) & ENET_DMA_SYSBUS_MODE_BLEN128_MASK)
#define ENET_DMA_SYSBUS_MODE_BLEN64_MASK                    (0x20U)
#define ENET_DMA_SYSBUS_MODE_BLEN64_SHIFT                   (5U)
#define ENET_DMA_SYSBUS_MODE_BLEN64(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_BLEN64_SHIFT)) & ENET_DMA_SYSBUS_MODE_BLEN64_MASK)
#define ENET_DMA_SYSBUS_MODE_BLEN32_MASK                    (0x10U)
#define ENET_DMA_SYSBUS_MODE_BLEN32_SHIFT                   (4U)
#define ENET_DMA_SYSBUS_MODE_BLEN32(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_BLEN32_SHIFT)) & ENET_DMA_SYSBUS_MODE_BLEN32_MASK)
#define ENET_DMA_SYSBUS_MODE_BLEN16_MASK                    (0x8U)
#define ENET_DMA_SYSBUS_MODE_BLEN16_SHIFT                   (3U)
#define ENET_DMA_SYSBUS_MODE_BLEN16(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_BLEN16_SHIFT)) & ENET_DMA_SYSBUS_MODE_BLEN16_MASK)
#define ENET_DMA_SYSBUS_MODE_BLEN8_MASK                     (0x4U)
#define ENET_DMA_SYSBUS_MODE_BLEN8_SHIFT                    (2U)
#define ENET_DMA_SYSBUS_MODE_BLEN8(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_BLEN8_SHIFT)) & ENET_DMA_SYSBUS_MODE_BLEN8_MASK)
#define ENET_DMA_SYSBUS_MODE_BLEN4_MASK                     (0x2U)
#define ENET_DMA_SYSBUS_MODE_BLEN4_SHIFT                    (1U)
#define ENET_DMA_SYSBUS_MODE_BLEN4(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_BLEN4_SHIFT)) & ENET_DMA_SYSBUS_MODE_BLEN4_MASK)
#define ENET_DMA_SYSBUS_MODE_FB_MASK                        (0x1U)
#define ENET_DMA_SYSBUS_MODE_FB_SHIFT                       (0U)
#define ENET_DMA_SYSBUS_MODE_FB(x)                          (((uint32)(((uint32)(x)) << ENET_DMA_SYSBUS_MODE_FB_SHIFT)) & ENET_DMA_SYSBUS_MODE_FB_MASK)
/* ENET_DMA_INTERRUPT_STATUS Register */
#define ENET_DMA_INTERRUPT_STATUS_MACIS_MASK                     (0x20000U)
#define ENET_DMA_INTERRUPT_STATUS_MACIS_SHIFT                    (17U)
#define ENET_DMA_INTERRUPT_STATUS_MACIS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_MACIS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_MACIS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_MTLIS_MASK                     (0x10000U)
#define ENET_DMA_INTERRUPT_STATUS_MTLIS_SHIFT                    (16U)
#define ENET_DMA_INTERRUPT_STATUS_MTLIS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_MTLIS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_MTLIS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_DC7IS_MASK                     (0x80U)
#define ENET_DMA_INTERRUPT_STATUS_DC7IS_SHIFT                    (7U)
#define ENET_DMA_INTERRUPT_STATUS_DC7IS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_DC7IS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_DC7IS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_DC6IS_MASK                     (0x40U)
#define ENET_DMA_INTERRUPT_STATUS_DC6IS_SHIFT                    (6U)
#define ENET_DMA_INTERRUPT_STATUS_DC6IS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_DC6IS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_DC6IS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_DC5IS_MASK                     (0x20U)
#define ENET_DMA_INTERRUPT_STATUS_DC5IS_SHIFT                    (5U)
#define ENET_DMA_INTERRUPT_STATUS_DC5IS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_DC5IS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_DC5IS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_DC4IS_MASK                     (0x10U)
#define ENET_DMA_INTERRUPT_STATUS_DC4IS_SHIFT                    (4U)
#define ENET_DMA_INTERRUPT_STATUS_DC4IS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_DC4IS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_DC4IS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_DC3IS_MASK                     (0x8U)
#define ENET_DMA_INTERRUPT_STATUS_DC3IS_SHIFT                    (3U)
#define ENET_DMA_INTERRUPT_STATUS_DC3IS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_DC3IS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_DC3IS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_DC2IS_MASK                     (0x4U)
#define ENET_DMA_INTERRUPT_STATUS_DC2IS_SHIFT                    (2U)
#define ENET_DMA_INTERRUPT_STATUS_DC2IS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_DC2IS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_DC2IS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_DC1IS_MASK                     (0x2U)
#define ENET_DMA_INTERRUPT_STATUS_DC1IS_SHIFT                    (1U)
#define ENET_DMA_INTERRUPT_STATUS_DC1IS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_DC1IS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_DC1IS_MASK)
#define ENET_DMA_INTERRUPT_STATUS_DC0IS_MASK                     (0x1U)
#define ENET_DMA_INTERRUPT_STATUS_DC0IS_SHIFT                    (0U)
#define ENET_DMA_INTERRUPT_STATUS_DC0IS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_INTERRUPT_STATUS_DC0IS_SHIFT)) & ENET_DMA_INTERRUPT_STATUS_DC0IS_MASK)
/* ENET_DMA_DEBUG_STATUS0 Register */
#define ENET_DMA_DEBUG_STATUS0_TPS2_MASK                      (0xF0000000U)
#define ENET_DMA_DEBUG_STATUS0_TPS2_SHIFT                     (28U)
#define ENET_DMA_DEBUG_STATUS0_TPS2(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS0_TPS2_SHIFT)) & ENET_DMA_DEBUG_STATUS0_TPS2_MASK)
#define ENET_DMA_DEBUG_STATUS0_RPS2_MASK                      (0xF000000U)
#define ENET_DMA_DEBUG_STATUS0_RPS2_SHIFT                     (24U)
#define ENET_DMA_DEBUG_STATUS0_RPS2(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS0_RPS2_SHIFT)) & ENET_DMA_DEBUG_STATUS0_RPS2_MASK)
#define ENET_DMA_DEBUG_STATUS0_TPS1_MASK                      (0xF00000U)
#define ENET_DMA_DEBUG_STATUS0_TPS1_SHIFT                     (20U)
#define ENET_DMA_DEBUG_STATUS0_TPS1(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS0_TPS1_SHIFT)) & ENET_DMA_DEBUG_STATUS0_TPS1_MASK)
#define ENET_DMA_DEBUG_STATUS0_RPS1_MASK                      (0xF0000U)
#define ENET_DMA_DEBUG_STATUS0_RPS1_SHIFT                     (16U)
#define ENET_DMA_DEBUG_STATUS0_RPS1(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS0_RPS1_SHIFT)) & ENET_DMA_DEBUG_STATUS0_RPS1_MASK)
#define ENET_DMA_DEBUG_STATUS0_TPS0_MASK                      (0xF000U)
#define ENET_DMA_DEBUG_STATUS0_TPS0_SHIFT                     (12U)
#define ENET_DMA_DEBUG_STATUS0_TPS0(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS0_TPS0_SHIFT)) & ENET_DMA_DEBUG_STATUS0_TPS0_MASK)
#define ENET_DMA_DEBUG_STATUS0_RPS0_MASK                      (0xF00U)
#define ENET_DMA_DEBUG_STATUS0_RPS0_SHIFT                     (8U)
#define ENET_DMA_DEBUG_STATUS0_RPS0(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS0_RPS0_SHIFT)) & ENET_DMA_DEBUG_STATUS0_RPS0_MASK)
#define ENET_DMA_DEBUG_STATUS0_AXRHSTS_MASK                   (0x2U)
#define ENET_DMA_DEBUG_STATUS0_AXRHSTS_SHIFT                  (1U)
#define ENET_DMA_DEBUG_STATUS0_AXRHSTS(x)                     (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS0_AXRHSTS_SHIFT)) & ENET_DMA_DEBUG_STATUS0_AXRHSTS_MASK)
#define ENET_DMA_DEBUG_STATUS0_AXWHSTS_MASK                   (0x1U)
#define ENET_DMA_DEBUG_STATUS0_AXWHSTS_SHIFT                  (0U)
#define ENET_DMA_DEBUG_STATUS0_AXWHSTS(x)                     (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS0_AXWHSTS_SHIFT)) & ENET_DMA_DEBUG_STATUS0_AXWHSTS_MASK)
/* ENET_DMA_DEBUG_STATUS1 Register */
#define ENET_DMA_DEBUG_STATUS1_TPS6_MASK                      (0xF0000000U)
#define ENET_DMA_DEBUG_STATUS1_TPS6_SHIFT                     (28U)
#define ENET_DMA_DEBUG_STATUS1_TPS6(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS1_TPS6_SHIFT)) & ENET_DMA_DEBUG_STATUS1_TPS6_MASK)
#define ENET_DMA_DEBUG_STATUS1_RPS6_MASK                      (0xF000000U)
#define ENET_DMA_DEBUG_STATUS1_RPS6_SHIFT                     (24U)
#define ENET_DMA_DEBUG_STATUS1_RPS6(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS1_RPS6_SHIFT)) & ENET_DMA_DEBUG_STATUS1_RPS6_MASK)
#define ENET_DMA_DEBUG_STATUS1_TPS5_MASK                      (0xF00000U)
#define ENET_DMA_DEBUG_STATUS1_TPS5_SHIFT                     (20U)
#define ENET_DMA_DEBUG_STATUS1_TPS5(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS1_TPS5_SHIFT)) & ENET_DMA_DEBUG_STATUS1_TPS5_MASK)
#define ENET_DMA_DEBUG_STATUS1_RPS5_MASK                      (0xF0000U)
#define ENET_DMA_DEBUG_STATUS1_RPS5_SHIFT                     (16U)
#define ENET_DMA_DEBUG_STATUS1_RPS5(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS1_RPS5_SHIFT)) & ENET_DMA_DEBUG_STATUS1_RPS5_MASK)
#define ENET_DMA_DEBUG_STATUS1_TPS4_MASK                      (0xF000U)
#define ENET_DMA_DEBUG_STATUS1_TPS4_SHIFT                     (12U)
#define ENET_DMA_DEBUG_STATUS1_TPS4(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS1_TPS4_SHIFT)) & ENET_DMA_DEBUG_STATUS1_TPS4_MASK)
#define ENET_DMA_DEBUG_STATUS1_RPS4_MASK                      (0xF00U)
#define ENET_DMA_DEBUG_STATUS1_RPS4_SHIFT                     (8U)
#define ENET_DMA_DEBUG_STATUS1_RPS4(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS1_RPS4_SHIFT)) & ENET_DMA_DEBUG_STATUS1_RPS4_MASK)
#define ENET_DMA_DEBUG_STATUS1_TPS3_MASK                      (0xF0U)
#define ENET_DMA_DEBUG_STATUS1_TPS3_SHIFT                     (4U)
#define ENET_DMA_DEBUG_STATUS1_TPS3(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS1_TPS3_SHIFT)) & ENET_DMA_DEBUG_STATUS1_TPS3_MASK)
#define ENET_DMA_DEBUG_STATUS1_RPS3_MASK                      (0xFU)
#define ENET_DMA_DEBUG_STATUS1_RPS3_SHIFT                     (0U)
#define ENET_DMA_DEBUG_STATUS1_RPS3(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS1_RPS3_SHIFT)) & ENET_DMA_DEBUG_STATUS1_RPS3_MASK)
/* ENET_DMA_DEBUG_STATUS2 Register */
#define ENET_DMA_DEBUG_STATUS2_TPS7_MASK                      (0xF0U)
#define ENET_DMA_DEBUG_STATUS2_TPS7_SHIFT                     (4U)
#define ENET_DMA_DEBUG_STATUS2_TPS7(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS2_TPS7_SHIFT)) & ENET_DMA_DEBUG_STATUS2_TPS7_MASK)
#define ENET_DMA_DEBUG_STATUS2_RPS7_MASK                      (0xFU)
#define ENET_DMA_DEBUG_STATUS2_RPS7_SHIFT                     (0U)
#define ENET_DMA_DEBUG_STATUS2_RPS7(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_DEBUG_STATUS2_RPS7_SHIFT)) & ENET_DMA_DEBUG_STATUS2_RPS7_MASK)
/* ENET_AXI4_TX_AR_ACE_CONTROL Register */
#define ENET_AXI4_TX_AR_ACE_CONTROL_THD_MASK                       (0x300000U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_THD_SHIFT                      (20U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_THD(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_TX_AR_ACE_CONTROL_THD_SHIFT)) & ENET_AXI4_TX_AR_ACE_CONTROL_THD_MASK)
#define ENET_AXI4_TX_AR_ACE_CONTROL_THC_MASK                       (0xF0000U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_THC_SHIFT                      (16U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_THC(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_TX_AR_ACE_CONTROL_THC_SHIFT)) & ENET_AXI4_TX_AR_ACE_CONTROL_THC_MASK)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TED_MASK                       (0x3000U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TED_SHIFT                      (12U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TED(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_TX_AR_ACE_CONTROL_TED_SHIFT)) & ENET_AXI4_TX_AR_ACE_CONTROL_TED_MASK)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TEC_MASK                       (0xF00U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TEC_SHIFT                      (8U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TEC(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_TX_AR_ACE_CONTROL_TEC_SHIFT)) & ENET_AXI4_TX_AR_ACE_CONTROL_TEC_MASK)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TDRD_MASK                      (0x30U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TDRD_SHIFT                     (4U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TDRD(x)                        (((uint32)(((uint32)(x)) << ENET_AXI4_TX_AR_ACE_CONTROL_TDRD_SHIFT)) & ENET_AXI4_TX_AR_ACE_CONTROL_TDRD_MASK)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TDRC_MASK                      (0xFU)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TDRC_SHIFT                     (0U)
#define ENET_AXI4_TX_AR_ACE_CONTROL_TDRC(x)                        (((uint32)(((uint32)(x)) << ENET_AXI4_TX_AR_ACE_CONTROL_TDRC_SHIFT)) & ENET_AXI4_TX_AR_ACE_CONTROL_TDRC_MASK)
/* ENET_AXI4_RX_AW_ACE_CONTROL Register */
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDD_MASK                       (0x30000000U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDD_SHIFT                      (28U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDD(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_RX_AW_ACE_CONTROL_RDD_SHIFT)) & ENET_AXI4_RX_AW_ACE_CONTROL_RDD_MASK)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDC_MASK                       (0xF000000U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDC_SHIFT                      (24U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDC(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_RX_AW_ACE_CONTROL_RDC_SHIFT)) & ENET_AXI4_RX_AW_ACE_CONTROL_RDC_MASK)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RHD_MASK                       (0x300000U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RHD_SHIFT                      (20U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RHD(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_RX_AW_ACE_CONTROL_RHD_SHIFT)) & ENET_AXI4_RX_AW_ACE_CONTROL_RHD_MASK)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RHC_MASK                       (0xF0000U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RHC_SHIFT                      (16U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RHC(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_RX_AW_ACE_CONTROL_RHC_SHIFT)) & ENET_AXI4_RX_AW_ACE_CONTROL_RHC_MASK)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RPD_MASK                       (0x3000U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RPD_SHIFT                      (12U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RPD(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_RX_AW_ACE_CONTROL_RPD_SHIFT)) & ENET_AXI4_RX_AW_ACE_CONTROL_RPD_MASK)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RPC_MASK                       (0xF00U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RPC_SHIFT                      (8U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RPC(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_RX_AW_ACE_CONTROL_RPC_SHIFT)) & ENET_AXI4_RX_AW_ACE_CONTROL_RPC_MASK)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDWD_MASK                      (0x30U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDWD_SHIFT                     (4U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDWD(x)                        (((uint32)(((uint32)(x)) << ENET_AXI4_RX_AW_ACE_CONTROL_RDWD_SHIFT)) & ENET_AXI4_RX_AW_ACE_CONTROL_RDWD_MASK)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDWC_MASK                      (0xFU)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDWC_SHIFT                     (0U)
#define ENET_AXI4_RX_AW_ACE_CONTROL_RDWC(x)                        (((uint32)(((uint32)(x)) << ENET_AXI4_RX_AW_ACE_CONTROL_RDWC_SHIFT)) & ENET_AXI4_RX_AW_ACE_CONTROL_RDWC_MASK)
/* ENET_AXI4_TXRX_AWAR_ACE_CONTROL Register */
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_WRP_MASK                       (0x700000U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_WRP_SHIFT                      (20U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_WRP(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_TXRX_AWAR_ACE_CONTROL_WRP_SHIFT)) & ENET_AXI4_TXRX_AWAR_ACE_CONTROL_WRP_MASK)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDP_MASK                       (0x70000U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDP_SHIFT                      (16U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDP(x)                         (((uint32)(((uint32)(x)) << ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDP_SHIFT)) & ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDP_MASK)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRD_MASK                      (0x3000U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRD_SHIFT                     (12U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRD(x)                        (((uint32)(((uint32)(x)) << ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRD_SHIFT)) & ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRD_MASK)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRC_MASK                      (0xF00U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRC_SHIFT                     (8U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRC(x)                        (((uint32)(((uint32)(x)) << ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRC_SHIFT)) & ENET_AXI4_TXRX_AWAR_ACE_CONTROL_RDRC_MASK)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWD_MASK                      (0x30U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWD_SHIFT                     (4U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWD(x)                        (((uint32)(((uint32)(x)) << ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWD_SHIFT)) & ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWD_MASK)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWC_MASK                      (0xFU)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWC_SHIFT                     (0U)
#define ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWC(x)                        (((uint32)(((uint32)(x)) << ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWC_SHIFT)) & ENET_AXI4_TXRX_AWAR_ACE_CONTROL_TDWC_MASK)
/* ENET_AXI_LPI_ENTRY_INTERVAL Register */
#define ENET_AXI_LPI_ENTRY_INTERVAL_LPIEI_MASK                     (0xFU)
#define ENET_AXI_LPI_ENTRY_INTERVAL_LPIEI_SHIFT                    (0U)
#define ENET_AXI_LPI_ENTRY_INTERVAL_LPIEI(x)                       (((uint32)(((uint32)(x)) << ENET_AXI_LPI_ENTRY_INTERVAL_LPIEI_SHIFT)) & ENET_AXI_LPI_ENTRY_INTERVAL_LPIEI_MASK)
/* ENET_DMA_TBS_CTRL Register */
#define ENET_DMA_TBS_CTRL_FTOS_MASK                         (0xFFFFFF00U)
#define ENET_DMA_TBS_CTRL_FTOS_SHIFT                        (8U)
#define ENET_DMA_TBS_CTRL_FTOS(x)                           (((uint32)(((uint32)(x)) << ENET_DMA_TBS_CTRL_FTOS_SHIFT)) & ENET_DMA_TBS_CTRL_FTOS_MASK)
#define ENET_DMA_TBS_CTRL_FGOS_MASK                         (0x70U)
#define ENET_DMA_TBS_CTRL_FGOS_SHIFT                        (4U)
#define ENET_DMA_TBS_CTRL_FGOS(x)                           (((uint32)(((uint32)(x)) << ENET_DMA_TBS_CTRL_FGOS_SHIFT)) & ENET_DMA_TBS_CTRL_FGOS_MASK)
#define ENET_DMA_TBS_CTRL_FTOV_MASK                         (0x1U)
#define ENET_DMA_TBS_CTRL_FTOV_SHIFT                        (0U)
#define ENET_DMA_TBS_CTRL_FTOV(x)                           (((uint32)(((uint32)(x)) << ENET_DMA_TBS_CTRL_FTOV_SHIFT)) & ENET_DMA_TBS_CTRL_FTOV_MASK)
/* ENET_DMA_SAFETY_INTERRUPT_STATUS Register */
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MCSIS_MASK                     (0x80000000U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MCSIS_SHIFT                    (31U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MCSIS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_SAFETY_INTERRUPT_STATUS_MCSIS_SHIFT)) & ENET_DMA_SAFETY_INTERRUPT_STATUS_MCSIS_MASK)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MSUIS_MASK                     (0x20000000U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MSUIS_SHIFT                    (29U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MSUIS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_SAFETY_INTERRUPT_STATUS_MSUIS_SHIFT)) & ENET_DMA_SAFETY_INTERRUPT_STATUS_MSUIS_MASK)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MSCIS_MASK                     (0x10000000U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MSCIS_SHIFT                    (28U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_MSCIS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_SAFETY_INTERRUPT_STATUS_MSCIS_SHIFT)) & ENET_DMA_SAFETY_INTERRUPT_STATUS_MSCIS_MASK)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_DEUIS_MASK                     (0x2U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_DEUIS_SHIFT                    (1U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_DEUIS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_SAFETY_INTERRUPT_STATUS_DEUIS_SHIFT)) & ENET_DMA_SAFETY_INTERRUPT_STATUS_DEUIS_MASK)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_DECIS_MASK                     (0x1U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_DECIS_SHIFT                    (0U)
#define ENET_DMA_SAFETY_INTERRUPT_STATUS_DECIS(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_SAFETY_INTERRUPT_STATUS_DECIS_SHIFT)) & ENET_DMA_SAFETY_INTERRUPT_STATUS_DECIS_MASK)
/* ENET_DMA_ECC_INTERRUPT_ENABLE Register */
#define ENET_DMA_ECC_INTERRUPT_ENABLE_TCEIE_MASK                     (0x1U)
#define ENET_DMA_ECC_INTERRUPT_ENABLE_TCEIE_SHIFT                    (0U)
#define ENET_DMA_ECC_INTERRUPT_ENABLE_TCEIE(x)                       (((uint32)(((uint32)(x)) << ENET_DMA_ECC_INTERRUPT_ENABLE_TCEIE_SHIFT)) & ENET_DMA_ECC_INTERRUPT_ENABLE_TCEIE_MASK)
/* ENET_DMA_ECC_INTERRUPT_STATUS Register */
#define ENET_DMA_ECC_INTERRUPT_STATUS_TUES_MASK                      (0x4U)
#define ENET_DMA_ECC_INTERRUPT_STATUS_TUES_SHIFT                     (2U)
#define ENET_DMA_ECC_INTERRUPT_STATUS_TUES(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_ECC_INTERRUPT_STATUS_TUES_SHIFT)) & ENET_DMA_ECC_INTERRUPT_STATUS_TUES_MASK)
#define ENET_DMA_ECC_INTERRUPT_STATUS_TAMS_MASK                      (0x2U)
#define ENET_DMA_ECC_INTERRUPT_STATUS_TAMS_SHIFT                     (1U)
#define ENET_DMA_ECC_INTERRUPT_STATUS_TAMS(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_ECC_INTERRUPT_STATUS_TAMS_SHIFT)) & ENET_DMA_ECC_INTERRUPT_STATUS_TAMS_MASK)
#define ENET_DMA_ECC_INTERRUPT_STATUS_TCES_MASK                      (0x1U)
#define ENET_DMA_ECC_INTERRUPT_STATUS_TCES_SHIFT                     (0U)
#define ENET_DMA_ECC_INTERRUPT_STATUS_TCES(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_ECC_INTERRUPT_STATUS_TCES_SHIFT)) & ENET_DMA_ECC_INTERRUPT_STATUS_TCES_MASK)
/* ENET_DMA_CH_CONTROL Register */
#define ENET_DMA_CH_CONTROL_SPH_MASK                               (0x1000000U)
#define ENET_DMA_CH_CONTROL_SPH_SHIFT                              (24U)
#define ENET_DMA_CH_CONTROL_SPH(x)                                 (((uint32)(((uint32)(x)) << ENET_DMA_CH_CONTROL_SPH_SHIFT)) & ENET_DMA_CH_CONTROL_SPH_MASK)
#define ENET_DMA_CH_CONTROL_DSL_MASK                               (0x1C0000U)
#define ENET_DMA_CH_CONTROL_DSL_SHIFT                              (18U)
#define ENET_DMA_CH_CONTROL_DSL(x)                                 (((uint32)(((uint32)(x)) << ENET_DMA_CH_CONTROL_DSL_SHIFT)) & ENET_DMA_CH_CONTROL_DSL_MASK)
#define ENET_DMA_CH_CONTROL_PBLx8_MASK                             (0x10000U)
#define ENET_DMA_CH_CONTROL_PBLx8_SHIFT                            (16U)
#define ENET_DMA_CH_CONTROL_PBLx8(x)                               (((uint32)(((uint32)(x)) << ENET_DMA_CH_CONTROL_PBLx8_SHIFT)) & ENET_DMA_CH_CONTROL_PBLx8_MASK)
#define ENET_DMA_CH_CONTROL_MSS_MASK                               (0x3FFFU)
#define ENET_DMA_CH_CONTROL_MSS_SHIFT                              (0U)
#define ENET_DMA_CH_CONTROL_MSS(x)                                 (((uint32)(((uint32)(x)) << ENET_DMA_CH_CONTROL_MSS_SHIFT)) & ENET_DMA_CH_CONTROL_MSS_MASK)
/* ENET_DMA_CH_TX_CONTROL Register */
#define ENET_DMA_CH_TX_CONTROL_EDSE_MASK                           (0x10000000U)
#define ENET_DMA_CH_TX_CONTROL_EDSE_SHIFT                          (28U)
#define ENET_DMA_CH_TX_CONTROL_EDSE(x)                             (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_EDSE_SHIFT)) & ENET_DMA_CH_TX_CONTROL_EDSE_MASK)
#define ENET_DMA_CH_TX_CONTROL_TQOS_MASK                           (0xF000000U)
#define ENET_DMA_CH_TX_CONTROL_TQOS_SHIFT                          (24U)
#define ENET_DMA_CH_TX_CONTROL_TQOS(x)                             (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_TQOS_SHIFT)) & ENET_DMA_CH_TX_CONTROL_TQOS_MASK)
#define ENET_DMA_CH_TX_CONTROL_ETIC_MASK                           (0x400000U)
#define ENET_DMA_CH_TX_CONTROL_ETIC_SHIFT                          (22U)
#define ENET_DMA_CH_TX_CONTROL_ETIC(x)                             (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_ETIC_SHIFT)) & ENET_DMA_CH_TX_CONTROL_ETIC_MASK)
#define ENET_DMA_CH_TX_CONTROL_TxPBL_MASK                          (0x3F0000U)
#define ENET_DMA_CH_TX_CONTROL_TxPBL_SHIFT                         (16U)
#define ENET_DMA_CH_TX_CONTROL_TxPBL(x)                            (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_TxPBL_SHIFT)) & ENET_DMA_CH_TX_CONTROL_TxPBL_MASK)
#define ENET_DMA_CH_TX_CONTROL_IPBL_MASK                           (0x8000U)
#define ENET_DMA_CH_TX_CONTROL_IPBL_SHIFT                          (15U)
#define ENET_DMA_CH_TX_CONTROL_IPBL(x)                             (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_IPBL_SHIFT)) & ENET_DMA_CH_TX_CONTROL_IPBL_MASK)
#define ENET_DMA_CH_TX_CONTROL_TSE_MODE_MASK                       (0x6000U)
#define ENET_DMA_CH_TX_CONTROL_TSE_MODE_SHIFT                      (13U)
#define ENET_DMA_CH_TX_CONTROL_TSE_MODE(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_TSE_MODE_SHIFT)) & ENET_DMA_CH_TX_CONTROL_TSE_MODE_MASK)
#define ENET_DMA_CH_TX_CONTROL_TSE_MASK                            (0x1000U)
#define ENET_DMA_CH_TX_CONTROL_TSE_SHIFT                           (12U)
#define ENET_DMA_CH_TX_CONTROL_TSE(x)                              (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_TSE_SHIFT)) & ENET_DMA_CH_TX_CONTROL_TSE_MASK)
#define ENET_DMA_CH_TX_CONTROL_OSF_MASK                            (0x10U)
#define ENET_DMA_CH_TX_CONTROL_OSF_SHIFT                           (4U)
#define ENET_DMA_CH_TX_CONTROL_OSF(x)                              (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_OSF_SHIFT)) & ENET_DMA_CH_TX_CONTROL_OSF_MASK)
#define ENET_DMA_CH_TX_CONTROL_TCW_MASK                            (0xEU)
#define ENET_DMA_CH_TX_CONTROL_TCW_SHIFT                           (1U)
#define ENET_DMA_CH_TX_CONTROL_TCW(x)                              (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_TCW_SHIFT)) & ENET_DMA_CH_TX_CONTROL_TCW_MASK)
#define ENET_DMA_CH_TX_CONTROL_ST_MASK                             (0x1U)
#define ENET_DMA_CH_TX_CONTROL_ST_SHIFT                            (0U)
#define ENET_DMA_CH_TX_CONTROL_ST(x)                               (((uint32)(((uint32)(x)) << ENET_DMA_CH_TX_CONTROL_ST_SHIFT)) & ENET_DMA_CH_TX_CONTROL_ST_MASK)
/* ENET_DMA_CH_RX_CONTROL Register */
#define ENET_DMA_CH_RX_CONTROL_RPF_MASK                            (0x80000000U)
#define ENET_DMA_CH_RX_CONTROL_RPF_SHIFT                           (31U)
#define ENET_DMA_CH_RX_CONTROL_RPF(x)                              (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_CONTROL_RPF_SHIFT)) & ENET_DMA_CH_RX_CONTROL_RPF_MASK)
#define ENET_DMA_CH_RX_CONTROL_RQOS_MASK                           (0xF000000U)
#define ENET_DMA_CH_RX_CONTROL_RQOS_SHIFT                          (24U)
#define ENET_DMA_CH_RX_CONTROL_RQOS(x)                             (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_CONTROL_RQOS_SHIFT)) & ENET_DMA_CH_RX_CONTROL_RQOS_MASK)
#define ENET_DMA_CH_RX_CONTROL_ERIC_MASK                           (0x400000U)
#define ENET_DMA_CH_RX_CONTROL_ERIC_SHIFT                          (22U)
#define ENET_DMA_CH_RX_CONTROL_ERIC(x)                             (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_CONTROL_ERIC_SHIFT)) & ENET_DMA_CH_RX_CONTROL_ERIC_MASK)
#define ENET_DMA_CH_RX_CONTROL_RxPBL_MASK                          (0x3F0000U)
#define ENET_DMA_CH_RX_CONTROL_RxPBL_SHIFT                         (16U)
#define ENET_DMA_CH_RX_CONTROL_RxPBL(x)                            (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_CONTROL_RxPBL_SHIFT)) & ENET_DMA_CH_RX_CONTROL_RxPBL_MASK)
#define ENET_DMA_CH_RX_CONTROL_RBSZ_H_MASK                         (0x7FFEU)
#define ENET_DMA_CH_RX_CONTROL_RBSZ_H_SHIFT                        (1U)
#define ENET_DMA_CH_RX_CONTROL_RBSZ_H(x)                           (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_CONTROL_RBSZ_H_SHIFT)) & ENET_DMA_CH_RX_CONTROL_RBSZ_H_MASK)
#define ENET_DMA_CH_RX_CONTROL_SR_MASK                             (0x1U)
#define ENET_DMA_CH_RX_CONTROL_SR_SHIFT                            (0U)
#define ENET_DMA_CH_RX_CONTROL_SR(x)                               (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_CONTROL_SR_SHIFT)) & ENET_DMA_CH_RX_CONTROL_SR_MASK)
/* ENET_DMA_CH_TXDESC_LIST_HADDRESS Register */
#define ENET_DMA_CH_TXDESC_LIST_HADDRESS_TDESHA_MASK                    (0xFFU)
#define ENET_DMA_CH_TXDESC_LIST_HADDRESS_TDESHA_SHIFT                   (0U)
#define ENET_DMA_CH_TXDESC_LIST_HADDRESS_TDESHA(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_CH_TXDESC_LIST_HADDRESS_TDESHA_SHIFT)) & ENET_DMA_CH_TXDESC_LIST_HADDRESS_TDESHA_MASK)
/* ENET_DMA_CH_TXDESC_LIST_ADDRESS Register */
#define ENET_DMA_CH_TXDESC_LIST_ADDRESS_TDESLA_MASK                    (0xFFFFFFF8U)
#define ENET_DMA_CH_TXDESC_LIST_ADDRESS_TDESLA_SHIFT                   (3U)
#define ENET_DMA_CH_TXDESC_LIST_ADDRESS_TDESLA(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_CH_TXDESC_LIST_ADDRESS_TDESLA_SHIFT)) & ENET_DMA_CH_TXDESC_LIST_ADDRESS_TDESLA_MASK)
/* ENET_DMA_CH_RXDESC_LIST_HADDRESS Register */
#define ENET_DMA_CH_RXDESC_LIST_HADDRESS_RDESHA_MASK                    (0xFFU)
#define ENET_DMA_CH_RXDESC_LIST_HADDRESS_RDESHA_SHIFT                   (0U)
#define ENET_DMA_CH_RXDESC_LIST_HADDRESS_RDESHA(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_CH_RXDESC_LIST_HADDRESS_RDESHA_SHIFT)) & ENET_DMA_CH_RXDESC_LIST_HADDRESS_RDESHA_MASK)
/* ENET_DMA_CH_RXDESC_LIST_ADDRESS Register */
#define ENET_DMA_CH_RXDESC_LIST_ADDRESS_RDESLA_MASK                    (0xFFFFFFF8U)
#define ENET_DMA_CH_RXDESC_LIST_ADDRESS_RDESLA_SHIFT                   (3U)
#define ENET_DMA_CH_RXDESC_LIST_ADDRESS_RDESLA(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_CH_RXDESC_LIST_ADDRESS_RDESLA_SHIFT)) & ENET_DMA_CH_RXDESC_LIST_ADDRESS_RDESLA_MASK)
/* ENET_DMA_CH_TXDESC_TAIL_POINTER Register */
#define ENET_DMA_CH_TXDESC_TAIL_POINTER_TDTP_MASK                      (0xFFFFFFF8U)
#define ENET_DMA_CH_TXDESC_TAIL_POINTER_TDTP_SHIFT                     (3U)
#define ENET_DMA_CH_TXDESC_TAIL_POINTER_TDTP(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_TXDESC_TAIL_POINTER_TDTP_SHIFT)) & ENET_DMA_CH_TXDESC_TAIL_POINTER_TDTP_MASK)
/* ENET_DMA_CH_RXDESC_TAIL_POINTER Register */
#define ENET_DMA_CH_RXDESC_TAIL_POINTER_RDTP_MASK                      (0xFFFFFFF8U)
#define ENET_DMA_CH_RXDESC_TAIL_POINTER_RDTP_SHIFT                     (3U)
#define ENET_DMA_CH_RXDESC_TAIL_POINTER_RDTP(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_RXDESC_TAIL_POINTER_RDTP_SHIFT)) & ENET_DMA_CH_RXDESC_TAIL_POINTER_RDTP_MASK)
/* ENET_DMA_CH_TXDESC_RING_LENGTH Register */
#define ENET_DMA_CH_TXDESC_RING_LENGTH_TDRL_MASK                      (0x3FFU)
#define ENET_DMA_CH_TXDESC_RING_LENGTH_TDRL_SHIFT                     (0U)
#define ENET_DMA_CH_TXDESC_RING_LENGTH_TDRL(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_TXDESC_RING_LENGTH_TDRL_SHIFT)) & ENET_DMA_CH_TXDESC_RING_LENGTH_TDRL_MASK)
/* ENET_DMA_CH_RXDESC_RING_LENGTH Register */
#define ENET_DMA_CH_RXDESC_RING_LENGTH_RDRL_MASK                      (0x3FFU)
#define ENET_DMA_CH_RXDESC_RING_LENGTH_RDRL_SHIFT                     (0U)
#define ENET_DMA_CH_RXDESC_RING_LENGTH_RDRL(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_RXDESC_RING_LENGTH_RDRL_SHIFT)) & ENET_DMA_CH_RXDESC_RING_LENGTH_RDRL_MASK)
/* ENET_DMA_CH_INTERRUPT_ENABLE Register */
#define ENET_DMA_CH_INTERRUPT_ENABLE_NIE_MASK                       (0x8000U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_NIE_SHIFT                      (15U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_NIE(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_NIE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_NIE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_AIE_MASK                       (0x4000U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_AIE_SHIFT                      (14U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_AIE(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_AIE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_AIE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_CDEE_MASK                      (0x2000U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_CDEE_SHIFT                     (13U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_CDEE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_CDEE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_CDEE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_FBEE_MASK                      (0x1000U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_FBEE_SHIFT                     (12U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_FBEE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_FBEE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_FBEE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_ERIE_MASK                      (0x800U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_ERIE_SHIFT                     (11U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_ERIE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_ERIE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_ERIE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_ETIE_MASK                      (0x400U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_ETIE_SHIFT                     (10U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_ETIE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_ETIE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_ETIE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RWTE_MASK                      (0x200U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RWTE_SHIFT                     (9U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RWTE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_RWTE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_RWTE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RSE_MASK                       (0x100U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RSE_SHIFT                      (8U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RSE(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_RSE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_RSE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RBUE_MASK                      (0x80U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RBUE_SHIFT                     (7U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RBUE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_RBUE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_RBUE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RIE_MASK                       (0x40U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RIE_SHIFT                      (6U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_RIE(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_RIE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_RIE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TBUE_MASK                      (0x4U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TBUE_SHIFT                     (2U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TBUE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_TBUE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_TBUE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TXSE_MASK                      (0x2U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TXSE_SHIFT                     (1U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TXSE(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_TXSE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_TXSE_MASK)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TIE_MASK                       (0x1U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TIE_SHIFT                      (0U)
#define ENET_DMA_CH_INTERRUPT_ENABLE_TIE(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_INTERRUPT_ENABLE_TIE_SHIFT)) & ENET_DMA_CH_INTERRUPT_ENABLE_TIE_MASK)
/* ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER Register */
#define ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWTU_MASK                      (0x30000U)
#define ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWTU_SHIFT                     (16U)
#define ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWTU(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWTU_SHIFT)) & ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWTU_MASK)
#define ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWT_MASK                       (0xFFU)
#define ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWT_SHIFT                      (0U)
#define ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWT(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWT_SHIFT)) & ENET_DMA_CH_RX_INTERRUPT_WATCHDOG_TIMER_RWT_MASK)
/* ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS Register */
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_RSN_MASK                       (0xF0000U)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_RSN_SHIFT                      (16U)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_RSN(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_RSN_SHIFT)) & ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_RSN_MASK)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_SIV_MASK                       (0xFFF0U)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_SIV_SHIFT                      (4U)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_SIV(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_SIV_SHIFT)) & ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_SIV_MASK)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ASC_MASK                       (0x2U)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ASC_SHIFT                      (1U)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ASC(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ASC_SHIFT)) & ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ASC_MASK)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ESC_MASK                       (0x1U)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ESC_SHIFT                      (0U)
#define ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ESC(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ESC_SHIFT)) & ENET_DMA_CH_SLOT_FUNCTION_CONTROL_STATUS_ESC_MASK)
/* ENET_DMA_CH_CURRENT_APP_TXDESC Register */
#define ENET_DMA_CH_CURRENT_APP_TXDESC_CURTDESAPTR_MASK               (0xFFFFFFFFU)
#define ENET_DMA_CH_CURRENT_APP_TXDESC_CURTDESAPTR_SHIFT              (0U)
#define ENET_DMA_CH_CURRENT_APP_TXDESC_CURTDESAPTR(x)                 (((uint32)(((uint32)(x)) << ENET_DMA_CH_CURRENT_APP_TXDESC_CURTDESAPTR_SHIFT)) & ENET_DMA_CH_CURRENT_APP_TXDESC_CURTDESAPTR_MASK)
/* ENET_DMA_CH_CURRENT_APP_RXDESC Register */
#define ENET_DMA_CH_CURRENT_APP_RXDESC_CURRDESAPTR_MASK               (0xFFFFFFFFU)
#define ENET_DMA_CH_CURRENT_APP_RXDESC_CURRDESAPTR_SHIFT              (0U)
#define ENET_DMA_CH_CURRENT_APP_RXDESC_CURRDESAPTR(x)                 (((uint32)(((uint32)(x)) << ENET_DMA_CH_CURRENT_APP_RXDESC_CURRDESAPTR_SHIFT)) & ENET_DMA_CH_CURRENT_APP_RXDESC_CURRDESAPTR_MASK)
/* ENET_DMA_CH_CURRENT_APP_TXBUFFER_H Register */
#define ENET_DMA_CH_CURRENT_APP_TXBUFFER_H_CURTBUFAPTRH_MASK              (0xFFU)
#define ENET_DMA_CH_CURRENT_APP_TXBUFFER_H_CURTBUFAPTRH_SHIFT             (0U)
#define ENET_DMA_CH_CURRENT_APP_TXBUFFER_H_CURTBUFAPTRH(x)                (((uint32)(((uint32)(x)) << ENET_DMA_CH_CURRENT_APP_TXBUFFER_H_CURTBUFAPTRH_SHIFT)) & ENET_DMA_CH_CURRENT_APP_TXBUFFER_H_CURTBUFAPTRH_MASK)
/* ENET_DMA_CH_CURRENT_APP_TXBUFFER Register */
#define ENET_DMA_CH_CURRENT_APP_TXBUFFER_CURTBUFAPTR_MASK               (0xFFFFFFFFU)
#define ENET_DMA_CH_CURRENT_APP_TXBUFFER_CURTBUFAPTR_SHIFT              (0U)
#define ENET_DMA_CH_CURRENT_APP_TXBUFFER_CURTBUFAPTR(x)                 (((uint32)(((uint32)(x)) << ENET_DMA_CH_CURRENT_APP_TXBUFFER_CURTBUFAPTR_SHIFT)) & ENET_DMA_CH_CURRENT_APP_TXBUFFER_CURTBUFAPTR_MASK)
/* ENET_DMA_CH_Current_App_RxBuffer_H Register */
#define ENET_DMA_CH_Current_App_RxBuffer_H_CURRBUFAPTRH_MASK              (0xFFU)
#define ENET_DMA_CH_Current_App_RxBuffer_H_CURRBUFAPTRH_SHIFT             (0U)
#define ENET_DMA_CH_Current_App_RxBuffer_H_CURRBUFAPTRH(x)                (((uint32)(((uint32)(x)) << ENET_DMA_CH_Current_App_RxBuffer_H_CURRBUFAPTRH_SHIFT)) & ENET_DMA_CH_Current_App_RxBuffer_H_CURRBUFAPTRH_MASK)
/* ENET_DMA_CH_CURRENT_APP_RXBUFFER Register */
#define ENET_DMA_CH_CURRENT_APP_RXBUFFER_CURRBUFAPTR_MASK               (0xFFFFFFFFU)
#define ENET_DMA_CH_CURRENT_APP_RXBUFFER_CURRBUFAPTR_SHIFT              (0U)
#define ENET_DMA_CH_CURRENT_APP_RXBUFFER_CURRBUFAPTR(x)                 (((uint32)(((uint32)(x)) << ENET_DMA_CH_CURRENT_APP_RXBUFFER_CURRBUFAPTR_SHIFT)) & ENET_DMA_CH_CURRENT_APP_RXBUFFER_CURRBUFAPTR_MASK)
/* ENET_DMA_CH_STATUS Register */
#define ENET_DMA_CH_STATUS_REB_MASK                                (0x380000U)
#define ENET_DMA_CH_STATUS_REB_SHIFT                               (19U)
#define ENET_DMA_CH_STATUS_REB(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_REB_SHIFT)) & ENET_DMA_CH_STATUS_REB_MASK)
#define ENET_DMA_CH_STATUS_TEB_MASK                                (0x70000U)
#define ENET_DMA_CH_STATUS_TEB_SHIFT                               (16U)
#define ENET_DMA_CH_STATUS_TEB(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_TEB_SHIFT)) & ENET_DMA_CH_STATUS_TEB_MASK)
#define ENET_DMA_CH_STATUS_NIS_MASK                                (0x8000U)
#define ENET_DMA_CH_STATUS_NIS_SHIFT                               (15U)
#define ENET_DMA_CH_STATUS_NIS(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_NIS_SHIFT)) & ENET_DMA_CH_STATUS_NIS_MASK)
#define ENET_DMA_CH_STATUS_AIS_MASK                                (0x4000U)
#define ENET_DMA_CH_STATUS_AIS_SHIFT                               (14U)
#define ENET_DMA_CH_STATUS_AIS(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_AIS_SHIFT)) & ENET_DMA_CH_STATUS_AIS_MASK)
#define ENET_DMA_CH_STATUS_CDE_MASK                                (0x2000U)
#define ENET_DMA_CH_STATUS_CDE_SHIFT                               (13U)
#define ENET_DMA_CH_STATUS_CDE(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_CDE_SHIFT)) & ENET_DMA_CH_STATUS_CDE_MASK)
#define ENET_DMA_CH_STATUS_FBE_MASK                                (0x1000U)
#define ENET_DMA_CH_STATUS_FBE_SHIFT                               (12U)
#define ENET_DMA_CH_STATUS_FBE(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_FBE_SHIFT)) & ENET_DMA_CH_STATUS_FBE_MASK)
#define ENET_DMA_CH_STATUS_ERI_MASK                                (0x800U)
#define ENET_DMA_CH_STATUS_ERI_SHIFT                               (11U)
#define ENET_DMA_CH_STATUS_ERI(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_ERI_SHIFT)) & ENET_DMA_CH_STATUS_ERI_MASK)
#define ENET_DMA_CH_STATUS_ETI_MASK                                (0x400U)
#define ENET_DMA_CH_STATUS_ETI_SHIFT                               (10U)
#define ENET_DMA_CH_STATUS_ETI(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_ETI_SHIFT)) & ENET_DMA_CH_STATUS_ETI_MASK)
#define ENET_DMA_CH_STATUS_RWT_MASK                                (0x200U)
#define ENET_DMA_CH_STATUS_RWT_SHIFT                               (9U)
#define ENET_DMA_CH_STATUS_RWT(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_RWT_SHIFT)) & ENET_DMA_CH_STATUS_RWT_MASK)
#define ENET_DMA_CH_STATUS_RPS_MASK                                (0x100U)
#define ENET_DMA_CH_STATUS_RPS_SHIFT                               (8U)
#define ENET_DMA_CH_STATUS_RPS(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_RPS_SHIFT)) & ENET_DMA_CH_STATUS_RPS_MASK)
#define ENET_DMA_CH_STATUS_RBU_MASK                                (0x80U)
#define ENET_DMA_CH_STATUS_RBU_SHIFT                               (7U)
#define ENET_DMA_CH_STATUS_RBU(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_RBU_SHIFT)) & ENET_DMA_CH_STATUS_RBU_MASK)
#define ENET_DMA_CH_STATUS_RI_MASK                                 (0x40U)
#define ENET_DMA_CH_STATUS_RI_SHIFT                                (6U)
#define ENET_DMA_CH_STATUS_RI(x)                                   (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_RI_SHIFT)) & ENET_DMA_CH_STATUS_RI_MASK)
#define ENET_DMA_CH_STATUS_TBU_MASK                                (0x4U)
#define ENET_DMA_CH_STATUS_TBU_SHIFT                               (2U)
#define ENET_DMA_CH_STATUS_TBU(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_TBU_SHIFT)) & ENET_DMA_CH_STATUS_TBU_MASK)
#define ENET_DMA_CH_STATUS_TPS_MASK                                (0x2U)
#define ENET_DMA_CH_STATUS_TPS_SHIFT                               (1U)
#define ENET_DMA_CH_STATUS_TPS(x)                                  (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_TPS_SHIFT)) & ENET_DMA_CH_STATUS_TPS_MASK)
#define ENET_DMA_CH_STATUS_TI_MASK                                 (0x1U)
#define ENET_DMA_CH_STATUS_TI_SHIFT                                (0U)
#define ENET_DMA_CH_STATUS_TI(x)                                   (((uint32)(((uint32)(x)) << ENET_DMA_CH_STATUS_TI_SHIFT)) & ENET_DMA_CH_STATUS_TI_MASK)
/* ENET_DMA_CH_MISS_FRAME_CNT Register */
#define ENET_DMA_CH_MISS_FRAME_CNT_MFCO_MASK                       (0x8000U)
#define ENET_DMA_CH_MISS_FRAME_CNT_MFCO_SHIFT                      (15U)
#define ENET_DMA_CH_MISS_FRAME_CNT_MFCO(x)                         (((uint32)(((uint32)(x)) << ENET_DMA_CH_MISS_FRAME_CNT_MFCO_SHIFT)) & ENET_DMA_CH_MISS_FRAME_CNT_MFCO_MASK)
#define ENET_DMA_CH_MISS_FRAME_CNT_MFC_MASK                        (0x7FFU)
#define ENET_DMA_CH_MISS_FRAME_CNT_MFC_SHIFT                       (0U)
#define ENET_DMA_CH_MISS_FRAME_CNT_MFC(x)                          (((uint32)(((uint32)(x)) << ENET_DMA_CH_MISS_FRAME_CNT_MFC_SHIFT)) & ENET_DMA_CH_MISS_FRAME_CNT_MFC_MASK)
/* ENET_DMA_CH_RXP_ACCEPT_CNT Register */
#define ENET_DMA_CH_RXP_ACCEPT_CNT_RXPACOF_MASK                    (0x80000000U)
#define ENET_DMA_CH_RXP_ACCEPT_CNT_RXPACOF_SHIFT                   (31U)
#define ENET_DMA_CH_RXP_ACCEPT_CNT_RXPACOF(x)                      (((uint32)(((uint32)(x)) << ENET_DMA_CH_RXP_ACCEPT_CNT_RXPACOF_SHIFT)) & ENET_DMA_CH_RXP_ACCEPT_CNT_RXPACOF_MASK)
#define ENET_DMA_CH_RXP_ACCEPT_CNT_RXPAC_MASK                      (0x7FFFFFFFU)
#define ENET_DMA_CH_RXP_ACCEPT_CNT_RXPAC_SHIFT                     (0U)
#define ENET_DMA_CH_RXP_ACCEPT_CNT_RXPAC(x)                        (((uint32)(((uint32)(x)) << ENET_DMA_CH_RXP_ACCEPT_CNT_RXPAC_SHIFT)) & ENET_DMA_CH_RXP_ACCEPT_CNT_RXPAC_MASK)
/* ENET_DMA_CH_RX_ERI_CNT Register */
#define ENET_DMA_CH_RX_ERI_CNT_ECNT_MASK                           (0xFFFU)
#define ENET_DMA_CH_RX_ERI_CNT_ECNT_SHIFT                          (0U)
#define ENET_DMA_CH_RX_ERI_CNT_ECNT(x)                             (((uint32)(((uint32)(x)) << ENET_DMA_CH_RX_ERI_CNT_ECNT_SHIFT)) & ENET_DMA_CH_RX_ERI_CNT_ECNT_MASK)



/**
 * @}
 */ /* end of group ENET_Register_Masks */


/**
 * @}
 */ /* end of group ENET_Peripheral_Access_Layer */

#endif /* PSIP_ENET_H */

