/*
 * Copyright 2020-2025 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 * used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/*!
 * @file YTM32B1HA0_features.h
 * @brief Chip specific module features
 */

#ifndef YTM32B1HA0_FEATURES_H
#define YTM32B1HA0_FEATURES_H

/* ERRATA sections*/

/* PRQA S 0779 EOF */

/* @brief Number of cores. */
#ifndef NUMBER_OF_CORES
#define NUMBER_OF_CORES         (uint8)(1U)
#endif /* NUMBER_OF_CORES */

/* SOC module features */

/* @brief PORT availability on the SoC. */
#define FEATURE_SOC_PORT_COUNT              (5)

#define FEATURE_SOC_IPC_COUNT               (1)

#ifdef FPGA_PLATFORM
/* @brief Slow IRC clock frequency(12MHz). */
#define FEATURE_SCU_SIRC_FREQ               (2000000U)
/* @brief Fast IRC trimmed clock frequency(96MHz). */
#define FEATURE_SCU_FIRC_FREQ               (20000000U)
/* @brief FXOSC clock frequency(24MHz). */
#define FEATURE_SCU_FXOSC_FREQ              (20000000U)
/* @brief SXOSC clock frequency(32.768K). */
#define FEATURE_SCU_SXOSC_FREQ              (32768U)
#else
/* @brief Slow IRC clock frequency(12MHz). */
#define FEATURE_SCU_SIRC_FREQ               (10000000U)
/* @brief Fast IRC trimmed clock frequency(96MHz). */
#define FEATURE_SCU_FIRC_FREQ               (10000000U)
/* @brief FXOSC clock frequency(24MHz). */
#define FEATURE_SCU_FXOSC_FREQ              (10000000U)
/* @brief SCU support SXOSC 32.768K clock. */
#define FEATURE_SCU_SUPPORT_SXOSC           (1)
#if FEATURE_SCU_SUPPORT_SXOSC
/* @brief SXOSC clock frequency(32.768K). */
#define FEATURE_SCU_SXOSC_FREQ              (32768U)
#endif /* FEATURE_SCU_SUPPORT_SXOSC */
#endif /* FPGA_PLATFORM */
/* @brief Clock switch PLL lock timeout loop count. */
#define FEATURE_CLOCK_TIMEOUT_LOOP_CNT      (0x1FFFF)

/* FLASH module features */

/* @brief Flash Type. */
#define FEATURE_EFM_IS_FULL                 (1u)
/* @brief Flash block count. */
#define FEATURE_EFM_BLOCK_COUNT             (3u)
/* @brief Flash program flash end address. */
#define FEATURE_EFM_MAIN_ARRAY_END_ADDRESS  (0x021FFFFFU)
/* @brief Flash sector size. */
#define FEATURE_EFM_MAIN_ARRAY_SECTOR_SIZE  (0x800u)
/* @brief Flash has data flash. */
#define FEATURE_EFM_HAS_DATA_FLASH          (1)
#if FEATURE_EFM_HAS_DATA_FLASH
/* @brief Flash data flash start address. */
#define FEATURE_EFM_DATA_ARRAY_START_ADDRESS (0x06000000U)
/* @brief Flash data flash end address. */
#define FEATURE_EFM_DATA_ARRAY_END_ADDRESS  (0x06040000U)
/* @brief Flash data flash sector size. */
#define FEATURE_EFM_DATA_ARRAY_SECTOR_SIZE  (0x400u)
#endif /* FEATURE_EFM_HAS_DATA_FLASH */
#define FEATURE_EFM_HAS_NVR_FLASH          (1)
#if FEATURE_EFM_HAS_NVR_FLASH
/* @brief Flash NVR start address. */
#define FEATURE_EFM_NVR_ARRAY_START_ADDRESS (0x10000000U)
/* @brief Flash NVR end address. */
#define FEATURE_EFM_NVR_ARRAY_END_ADDRESS   (0x10040400U)
/* @brief Flash NVR sector size. */
#define FEATURE_EFM_NVR_ARRAY_SECTOR_SIZE   (0x400u)
#endif /* FEATURE_EFM_HAS_NVR_FLASH */

/* @brief Flash sector minimum size. */
#define FEATURE_EFM_FLASH_MIN_SECTOR_SIZE   (0x400u)

/* @brief Flash BOOT NVR start address. */
#define FEATURE_EFM_BOOT_NVR_START_ADDRESS (0x10003800U)

/* @brief Flash write unit size. */
#define FEATURE_EFM_WRITE_MAIN_UNIT_SIZE            (32u)
/* @brief Flash write unit size in words. */
#define FEATURE_EFM_WRITE_MAIN_UNIT_WORD_SIZE       (FEATURE_EFM_WRITE_MAIN_UNIT_SIZE >> 2)
/* @brief Flash write unit size. */
#define FEATURE_EFM_WRITE_DATA_UNIT_SIZE            (8u)
/* @brief Flash write unit size in words. */
#define FEATURE_EFM_WRITE_DATA_UNIT_WORD_SIZE       (FEATURE_EFM_WRITE_DATA_UNIT_SIZE >> 2)
/* @brief Flash Program unit command. */
#define FEATURE_EFM_PROGRAM_DATA_CMD_CODE           (0x01u)
/* @brief Flash Program unit command. */
#define FEATURE_EFM_PROGRAM_MAIN_CMD_CODE           (0x02u)
/* @brief Flash Erase Sector command. */
#define FEATURE_EFM_ERASE_SECTOR_CMD_CODE           (0x10u)
/* @brief Erase Flash Block command. */
#define FEATURE_EFM_ERASE_BLOCK_CMD_CODE            (0x11u)
/* @brief Erase Flash Chip command. */
#define FEATURE_EFM_ERASE_CHIP_CMD_CODE             (0x12u)
/* @brief Erase Sector Retry Flash command. */
#define FEATURE_EFM_ERASE_SECTOR_AUTO_RETRY_CMD_CODE     (0x13u)
/* @brief Erase Sector Retry Flash command. */
#define FEATURE_EFM_ERASE_SECTOR_MANUAL_RETRY_CMD_CODE     (0x14u)
/* @brief Erase Sector Retry Flash command. */
#define FEATURE_EFM_ERASE_SECTOR_VREAD_RETRY_CMD_CODE     (0x15u)
/* @brief Flash Load AES KEY command. */
#define FEATURE_EFM_LOAD_AES_KEY_CMD_CODE           (0x20u)
/* @brief Flash Boot Swap command. */
#define FEATURE_EFM_BOOT_SWAP_CMD_CODE              (0x30u)
/* @brief EFM command protect/disable. */
#define FEATURE_EFM_CMD_PROT_CODE                   (0x1Du)
/* @brief Flash Status Error Code Mask. */
#define FEATURE_EFM_CMD_ERROR_MASK          (EFM_STS_FAIL_MASK | EFM_STS_ACCERR_MASK | EFM_STS_UNRECOVERR_MASK | EFM_STS_RECOVERR_MASK)
/* @brief Flash Unlock command register. */
#define EFM_UNLOCK_CMD_REGISTER() do {                 \
    EFM->CMD_UNLOCK = 0xfd9573f5;                      \
    __asm("ISB");                                      \
    __asm("DSB");                                      \
} while (0)
/* @brief Flash enable flash write commands. */
#define EFM_ENABLE_WE_COMMAND() do {                   \
    EFM->CTRL |= EFM_CTRL_WE_MASK;                     \
    __asm("ISB");                                      \
    __asm("DSB");                                      \
} while (0)
/* @brief Flash disable flash write commands. */
#define EFM_DISABLE_WE_COMMAND() do {                  \
    __asm("ISB");                                      \
    __asm("DSB");                                      \
    EFM->CTRL &= ~EFM_CTRL_WE_MASK;                    \
} while (0)

/* RCU module features */
/* @brief Reset pin output feature */
#define FEATURE_RCU_HAS_PIN_OUTPUT                  (1U)

/* WDG module features */

/* @brief The 32-bit value used for unlocking the WDG. */
#define FEATURE_WDG_UNLOCK_VALUE_1                      (0xB631)
#define FEATURE_WDG_UNLOCK_VALUE_2                      (0xC278)
/* @brief The 32-bit value used for resetting the WDG counter. */
#define FEATURE_WDG_TRIGGER_VALUE_1                     (0xA518)
#define FEATURE_WDG_TRIGGER_VALUE_2                     (0xD826)
/* @brief The reset value of the timeout register. */
#define FEATURE_WDG_TOVR_RESET_VALUE                    (0x300U)
/* @brief The value minimum of the timeout register. */
#define FEATURE_WDG_MINIMUM_TIMEOUT_VALUE               (0x0U)
/* @brief The reset value of the window register. */
#define FEATURE_WDG_WVR_RESET_VALUE                     (0x0U)
/* @brief Default reset value of the LR register. */
#define FEATURE_WDG_LR_RESET_VALUE                      (0x1UL)
/* @brief The value used to set WDG source clock from LPO. */
#define FEATURE_WDG_CLK_FROM_LPO                        (0x1UL)

/* @brief Default reset value of the CR register. */
#define FEATURE_WDG_CR_RESET_VALUE                      (0x82U)

/* EWDG module features */
/* @brief First byte of the EWM Service key        */
#define FEATURE_EWDG_KEY_FIRST_BYTE                      (0xA8U)
/* @brief Second byte of the EWM Service key       */
#define FEATURE_EWDG_KEY_SECOND_BYTE                     (0x6CU)
/* @brief EWDG maximum compare high value. */
#define FEATURE_EWDG_CMPH_MAX_VALUE                      (0xFEU)
/* @brief EWDG minimum compare low value. */
#define FEATURE_EWDG_CMPL_MIN_VALUE                      (0x00U)


/* Interrupt module features */

/* @brief Lowest interrupt request number. */
#define FEATURE_INTERRUPT_IRQ_MIN                       (NMI_IRQn)
/* @brief Highest interrupt request number. */
#define FEATURE_INTERRUPT_IRQ_MAX                       (WKU_IRQn)
/**< Number of priority bits implemented in the NVIC */
#define FEATURE_NVIC_PRIO_BITS                          (3U)
/* @brief Has software interrupt. */
#define FEATURE_INTERRUPT_HAS_SOFTWARE_IRQ              (0u)
/* @brief Has pending interrupt state. */
#define FEATURE_INTERRUPT_HAS_PENDING_STATE             (1u)
/* @brief Has active interrupt state. */
#define FEATURE_INTERRUPT_HAS_ACTIVE_STATE              (0u)
/* @brief Multicore support for interrupts */
#define FEATURE_INTERRUPT_MULTICORE_SUPPORT             (0u)
/* @brief Registers in which the start of interrupt vector table needs to be configured */
#define FEATURE_INTERRUPT_INT_VECTORS                   {&SCB->VTOR}

/* MPWM module features */

#define MPWM_CLKS                                       {FAST_BUS_CLK, FAST_BUS_CLK, FAST_BUS_CLK}





/* eTMR module features */

/* @brief Number of PWM channels */
#define FEATURE_eTMR_CHANNEL_COUNTS                     {8U, 8U, 8U, 8U, 8U, 8U}
#define FEATURE_eTMR_CHANNEL_INTERRUPT_COUNTS           (4U)
#define FEATURE_eTMR_CHANNEL_MAX_COUNT                  (8U)
#define FEATURE_eTMR0_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR1_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR2_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR3_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR4_CHANNEL_MAX_COUNT                 (8U)
#define FEATURE_eTMR5_CHANNEL_MAX_COUNT                 (8U)
/* @brief Number of fault channels */
#define FEATURE_eTMR_FAULT_CHANNELS                     (4U)
/* @brief Output channel offset */
#define FEATURE_eTMR_OUTPUT_CHANNEL_OFFSET              (16U)
/* @brief Max counter value */
#define FEATURE_eTMR_CNT_MAX_VALUE_U32                  (0xFFFFFFFFU)
/* @brief Input capture for single shot */
#define FEATURE_eTMR_INPUT_CAPTURE_SINGLE_SHOT          (2U)
/* @brief Number of interrupt vector for channels of the eTMR module. */
#define FEATURE_eTMR_HAS_NUM_IRQS_CHANS                 (4U)
#define eTMR_INTERNAL_CLOCK                             (FAST_BUS_CLK)
/* @brief eTMR external clock source frequency */
#define FEATURE_eTMR_EXTERNAL_CLOCK_FREQ                (1000000U)
/* @brief eTMR Dither MOD configuration */
#define FEATURE_eTMR_DITHER_CONFIGURATION               (1U)

/* @brief eTMR  */

/* I2C module features */

/* @brief DMA instance used for I2C module */
#define I2C_DMA_INSTANCE                                (0U)

/* @brief DMA requests for I2C module. */
#define I2C_DMA_REQ                                     {{(uint8_t)DMA_REQ_I2C0_TX, (uint8_t)DMA_REQ_I2C0_RX}, \
                                                            {(uint8_t)DMA_REQ_I2C1_TX, (uint8_t)DMA_REQ_I2C1_RX}, \
                                                            {(uint8_t)DMA_REQ_I2C2_TX, (uint8_t)DMA_REQ_I2C2_RX}, \
                                                            {(uint8_t)DMA_REQ_I2C3_TX, (uint8_t)DMA_REQ_I2C3_RX}, \
                                                            {(uint8_t)DMA_REQ_I2C4_TX, (uint8_t)DMA_REQ_I2C4_RX}}
/* @brief IPC clocks for I2C module. */
#define I2C_IPC_CLOCKS                                  {I2C0_CLK, I2C1_CLK, I2C2_CLK, I2C3_CLK, I2C4_CLK}

/* @brief If all the i2c instance are support slave mode */
#define I2C_SLAVE_SUPPORT_ALL_INSTANCE                  (TRUE)
/* @brief Slave mode support I2C module. */
#define I2C_SLAVE_SUPPORT                               {TRUE, TRUE, TRUE, TRUE, TRUE}
/* @brief I2C module support TXCFG. */
#define I2C_SUPPORT_TXCFG                               (1)

/* @brief Disable high-speed and ultra-fast operating modes */
#define I2C_HAS_FAST_PLUS_MODE                          (1U)
#define I2C_HAS_HIGH_SPEED_MODE                         (1U)
#define I2C_HAS_ULTRA_FAST_MODE                         (1U)

/* CRC module features */

/* @brief CRC module */
#define FEATURE_CRC_DRIVER_SOFT_POLYNOMIAL
/* @brief Default CRC bit width */
#define FEATURE_CRC_DEFAULT_WIDTH                       CRC_BITS_16
/* @brief Default CRC read transpose */
#define FEATURE_CRC_DEFAULT_READ_TRANSPOSE              CRC_TRANSPOSE_NONE
/* @brief Default CRC write transpose */
#define FEATURE_CRC_DEFAULT_WRITE_TRANSPOSE             CRC_TRANSPOSE_NONE
/* @brief Default polynomial 0x1021U */
#define FEATURE_CRC_DEFAULT_POLYNOMIAL                  (0x1021U)
/* @brief Default seed value is 0xFFFFU */
#define FEATURE_CRC_DEFAULT_SEED                        (0xFFFFU)

/* PCTRL module features */
/* @brief Has control lock. */
#define FEATURE_PCTRL_HAS_PIN_CONTROL_LOCK              (0)
/* @brief Has ALT15. */
#define FEATURE_PCTRL_HAS_ALT_15                        (1)
/* @brief Has open drain control. */
#define FEATURE_PINS_HAS_OPEN_DRAIN                     (0)
/* @brief Has digital filter. */
#define FEATURE_PINS_HAS_DIGITAL_FILTER                 (1)
/* @brief Has digital filter clock selection. */
#define FEATURE_PINS_HAS_FILTER_CLOCK_SEL               (0)
/* @brief Has trigger output to trigger other peripherals. */
#define FEATURE_PINS_HAS_TRIGGER_OUT                    (0)
/* @brief Has setting flag only. */
#define FEATURE_PINS_HAS_FLAG_SET_ONLY                  (0)
/* @brief Has pull resistor selection available. */
#define FEATURE_PINS_HAS_PULL_SELECTION                 (1)
/* @brief Has slew rate control. */
#define FEATURE_PINS_HAS_SLEW_RATE                      (1)
/* @brief Has filter clock selection control. */
#define FEATURE_PINS_HAS_FILTER_CLOCK_SELECTION         (0)
/* @brief Has passive filter. */
#define FEATURE_PINS_HAS_PASSIVE_FILTER                 (1)
/* @brief Has drive strength. */
#define FEATURE_PINS_HAS_DRIVE_STRENGTH                 (1)
/* @brief Has drive strength control bits*/
#define FEATURE_PINS_HAS_DRIVE_STRENGTH_CONTROL         (1)
/* @brief Has port input disable control bits*/
#define FEATURE_PINS_HAS_INPUT_DISABLE                  (1)


/* ACMP module features */

/* @brief Comparator has internal DAC module */
#define FEATURE_ACMP_HAS_DAC                            (1U)
/* @brief Comparator has power mode control */
#define FEATURE_ACMP_HAS_POWER_MODE                     (1U)
/* @brief Comparator has pin output enable */
#define FEATURE_ACMP_HAS_PINOUT_ENABLE                  (0U)
/* @brief Comparator has mode config */
#define FEATURE_ACMP_HAS_MODE_CFG                       (1U)
/* @brief ACMP Function Clock Selection */
#define FEATURE_ACMP_HAS_CLK_SRC_SEL                    (1U)

#define FEATURE_ACMP_CTRL_RESET_VALUE                   (0x00000000)
#define FEATURE_ACMP_STS_CLEAR_MASK                     (0x00FF0000)
#define FEATURE_ACMP_STS_CH_OUT_MASK                    (0xFF000000)
#define FEATURE_ACMP_STS_CH_OUT_SHIFT                   (24)
#define FEATURE_ACMP_STS_CH_FLAG_MASK                   (0x00FF0000)
#define FEATURE_ACMP_STS_CH_FLAG_SHIFT                  (16)
#define FEATURE_ACMP_DEFAULT_CONT_PERIOD                (0x30)
#define FEATURE_ACMP_DEFAULT_CONT_POS                   (0x18)

#if FEATURE_ACMP_HAS_DAC
/* @brief Comparator fix DAC input to mux side */
#define FEATURE_ACMP_DAC_FIX_SELECTION                  (0U)
#define ACMP_DAC_RESOLUTION                             (255U)
#endif /* FEATURE_ACMP_HAS_DAC */

/*! @brief Clock names. */
typedef enum
{
    /* IPC clocks */
    DMA_CLK                  =  0U, /*!< DMA        clock source */
    TRACE_CLK                =  1U, /*!< TRACE      clock source */
    GPIO_CLK                 =  2U, /*!< GPIO       clock source */
    PCTRLA_CLK               =  3U, /*!< PCTRLA     clock source */
    PCTRLB_CLK               =  4U, /*!< PCTRLB     clock source */
    PCTRLC_CLK               =  5U, /*!< PCTRLC     clock source */
    PCTRLD_CLK               =  6U, /*!< PCTRLD     clock source */
    PCTRLE_CLK               =  7U, /*!< PCTRLE     clock source */
    LINFlexD0_CLK            =  8U, /*!< LINFlexD0  clock source */
    LINFlexD1_CLK            =  9U, /*!< LINFlexD1  clock source */
    LINFlexD2_CLK            = 10U, /*!< LINFlexD2  clock source */
    LINFlexD3_CLK            = 11U, /*!< LINFlexD3  clock source */
    LINFlexD4_CLK            = 12U, /*!< LINFlexD4  clock source */
    LINFlexD5_CLK            = 13U, /*!< LINFlexD5  clock source */
    LINFlexD6_CLK            = 14U, /*!< LINFlexD6  clock source */
    LINFlexD7_CLK            = 15U, /*!< LINFlexD7  clock source */
    LINFlexD8_CLK            = 16U, /*!< LINFlexD8  clock source */
    LINFlexD9_CLK            = 17U, /*!< LINFlexD9  clock source */
    I2C0_CLK                 = 18U, /*!< I2C0       clock source */
    I2C1_CLK                 = 19U, /*!< I2C1       clock source */
    I2C2_CLK                 = 20U, /*!< I2C2       clock source */
    I2C3_CLK                 = 21U, /*!< I2C3       clock source */
    I2C4_CLK                 = 22U, /*!< I2C4       clock source */
    SPI0_CLK                 = 23U, /*!< SPI0       clock source */
    SPI1_CLK                 = 24U, /*!< SPI1       clock source */
    SPI2_CLK                 = 25U, /*!< SPI2       clock source */
    SPI3_CLK                 = 26U, /*!< SPI3       clock source */
    SPI4_CLK                 = 27U, /*!< SPI4       clock source */
    SPI5_CLK                 = 28U, /*!< SPI5       clock source */
    SPI6_CLK                 = 29U, /*!< SPI6       clock source */
    SPI7_CLK                 = 30U, /*!< SPI7       clock source */
    FlexCAN0_CLK             = 31U, /*!< FlexCAN0   clock source */
    FlexCAN1_CLK             = 32U, /*!< FlexCAN1   clock source */
    FlexCAN2_CLK             = 33U, /*!< FlexCAN2   clock source */
    FlexCAN3_CLK             = 34U, /*!< FlexCAN3   clock source */
    FlexCAN4_CLK             = 35U, /*!< FlexCAN4   clock source */
    FlexCAN5_CLK             = 36U, /*!< FlexCAN5   clock source */
    FlexCAN6_CLK             = 37U, /*!< FlexCAN6   clock source */
    FlexCAN7_CLK             = 38U, /*!< FlexCAN7   clock source */
    ADC0_CLK                 = 39U, /*!< ADC0       clock source */
    ADC1_CLK                 = 40U, /*!< ADC1       clock source */
    ACMP0_CLK                = 41U, /*!< ACMP0      clock source */
    ACMP1_CLK                = 42U, /*!< ACMP1      clock source */
    PTU0_CLK                 = 43U, /*!< PTU0       clock source */
    PTU1_CLK                 = 44U, /*!< PTU1       clock source */
    TMU_CLK                  = 45U, /*!< TMU        clock source */
    eTMR0_CLK                = 46U, /*!< eTMR0      clock source */
    eTMR1_CLK                = 47U, /*!< eTMR1      clock source */
    eTMR2_CLK                = 48U, /*!< eTMR2      clock source */
    eTMR3_CLK                = 49U, /*!< eTMR3      clock source */
    eTMR4_CLK                = 50U, /*!< eTMR4      clock source */
    eTMR5_CLK                = 51U, /*!< eTMR5      clock source */
    MPWM0_CLK                = 52U, /*!< MPWM0      clock source */
    MPWM1_CLK                = 53U, /*!< MPWM1      clock source */
    MPWM2_CLK                = 54U, /*!< MPWM2      clock source */
    TMR0_CLK                 = 55U, /*!< TMR0       clock source */
    TMR1_CLK                 = 56U, /*!< TMR1       clock source */
    SENT0_CLK                = 57U, /*!< SENT0      clock source */
    SENT1_CLK                = 58U, /*!< SENT1      clock source */
    pTMR0_CLK                = 59U, /*!< pTMR0      clock source */
    pTMR1_CLK                = 60U, /*!< pTMR1      clock source */
    pTMR2_CLK                = 61U, /*!< pTMR2      clock source */
    lpTMR0_CLK               = 62U, /*!< lpTMR0     clock source */
    RTC_CLK                  = 63U, /*!< RTC        clock source */
    WKU_CLK                  = 64U, /*!< WKU        clock source */
    CRC_CLK                  = 65U, /*!< CRC        clock source */
    TRNG_CLK                 = 66U, /*!< TRNG       clock source */
    HCU_CLK                  = 67U, /*!< HCU        clock source */
    QSPI_CLK                 = 68U, /*!< QSPI       clock source */
    WDG_CLK                  = 69U, /*!< WDG        clock source */
    FMU_CLK                  = 70U, /*!< FMU        clock source */
    INTM_CLK                 = 71U, /*!< INTM       clock source */
    EMU_CLK                  = 72U, /*!< EMU        clock source */
    SAI0_CLK                 = 73U, /*!< SAI0       clock source */
    SAI1_CLK                 = 74U, /*!< SAI1       clock source */
    STU_CLK                  = 75U, /*!< STU        clock source */
    ENET_CLK                 = 76U, /*!< ENET       clock source */
    CIM_CLK                  = 77U, /*!< CIM        clock source */
    SCU_CLK                  = 78U, /*!< SCU        clock source */
    PCU_CLK                  = 79U, /*!< PCU        clock source */
    RCU_CLK                  = 80U, /*!< RCU        clock source */

    IPC_PERI_END_OF_CLOCK    = 81U, /*!< IPC_PERI_END_OF_CLOCK */
    IPC_SIRC_CLK             = 82U, /*!< IPC_SIRC   clock source */
    IPC_FIRC_CLK             = 83U, /*!< IPC_FIRC   clock source */
    IPC_FXOSC_CLK            = 84U, /*!< IPC_FXOSC  clock source */
    IPC_SXOSC_CLK            = 85U, /*!< IPC_SXOSC  clock source */
    IPC_PLL_CLK              = 86U, /*!< IPC_PLL    clock source */

    IPC_END_OF_CLOCK         = 87U, /*!< IPC_END_OF_CLOCK */
    CORE_CLK                 = 88U, /*!< CORE       clock source */
    FAST_BUS_CLK             = 89U, /*!< FAST_BUS   clock source */
    SLOW_BUS_CLK             = 90U, /*!< SLOW_BUS   clock source */

    CLOCK_NAME_COUNT         = 91U, /*!< CLOCK_NAME_COUNT */
} clock_names_t;
/*! @brief IPC clock name mappings
 *  Mappings between clock names and peripheral clock control indexes.
 *  If there is no peripheral clock control index for a clock name,
 *  then the corresponding value is IPC_INVALID_INDEX.
 */
#define IPC_DMA_INDEX          (  8U)
#define IPC_TRACE_INDEX        ( 15U)
#define IPC_GPIO_INDEX         ( 17U)
#define IPC_PCTRLA_INDEX       ( 18U)
#define IPC_PCTRLB_INDEX       ( 19U)
#define IPC_PCTRLC_INDEX       ( 20U)
#define IPC_PCTRLD_INDEX       ( 21U)
#define IPC_PCTRLE_INDEX       ( 22U)
#define IPC_LINFlexD0_INDEX    ( 27U)
#define IPC_LINFlexD1_INDEX    ( 28U)
#define IPC_LINFlexD2_INDEX    ( 29U)
#define IPC_LINFlexD3_INDEX    ( 30U)
#define IPC_LINFlexD4_INDEX    ( 31U)
#define IPC_LINFlexD5_INDEX    ( 32U)
#define IPC_LINFlexD6_INDEX    ( 33U)
#define IPC_LINFlexD7_INDEX    ( 34U)
#define IPC_LINFlexD8_INDEX    ( 35U)
#define IPC_LINFlexD9_INDEX    ( 36U)
#define IPC_I2C0_INDEX         ( 37U)
#define IPC_I2C1_INDEX         ( 38U)
#define IPC_I2C2_INDEX         ( 39U)
#define IPC_I2C3_INDEX         ( 40U)
#define IPC_I2C4_INDEX         ( 41U)
#define IPC_SPI0_INDEX         ( 42U)
#define IPC_SPI1_INDEX         ( 43U)
#define IPC_SPI2_INDEX         ( 44U)
#define IPC_SPI3_INDEX         ( 45U)
#define IPC_SPI4_INDEX         ( 46U)
#define IPC_SPI5_INDEX         ( 47U)
#define IPC_SPI6_INDEX         ( 48U)
#define IPC_SPI7_INDEX         ( 49U)
#define IPC_FlexCAN0_INDEX     ( 52U)
#define IPC_FlexCAN1_INDEX     ( 53U)
#define IPC_FlexCAN2_INDEX     ( 54U)
#define IPC_FlexCAN3_INDEX     ( 55U)
#define IPC_FlexCAN4_INDEX     ( 56U)
#define IPC_FlexCAN5_INDEX     ( 57U)
#define IPC_FlexCAN6_INDEX     ( 58U)
#define IPC_FlexCAN7_INDEX     ( 59U)
#define IPC_ADC0_INDEX         ( 64U)
#define IPC_ADC1_INDEX         ( 65U)
#define IPC_ACMP0_INDEX        ( 67U)
#define IPC_ACMP1_INDEX        ( 68U)
#define IPC_PTU0_INDEX         ( 70U)
#define IPC_PTU1_INDEX         ( 71U)
#define IPC_TMU_INDEX          ( 72U)
#define IPC_eTMR0_INDEX        ( 73U)
#define IPC_eTMR1_INDEX        ( 74U)
#define IPC_eTMR2_INDEX        ( 75U)
#define IPC_eTMR3_INDEX        ( 76U)
#define IPC_eTMR4_INDEX        ( 77U)
#define IPC_eTMR5_INDEX        ( 78U)
#define IPC_MWPM0_INDEX        ( 81U)
#define IPC_MWPM1_INDEX        ( 82U)
#define IPC_MWPM2_INDEX        ( 83U)
#define IPC_TMR0_INDEX         ( 84U)
#define IPC_TMR1_INDEX         ( 85U)
#define IPC_SENT0_INDEX        ( 88U)
#define IPC_SENT1_INDEX        ( 89U)
#define IPC_pTMR0_INDEX        ( 90U)
#define IPC_pTMR1_INDEX        ( 91U)
#define IPC_pTMR2_INDEX        ( 92U)
#define IPC_lpTMR0_INDEX       ( 93U)
#define IPC_RTC_INDEX          ( 96U)
#define IPC_WKU_INDEX          ( 98U)
#define IPC_CRC_INDEX          (101U)
#define IPC_TRNG_INDEX         (102U)
#define IPC_HCU_INDEX          (103U)
#define IPC_QSPI_INDEX         (104U)
#define IPC_WDG_INDEX          (106U)
#define IPC_FMU_INDEX          (112U)
#define IPC_INTM_INDEX         (113U)
#define IPC_EMU_INDEX          (114U)
#define IPC_SAI0_INDEX         (115U)
#define IPC_SAI1_INDEX         (116U)
#define IPC_STU_INDEX          (119U)
#define IPC_ENET_INDEX         (121U)
#define IPC_CIM_INDEX          (123U)
#define IPC_SCU_INDEX          (124U)
#define IPC_PCU_INDEX          (126U)
#define IPC_RCU_INDEX          (127U)
#define IPC_IPC_PERI_END_OF_CLOCK_INDEX   (128U)
#define IPC_IPC_SIRC_INDEX     (129U)
#define IPC_IPC_FIRC_INDEX     (130U)
#define IPC_IPC_FXOSC_INDEX    (131U)
#define IPC_IPC_SXOSC_INDEX    (132U)
#define IPC_IPC_PLL_INDEX      (133U)
#define IPC_IPC_END_OF_CLOCK_INDEX   (134U)
#define IPC_CORE_INDEX         (135U)
#define IPC_FAST_BUS_INDEX     (136U)
#define IPC_SLOW_BUS_INDEX     (137U)
#define IPC_CLOCK_NAME_COUNT_INDEX   (138U)

#define  IPC_CLOCK_NAME_MAPPINGS \
{                                \
    IPC_DMA_INDEX,            /*!< DMA        clock source    8u */ \
    IPC_TRACE_INDEX,          /*!< TRACE      clock source   15u */ \
    IPC_GPIO_INDEX,           /*!< GPIO       clock source   17u */ \
    IPC_PCTRLA_INDEX,         /*!< PCTRLA     clock source   18u */ \
    IPC_PCTRLB_INDEX,         /*!< PCTRLB     clock source   19u */ \
    IPC_PCTRLC_INDEX,         /*!< PCTRLC     clock source   20u */ \
    IPC_PCTRLD_INDEX,         /*!< PCTRLD     clock source   21u */ \
    IPC_PCTRLE_INDEX,         /*!< PCTRLE     clock source   22u */ \
    IPC_LINFlexD0_INDEX,      /*!< LINFlexD0  clock source   27u */ \
    IPC_LINFlexD1_INDEX,      /*!< LINFlexD1  clock source   28u */ \
    IPC_LINFlexD2_INDEX,      /*!< LINFlexD2  clock source   29u */ \
    IPC_LINFlexD3_INDEX,      /*!< LINFlexD3  clock source   30u */ \
    IPC_LINFlexD4_INDEX,      /*!< LINFlexD4  clock source   31u */ \
    IPC_LINFlexD5_INDEX,      /*!< LINFlexD5  clock source   32u */ \
    IPC_LINFlexD6_INDEX,      /*!< LINFlexD6  clock source   33u */ \
    IPC_LINFlexD7_INDEX,      /*!< LINFlexD7  clock source   34u */ \
    IPC_LINFlexD8_INDEX,      /*!< LINFlexD8  clock source   35u */ \
    IPC_LINFlexD9_INDEX,      /*!< LINFlexD9  clock source   36u */ \
    IPC_I2C0_INDEX,           /*!< I2C0       clock source   37u */ \
    IPC_I2C1_INDEX,           /*!< I2C1       clock source   38u */ \
    IPC_I2C2_INDEX,           /*!< I2C2       clock source   39u */ \
    IPC_I2C3_INDEX,           /*!< I2C3       clock source   40u */ \
    IPC_I2C4_INDEX,           /*!< I2C4       clock source   41u */ \
    IPC_SPI0_INDEX,           /*!< SPI0       clock source   42u */ \
    IPC_SPI1_INDEX,           /*!< SPI1       clock source   43u */ \
    IPC_SPI2_INDEX,           /*!< SPI2       clock source   44u */ \
    IPC_SPI3_INDEX,           /*!< SPI3       clock source   45u */ \
    IPC_SPI4_INDEX,           /*!< SPI4       clock source   46u */ \
    IPC_SPI5_INDEX,           /*!< SPI5       clock source   47u */ \
    IPC_SPI6_INDEX,           /*!< SPI6       clock source   48u */ \
    IPC_SPI7_INDEX,           /*!< SPI7       clock source   49u */ \
    IPC_FlexCAN0_INDEX,       /*!< FlexCAN0   clock source   52u */ \
    IPC_FlexCAN1_INDEX,       /*!< FlexCAN1   clock source   53u */ \
    IPC_FlexCAN2_INDEX,       /*!< FlexCAN2   clock source   54u */ \
    IPC_FlexCAN3_INDEX,       /*!< FlexCAN3   clock source   55u */ \
    IPC_FlexCAN4_INDEX,       /*!< FlexCAN4   clock source   56u */ \
    IPC_FlexCAN5_INDEX,       /*!< FlexCAN5   clock source   57u */ \
    IPC_FlexCAN6_INDEX,       /*!< FlexCAN6   clock source   58u */ \
    IPC_FlexCAN7_INDEX,       /*!< FlexCAN7   clock source   59u */ \
    IPC_ADC0_INDEX,           /*!< ADC0       clock source   64u */ \
    IPC_ADC1_INDEX,           /*!< ADC1       clock source   65u */ \
    IPC_ACMP0_INDEX,          /*!< ACMP0      clock source   67u */ \
    IPC_ACMP1_INDEX,          /*!< ACMP1      clock source   68u */ \
    IPC_PTU0_INDEX,           /*!< PTU0       clock source   70u */ \
    IPC_PTU1_INDEX,           /*!< PTU1       clock source   71u */ \
    IPC_TMU_INDEX,            /*!< TMU        clock source   72u */ \
    IPC_eTMR0_INDEX,          /*!< eTMR0      clock source   73u */ \
    IPC_eTMR1_INDEX,          /*!< eTMR1      clock source   74u */ \
    IPC_eTMR2_INDEX,          /*!< eTMR2      clock source   75u */ \
    IPC_eTMR3_INDEX,          /*!< eTMR3      clock source   76u */ \
    IPC_eTMR4_INDEX,          /*!< eTMR4      clock source   77u */ \
    IPC_eTMR5_INDEX,          /*!< eTMR5      clock source   78u */ \
    IPC_MWPM0_INDEX,          /*!< MWPM0      clock source   81u */ \
    IPC_MWPM1_INDEX,          /*!< MWPM1      clock source   82u */ \
    IPC_MWPM2_INDEX,          /*!< MWPM2      clock source   83u */ \
    IPC_TMR0_INDEX,           /*!< TMR0       clock source   84u */ \
    IPC_TMR1_INDEX,           /*!< TMR1       clock source   85u */ \
    IPC_SENT0_INDEX,          /*!< SENT0      clock source   88u */ \
    IPC_SENT1_INDEX,          /*!< SENT1      clock source   89u */ \
    IPC_pTMR0_INDEX,          /*!< pTMR0      clock source   90u */ \
    IPC_pTMR1_INDEX,          /*!< pTMR1      clock source   91u */ \
    IPC_pTMR2_INDEX,          /*!< pTMR2      clock source   92u */ \
    IPC_lpTMR0_INDEX,         /*!< lpTMR0     clock source   93u */ \
    IPC_RTC_INDEX,            /*!< RTC        clock source   96u */ \
    IPC_WKU_INDEX,            /*!< WKU        clock source   98u */ \
    IPC_CRC_INDEX,            /*!< CRC        clock source  101u */ \
    IPC_TRNG_INDEX,           /*!< TRNG       clock source  102u */ \
    IPC_HCU_INDEX,            /*!< HCU        clock source  103u */ \
    IPC_QSPI_INDEX,           /*!< QSPI       clock source  104u */ \
    IPC_WDG_INDEX,            /*!< WDG        clock source  106u */ \
    IPC_FMU_INDEX,            /*!< FMU        clock source  112u */ \
    IPC_INTM_INDEX,           /*!< INTM       clock source  113u */ \
    IPC_EMU_INDEX,            /*!< EMU        clock source  114u */ \
    IPC_SAI0_INDEX,           /*!< SAI0       clock source  115u */ \
    IPC_SAI1_INDEX,           /*!< SAI1       clock source  116u */ \
    IPC_STU_INDEX,            /*!< STU        clock source  119u */ \
    IPC_ENET_INDEX,           /*!< ENET       clock source  121u */ \
    IPC_CIM_INDEX,            /*!< CIM        clock source  123u */ \
    IPC_SCU_INDEX,            /*!< SCU        clock source  124u */ \
    IPC_PCU_INDEX,            /*!< PCU        clock source  126u */ \
    IPC_RCU_INDEX,            /*!< RCU        clock source  127u */ \
    IPC_IPC_PERI_END_OF_CLOCK_INDEX,     /*!< IPC_PERI_END_OF_CLOCK clock source  128u */ \
    IPC_IPC_SIRC_INDEX,       /*!< IPC_SIRC   clock source  129u */ \
    IPC_IPC_FIRC_INDEX,       /*!< IPC_FIRC   clock source  130u */ \
    IPC_IPC_FXOSC_INDEX,      /*!< IPC_FXOSC  clock source  131u */ \
    IPC_IPC_SXOSC_INDEX,      /*!< IPC_SXOSC  clock source  132u */ \
    IPC_IPC_PLL_INDEX,        /*!< IPC_PLL    clock source  133u */ \
    IPC_IPC_END_OF_CLOCK_INDEX,     /*!< IPC_END_OF_CLOCK clock source  134u */ \
    IPC_CORE_INDEX,           /*!< CORE       clock source  135u */ \
    IPC_FAST_BUS_INDEX,       /*!< FAST_BUS   clock source  136u */ \
    IPC_SLOW_BUS_INDEX,       /*!< SLOW_BUS   clock source  137u */ \
}

/* DMAMUX module features */

/* @brief DMAMUX peripheral is available in silicon. */
#define FEATURE_DMAMUX_AVAILABLE
/* @brief Number of DMA channels. */
#define FEATURE_DMAMUX_CHANNELS (16U)
/* @brief Conversion from request source to the actual DMAMUX channel */
#define FEATURE_DMAMUX_REQ_SRC_TO_CH(x) (x)
/* @brief Mapping between request source and DMAMUX instance */
#define FEATURE_DMAMUX_REQ_SRC_TO_INSTANCE(x) (0U)
/* @brief Conversion from DMA channel index to DMAMUX channel. */
#define FEATURE_DMAMUX_DMA_CH_TO_CH(x) (x)
/* @brief Conversion from DMAMUX channel DMAMUX register index. */
#define FEATURE_DMAMUX_CHN_REG_INDEX(x) (x)

/* SPI module features */
/* @brief Initial value for state structure */
#define FEATURE_SPI_STATE_STRUCTURES_NULL {NULL, NULL, NULL, NULL, NULL, NULL}
/* @brief Has transmitter/receiver DMA enable bits. */
#define FEATURE_SPI_HAS_DMA_ENABLE (1)
/* @brief Clock indexes for SPI clock */
#define FEATURE_SPI_CLOCKS_NAMES {SPI0_CLK, SPI1_CLK, SPI2_CLK, SPI3_CLK, SPI4_CLK, SPI5_CLK, SPI6_CLK, SPI7_CLK}

/* ADC module features */

/*! @brief ADC feature flag for extended number of SC1 and R registers,
 * generically named 'alias registers' */
#define FEATURE_ADC_HAS_EXTRA_NUM_REGS                    (0)

#define NUMBER_OF_ALT_CLOCKS  ADC_CLK_ALT_1
/*! @brief ADC feature flag for defining number of external ADC channels.
 * If each ADC instance has different number of external channels, then
 * this define is set with the maximum value. */
#define FEATURE_ADC_MAX_NUM_EXT_CHANS                     (24)
#define ADC_CLOCKS                                        {ADC0_CLK, ADC1_CLK}

/*! @brief ADC default clock divider from RM */
#define ADC_DEFAULT_CLOCK_DIVIDER                         (0x02U)
/*! @brief ADC default Sample Time from RM */
#define ADC_DEFAULT_SAMPLE_TIME                           (0x0CU)
/*! @brief ADC default Start Time from RM */
#define ADC_DEFAULT_START_TIME                            (0x20U)
/* @brief Max of adc clock frequency */
#define ADC_CLOCK_FREQ_MAX_RUNTIME                        (32000000u)
/* @brief Min of adc clock frequency */
#define ADC_CLOCK_FREQ_MIN_RUNTIME                        (2000000u)

/* CAN module features */

/* @brief Frames available in Rx FIFO flag shift */
#define FEATURE_CAN_RXFIFO_FRAME_AVAILABLE  (5U)
/* @brief Rx FIFO warning flag shift */
#define FEATURE_CAN_RXFIFO_WARNING          (6U)
/* @brief Rx FIFO overflow flag shift */
#define FEATURE_CAN_RXFIFO_OVERFLOW         (7U)
/* @brief The list contains definitions of the FD feature support on all instances */
#define FEATURE_CAN_INSTANCES_HAS_FD        {true, true, true, true, true, true}
/* @brief Has Flexible Data Rate for CAN0 */
#define FEATURE_CAN0_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN1 */
#define FEATURE_CAN1_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN2 */
#define FEATURE_CAN2_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN3 */
#define FEATURE_CAN3_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN4 */
#define FEATURE_CAN4_HAS_FD                 (1)
/* @brief Has Flexible Data Rate for CAN5 */
#define FEATURE_CAN5_HAS_FD                 (1)

/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN0 */
#define FEATURE_CAN0_MAX_MB_NUM             (64U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN1 */
#define FEATURE_CAN1_MAX_MB_NUM             (64U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN2 */
#define FEATURE_CAN2_MAX_MB_NUM             (64U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN3 */
#define FEATURE_CAN3_MAX_MB_NUM             (32U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN4 */
#define FEATURE_CAN4_MAX_MB_NUM             (32U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for CAN5 */
#define FEATURE_CAN5_MAX_MB_NUM             (32U)

/* @brief Has PE clock source select (bit field CAN_CTRL1[CLKSRC]). */
#define FEATURE_CAN_HAS_PE_CLKSRC_SELECT    (1)
/* @brief Has DMA enable (bit field MCR[DMA]). */
#define FEATURE_CAN_HAS_DMA_ENABLE          (1)

/* @brief Maximum number of Message Buffers supported for payload size 8 for any of the CAN instances */
#define FEATURE_CAN_MAX_MB_NUM              (64U)
/* @brief Maximum number of Message Buffers supported for payload size 8 for any of the CAN instances */
#define FEATURE_CAN_MAX_MB_NUM_ARRAY        { FEATURE_CAN0_MAX_MB_NUM, \
                                              FEATURE_CAN1_MAX_MB_NUM, \
                                              FEATURE_CAN2_MAX_MB_NUM, \
                                              FEATURE_CAN3_MAX_MB_NUM, \
                                              FEATURE_CAN4_MAX_MB_NUM, \
                                              FEATURE_CAN5_MAX_MB_NUM }
/* @brief Has Pretending Networking mode */
#define FEATURE_CAN_HAS_PRETENDED_NETWORKING    (0)
/* @brief Has Stuff Bit Count Enable Bit */
#define FEATURE_CAN_HAS_STFCNTEN_ENABLE         (0)
/* @brief Has ISO CAN FD Enable Bit */
#define FEATURE_CAN_HAS_ISOCANFDEN_ENABLE       (1)
/* @brief Has Message Buffer Data Size Region 1 */
#define FEATURE_CAN_HAS_MBDSR1                  (1)
/* @brief Has Message Buffer Data Size Region 2 */
#define FEATURE_CAN_HAS_MBDSR2                  (0)
/* @brief DMA hardware requests for all FlexCAN instances */
#define FEATURE_CAN_DMA_REQUESTS              { DMA_REQ_FLEXCAN0, \
                                                DMA_REQ_FLEXCAN1, \
                                                DMA_REQ_FLEXCAN2, \
                                                DMA_REQ_FLEXCAN3, \
                                                DMA_REQ_FLEXCAN4, \
                                                DMA_REQ_FLEXCAN5 }

/* @brief Maximum number of Message Buffers IRQs */
#define FEATURE_CAN_MB_IRQS_MAX_COUNT       (4U)
/* @brief Message Buffers IRQs */
#define FEATURE_CAN_MB_IRQS                 { CAN_ORed_0_15_MB_IRQS, CAN_ORed_16_31_MB_IRQS,  CAN_ORed_32_47_MB_IRQS,  CAN_ORed_48_63_MB_IRQS }
/* @brief Has Wake Up Irq channels (CAN_Wake_Up_IRQS_CH_COUNT > 0u) */
#define FEATURE_CAN_HAS_WAKE_UP_IRQ         (1)
/* @brief Has Self Wake Up mode */
#define FEATURE_CAN_HAS_SELF_WAKE_UP        (1)
/* @brief Has Flexible Data Rate */
#define FEATURE_CAN_HAS_FD                  (1)
/* @brief Clock name for the PE oscillator clock source */
#define FEATURE_CAN_PE_OSC_CLK_NAME         IPC_FXOSC_CLK
/* @bried FlexCAN has Detection And Correction of Memory Errors */
#define FEATURE_CAN_HAS_MEM_ERR_DET         (1)


/* This this will set the timer source for osif that will be used for timeout */
#define CAN_SERVICE_TIMEOUT_TYPE    (OSIF_COUNTER_DUMMY)

/* @brief Has Supervisor Mode MCR[SUPV] */
#define FEATURE_CAN_HAS_SUPV    (1)
/* @brief Has FlexCAN Enhanced Rx FIFO mode */
#define FEATURE_CAN_HAS_ENHANCED_RX_FIFO    (STD_ON)
/* @brief Has FlexCAN expandable memory */
#define FEATURE_CAN_HAS_EXPANDABLE_MEMORY    (STD_OFF)
/* @brief Has FlexCAN High Resolution Timer for Time stamp CAN Message */
#define FEATURE_CAN_HAS_HR_TIMER    (STD_OFF)
/* @brief Has FlexCAN Enhanced Rx FIFO has interrupts available */
#define FEATURE_CAN_HAS_ENHANCED_RX_FIFO_INTERRUPT    (STD_OFF)
/* @brief Has FD Iso Option Mode  */
#define FEATURE_CAN_SWITCHINGISOMODE   (STD_ON)
/* @brief Has Protocol exception Mode */
#define FEATURE_CAN_PROTOCOLEXCEPTION  (STD_ON)
/* @brief Has Edge filter Feature */
#define FEATURE_CAN_EDGEFILTER         (STD_ON)
/* Defines if interface will enable any of interrupts */
#define CAN_MB_INTERRUPT_SUPPORT    (STD_ON)
/* Defines the No Of Message Buffers Partions Suppport MBDSR regions */
#define FEATURE_CAN_MBDSR_COUNT    (3U)
/* Enable Development Error Detection */
/* Trebuie adagudat un field in configuratorul de FlexCAN */
#define FLEXCAN_DEV_ERROR_DETECT    (STD_OFF)

/* [Added for YTM32B1M] */
/* @brief Has enhance rx fifo feture for CAN0 */
#define FEATURE_CAN0_HAS_ENHANCE_RX_FIFO (1)
/* @brief Has enhance rx fifo feture for CAN1 */
#define FEATURE_CAN1_HAS_ENHANCE_RX_FIFO (1)
/* @brief Has enhance rx fifo feture for CAN2 */
#define FEATURE_CAN2_HAS_ENHANCE_RX_FIFO (1)
/* @brief Has enhance rx fifo feture for CAN3 */
#define FEATURE_CAN3_HAS_ENHANCE_RX_FIFO (0)
/* @brief Has enhance rx fifo feture for CAN4 */
#define FEATURE_CAN4_HAS_ENHANCE_RX_FIFO (0)
/* @brief Has enhance rx fifo feture for CAN5 */
#define FEATURE_CAN5_HAS_ENHANCE_RX_FIFO (0)

/* @brief Has enhance rx fifo feture for all CAN */
#define FEATURE_CAN_HAS_ENHANCE_RX_FIFO  (1)

/* @brief The list contains definitions of the rx-fifo feature support on all instances */
#define FEATURE_CAN_INSTANCES_HAS_ENHANCE_RX_FIFO        { FEATURE_CAN0_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN1_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN2_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN3_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN4_HAS_ENHANCE_RX_FIFO, \
                                                           FEATURE_CAN5_HAS_ENHANCE_RX_FIFO }

#define FEATURE_CAN_ENHANCE_RX_FIFO_COUNT (32U)

/* pTMR module features */

/* @brief Clock names for pTMR */
#define pTMR_CLOCK_NAMES {SLOW_BUS_CLK}

/* @brief pTMR instance list */
typedef enum
{
    pTMR0_INST = 0,
} ptmr_instance_t;

/* @brief pTMR channels list for each instance */
typedef enum
{
    pTMR0_CH0 = 0U,
    pTMR0_CH1 = 1U,
    pTMR0_CH2 = 2U,
    pTMR0_CH3 = 3U,
} ptmr_channel_t;

/* TMR module features */
#define TMR_CLOCK_NAMES {SLOW_BUS_CLK}

/* lpTMR module features */

/* @brief lpTMR pulse counter input options */
#define FEATURE_LPTMR_HAS_INPUT_ALT1_SELECTION  (1U)
/* @brief Clock names for lpTMR. */
#define lpTMR_CLOCK_NAMES {lpTMR0_CLK }
/* @brief alt1 input for lpTMR. */
#define FEATURE_lpTMR_HAS_INPUT_ALT1_SELECTION  (1U)
/* @brief lpTMR ip clock selection */
#define FEATURE_lpTMR_HAS_IP_CLOCK_SELECTION    (1U)

/* RTC module features */

/*!
 * @brief RTC clock select
 * Implements : rtc_clk_source_t_Class
 */
typedef enum
{
    RTC_CLK_SRC_BUS = 0x00U, /*!< RTC Prescaler increments using bus clock */
    RTC_CLK_SRC_RTC_PIN = 0x01U,   /*!< RTC Prescaler increments from RTC pin */
    RTC_CLK_SRC_SIRC = 0x02U,   /*!< RTC Prescaler increments from SIRC clock */
    RTC_CLK_SRC_IPC = 0x03U,   /*!< RTC Prescaler increments from IPC function clock */
} rtc_clk_source_t;
/* @brief RTC has separate second irq */
#define FEATURE_RTC_HAS_SEPARATE_SECOND_IRQ     (1U)
/* @brief RTC has 32.768 OSC clock support */
#define FEATURE_RTC_HAS_OSC                     (1U)

/* LINFlexD module features */

/* @brief LINFlexD has dma support */
#define LINFlexD_CLOCK_NAMES                     {LINFlexD0_CLK, \
                                                  LINFlexD1_CLK, \
                                                  LINFlexD2_CLK, \
                                                  LINFlexD3_CLK, \
                                                  LINFlexD4_CLK, \
                                                  LINFlexD5_CLK, \
                                                  LINFlexD6_CLK, \
                                                  LINFlexD7_CLK, \
                                                  LINFlexD8_CLK, \
                                                  LINFlexD9_CLK  }
#define FEATURE_LINFlexD_ORED_INT_LINES          (1)
#define FEATURE_LINFlexD_HAS_DMA_ENABLED         (1)
/* @brief Address of the least significant byte in a 32-bit register (depends on endianess) */
#define FEATURE_LINFlexD_LSB_ADDR(reg)  ((uint32_t)(&(reg)) + 3U)
/* @brief Address of the least significant word in a 32-bit register (depends on endianess) */
#define FEATURE_LINFlexD_LSW_ADDR(reg)  ((uint32_t)(&(reg)) + 2U)
/* @brief LINFlexD "instance-implemented filters" mapping */
#define FEATURE_LINFlexD_INST_HAS_IFCR {true, true}
#define FEATURE_LINFlexD_TIMEOUT_LOOPS           (0xFFU)
/* @brief LINFlexD no receive edge detection support */
#define FEATURE_LINFlexD_HAS_RED_IRQ             (0)
/* @brief LINFlexD no different memory map */
#define FEATURE_LINFlexD_HAS_DIFFERENT_MEM_MAP   (0)
/* @brief LINFlexD no RX TX error INT lines */
#define FEATURE_LINFlexD_RX_TX_ERR_INT_LINES     (0)

/* SENT module features */

/* @brief SENT interrupt sources */
#define FEATURE_SENT_HAS_COMBINED_IRQ           (1)
#define FEATURE_SENT_IRQS                       {{SENT0_IRQn,SENT0_IRQn}}
/* @brief SENT clock sources */
#define FEATURE_SENT_CLOCK_NAMES                {SENT0_CLK}
#define FEATURE_SENT_FAST_DMA_REQS              {(uint8_t)DMA_REQ_SENT0_FAST}
#define FEATURE_SENT_SLOW_DMA_REQS              {(uint8_t)DMA_REQ_SENT0_SLOW}

/* HCU module features */
#define FEATURE_HCU_AES_ENGINE                           (1U)
#define FEATURE_HCU_SM4_ENGINE                           (1U)
#define FEATURE_HCU_SHA_ENGINE                           (1U)
#define FEATURE_HCU_ONE_LOOP_DATA_SIZE                   (16U)
#define FEATURE_HCU_ONE_LOOP_OUT_WATERMARK               (3U)
#define FEATURE_HCU_ONE_LOOP_INPUT_WATERMARK             (3U)
#define FEATURE_HCU_HAS_FIXED_DMA                        (1U)
#define FEATURE_HCU_HAS_SHA_AUTHORIZE                    (1U)

#define FEATURE_HCU_FIFO_SIZE_IN_WORDS                   (8U)

/* @brief EFM AES area base address */
#define FEATURE_EFM_AES_KEY_BASE_ADDR                    (0x10003000U)
/* @brief EFM AES key size (5 for 256bit, 4 for 128bit)*/
#define FEATURE_EFM_AES_KEY_SIZE                         (0x5U)
#define FEATURE_EFM_AES_KEY_COUNT_MAX                    (32U)
#define FEATURE_HCU_GET_KEY_ADDR(key_id)                 (FEATURE_EFM_AES_KEY_BASE_ADDR + ((key_id) << 5U))

/* WKU module features */
#define FEATURE_WKU_MAX_CHANNEL_COUNT                    (32U)
/* WKU module support pin isolation feature */
#define FEATURE_WKU_SUPPORT_PIN_ISOLATION                (0U)



/* OSIF module features */

#define FEATURE_OSIF_USE_SYSTICK                         (1)
#define FEATURE_OSIF_FREERTOS_ISR_CONTEXT_METHOD         (1) /* Cortex M device */


/* PTU module features */

/* @brief PTU has instance back to back mode between PTU0 CH0 and PTU1 CH0 pre-triggers */
#define FEATURE_PTU_HAS_INSTANCE_BACKTOBACK     (0)

/* @brief PTU has inter-channel back to back mode between PTUx CH0 and PTUx CH1 pre-triggers */
#define FEATURE_PTU_HAS_INTERCHANNEL_BACKTOBACK (0)


/* QSPI module features */

/* @brief Message Buffers IRQs */
#define FEATURE_QSPI_IRQS                 { QSPI_IRQn }

/*! @brief First address of the serial flash device on the AHB bus. */
#define FEATURE_QSPI_AMBA_BASE  0x68000000U
/*! @brief Last address of the serial flash device on the AHB bus. */
#define FEATURE_QSPI_AMBA_END   0x6FFFFFFFU

/*! @brief Size of AHB buffer. */
#define FEATURE_QSPI_AHB_BUF_SIZE   256U
/*! @brief Size of TX buffer. */
#define FEATURE_QSPI_TX_BUF_SIZE    128U

/*! @brief List of DMA Tx requests */
#define FEATURE_QSPI_DMA_TX_REQ   {DMA_REQ_QSPI_TX}
/*! @brief List of DMA Rx requests */
#define FEATURE_QSPI_DMA_RX_REQ   {DMA_REQ_QSPI_RX}


/*! @brief Implements Key to lock for all registers */
#define FEATURE_FMU_LOCK_KEY               (uint16_t)(0x46B0)
/*! @brief Implements Key to unlock for all registers */
#define FEATURE_FMU_UNLOCK_KEY             (uint16_t)(0xA765)
/*! @brief Implements Key to set operation code */
#define FEATURE_FMU_OP_KEY                 (uint16_t)(0xFA65)
/*! @brief Implements Key to select functional clock */
#define FEATURE_FMU_FUNC_CLKSEL_KEY        (uint8_t)(0x5A)
/*! @brief Implements Fault Status Key */
#define FEATURE_FMU_FAULT_ST_KEY           (uint32_t)(0x52CB6701U)
/*! @brief Implements Key to select clock source of FMU fail to react detect function */
#define FEATURE_FMU_DETECT_CLKSEL_KEY      (uint8_t)(0xA5)
/*! @brief Implements Key to assert fault injection */
#define FEATURE_FMU_FAULT_INJECTION_KEY    (uint8_t)(0x69)

#if defined(USER_DEVICE_FEATURES_OVERRIDE)
#include "user_device_features_override.h"
#endif


#endif /* YTM32B1MD0_FEATURES_H */

/*******************************************************************************
 * EOF
 ******************************************************************************/

