/*
* @file    Fls_Qspi_Types.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Fls_Qspi_Types
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.3.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef FLS_QSPI_TYPES_H
#define FLS_QSPI_TYPES_H

#include "Std_Types.h"

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define FLS_QSPI_TYPES_VENDOR_ID                       (180)
#define FLS_QSPI_TYPES_AR_REL_MAJOR_VERSION              (4)
#define FLS_QSPI_TYPES_AR_REL_MINOR_VERSION              (4)
#define FLS_QSPI_TYPES_AR_REL_REVISION_VERSION           (0)
#define FLS_QSPI_TYPES_SW_MAJOR_VERSION                  (2)
#define FLS_QSPI_TYPES_SW_MINOR_VERSION                  (3)
#define FLS_QSPI_TYPES_SW_PATCH_VERSION                  (0)
/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/

/*! @brief Number of AHB buffers in the device
 */
#define QSPI_AHB_BUFFERS   4U

/*! @brief Key to lock/unlock LUT
 */
#define QSPI_LUT_LOCK_KEY   0x5AF05AF0U

/*! Number of erase types that can be supported by a flash device */
#define  QSPI_ERASE_TYPES      4U

/*! Invalid sequence number in virtual LUT, used for unsupported features */
#define     QSPI_LUT_INVALID     (uint16)0xFFFFU

/*! End operation for a LUT sequence */
#define     QSPI_LUT_SEQ_END     (uint16)0x0U
/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                ENUMS AND STRUCTURES
==================================================================================================*/

/*! @brief Qspi return codes
 */
typedef enum
{
    STATUS_QSPI_SUCCESS              = 0x00U,   /*!< Successful job */
    STATUS_QSPI_ERROR                = 0x01U,   /*!< IP is performing an operation */
    STATUS_QSPI_BUSY                 = 0x02U,   /*!< Error - general code */
    STATUS_QSPI_TIMEOUT              = 0x03U,   /*!< Error - exceeded timeout */
    STATUS_QSPI_ERROR_PROGRAM_VERIFY = 0x04U,   /*!< Error - selected memory area doesn't contain desired value */
} Qspi_StatusType;

/*! @brief flash operation type
 */
typedef enum
{
    QSPI_OP_TYPE_CMD          = 0x00U,  /*!< Simple command                              */
    QSPI_OP_TYPE_WRITE_REG    = 0x01U,  /*!< Write value in external flash register      */
    QSPI_OP_TYPE_RMW_REG      = 0x02U,  /*!< RMW command on external flash register      */
    QSPI_OP_TYPE_READ_REG     = 0x03U,  /*!< Read external flash register until expected value is read    */
    QSPI_OP_TYPE_QSPI_CFG     = 0x04U,  /*!< Re-configure QSPI controller                */
} Qspi_OperationType;

/*! @brief Lut commands
 */
typedef enum
{
    QSPI_LUT_INSTR_STOP            = (uint16)((uint16)0x0000U << 10U),  /*!<  End of sequence                            */
    QSPI_LUT_INSTR_CMD             = (uint16)((uint16)0x0001U << 10U),  /*!<  Command                                    */
    QSPI_LUT_INSTR_ADDR            = (uint16)((uint16)0x0002U << 10U),  /*!<  Address                                    */
    QSPI_LUT_INSTR_DUMMY           = (uint16)((uint16)0x0003U << 10U),  /*!<  Dummy cycles                               */
    QSPI_LUT_INSTR_CADDR           = (uint16)((uint16)0x0004U << 10U),  /*!<  Column address                             */
    QSPI_LUT_INSTR_MODE            = (uint16)((uint16)0x0005U << 10U),  /*!<  8-bit mode                                 */
    QSPI_LUT_INSTR_MODE2           = (uint16)((uint16)0x0006U << 10U),  /*!<  2-bit mode                                 */
    QSPI_LUT_INSTR_MODE4           = (uint16)((uint16)0x0007U << 10U),  /*!<  4-bit mode                                 */
    QSPI_LUT_INSTR_READ            = (uint16)((uint16)0x0008U << 10U),  /*!<  Read data                                  */
    QSPI_LUT_INSTR_WRITE           = (uint16)((uint16)0x0009U << 10U),  /*!<  Write data                                 */
    QSPI_LUT_INSTR_DATA_TEST       = (uint16)((uint16)0x000AU << 10U),  /*!<  Read data for test                         */
    QSPI_LUT_INSTR_STEST           = (uint16)((uint16)0x000BU << 10U),  /*!<  Read out data loopback test                */
    QSPI_LUT_INSTR_JUMP            = (uint16)((uint16)0x000FU << 10U),  /*!<  Jump to the instruction pointed by operand */
    QSPI_LUT_INSTR_CMD_DDR         = (uint16)((uint16)0x0011U << 10U),  /*!<  Command - DDR mode                         */
    QSPI_LUT_INSTR_ADDR_DDR        = (uint16)((uint16)0x0012U << 10U),  /*!<  Address - DDR mode                         */
    QSPI_LUT_INSTR_CADDR_DDR       = (uint16)((uint16)0x0014U << 10U),  /*!<  Column address - DDR mode                  */
    QSPI_LUT_INSTR_MODE_DDR        = (uint16)((uint16)0x0015U << 10U),  /*!<  8-bit mode - DDR mode                      */
    QSPI_LUT_INSTR_MODE2_DDR       = (uint16)((uint16)0x0016U << 10U),  /*!<  2-bit mode - DDR mode                      */
    QSPI_LUT_INSTR_MODE4_DDR       = (uint16)((uint16)0x0017U << 10U),  /*!<  4-bit mode - DDR mode                      */
    QSPI_LUT_INSTR_READ_DDR        = (uint16)((uint16)0x0018U << 10U),  /*!<  Read data - DDR mode                       */
    QSPI_LUT_INSTR_WRITE_DDR       = (uint16)((uint16)0x0019U << 10U),  /*!<  Write data - DDR mode                      */
    QSPI_LUT_INSTR_DATA_TSET_DDR   = (uint16)((uint16)0x001AU << 10U),  /*!<  Read data for test - DDR mode              */
    QSPI_LUT_INSTR_STEST_DDR       = (uint16)((uint16)0x001BU << 10U),  /*!<  Read out data loopback test - DDR mode     */

}  Qspi_LutCommandsType;

/*! @brief Lut pad options
 */
typedef enum
{
    QSPI_LUT_PADS_1              = (uint16)((uint16)0x0000U << 8U),    /*!<  1 Pad      */
    QSPI_LUT_PADS_2              = (uint16)((uint16)0x0001U << 8U),    /*!<  2 Pads     */
    QSPI_LUT_PADS_4              = (uint16)((uint16)0x0002U << 8U),    /*!<  4 Pads     */
    QSPI_LUT_PADS_8              = (uint16)((uint16)0x0003U << 8U),    /*!<  8 Pads     */
} Qspi_LutPadsType;

/*! @brief Endianess options
 */
typedef enum
{
    QSPI_END_NO_SWAP         = 0U,  /*!< No swapping  */
    QSPI_END_32BIT_SWAP      = 1U,  /*!< Byte swapping in 32 bit  */
    QSPI_END_64BIT_SWAP      = 2U,  /*!< Word swapping in 64 bit  */
    QSPI_END_ALL_SWAP        = 3U,  /*!< Word swapping in 64 bit and byte swapping in 32 bit  */
} Qspi_EndianessType;

/*! @brief Clock phase used for sampling Rx data
 */
typedef enum
{
    QSPI_DATA_RATE_SDR         = 0U,    /*!<  Single data rate    */
    QSPI_DATA_RATE_DDR         = 1U,    /*!<  Double data rate    */
} Qspi_DataRateType;

/*! @brief Clock edge used for sampling Rx data
 */
typedef enum
{
    QSPI_RISING_EDGE_SAMPLE    = 0U,    /*!<  Sampling on rising edge  */
    QSPI_FALLING_EDGE_SAMPLE     = 1U,    /*!<  Sampling on falling edge */
} Qspi_SampleEdgeType;

/*! @brief Alignment of outgoing data with serial clock
 */
typedef enum
{
    QSPI_FLASH_DATA_ALIGN_REFCLK      = 0U,    /*!<  Data aligned with the posedge of Internal reference clock of QSPI  */
    QSPI_FLASH_DATA_ALIGN_4X_REFCLK   = 1U,    /*!<  Data aligned with 4x serial flash half clock                          */
} Qspi_FlashDataAlignType;

/*! @brief flash connection to the QSPI module
 */
typedef enum
{
    QSPI_SIDE_A    = 0x00U,  /*!< Serial flash connected on side A    */
    QSPI_SIDE_B    = 0x01U,  /*!< Serial flash connected on side B    */
} Qspi_ConnectionType;

/*!
 * @brief Operation in a LUT sequence.
 *
 * This type describes one basic operation inside a LUT sequence. Each operation contains:
 *  - instruction (6 bits)
 *  - number of PADs (2 bits)
 *  - operand (8 bits)
 * Qspi_LutCommandsType and Qspi_LutPadsType types should be used to form operations
 */
typedef uint16 Qspi_InstrOpType;

/*! @brief Init callout pointer type
*/
typedef Qspi_StatusType(*Qspi_InitCalloutPtrType)(uint32 Instance);

/*! @brief Reset callout pointer type
*/
typedef Qspi_StatusType(*Qspi_ResetCalloutPtrType)(uint32 Instance);

/*! @brief Error Check callout pointer type
*/
typedef Qspi_StatusType(*Qspi_ErrorCheckCalloutPtrType)(uint32 Instance);

/*! @brief Ecc Check callout pointer type
*/
typedef Qspi_StatusType(*Qspi_EccCheckCalloutPtrType)(uint32 Instance, uint32 StartAddress, uint32 DataLength);



/*!
 * @brief AHB configuration structure
 *
 * This structure is used to provide configuration parameters for AHB access
 * to the external flash
 */
typedef struct
{
    uint8 Masters[QSPI_AHB_BUFFERS];    /*!< List of AHB masters assigned to each buffer          */
    uint16 Sizes[QSPI_AHB_BUFFERS];     /*!< List of buffer sizes                                 */
    boolean AllMasters;                    /*!< Indicates that any master may access the last buffer */
    boolean HighPriority;                  /*!< Indicates that the first buffer has high priority    */
} Qspi_ControllerAhbConfigType;

/*!
 * @brief Driver configuration structure
 *
 * This structure is used to provide configuration parameters for the qspi driver
 * at initialization time.
 */
typedef struct
{
    Qspi_DataRateType DataRate;              /*!< Single/double data rate                                 */
    boolean DmaSupport;                         /*!< Enables DMA support in the driver                       */
    uint8 RxDMAChannel;                         /*!< Rx DMA channel number. Only used if dmaSupport is true  */
    uint8 TxDMAChannel;                         /*!< Tx DMA channel number. Only used if dmaSupport is true  */
    boolean DqsEnable;                          /*!< Enable DQS, Only used in HyperFlash/HyperRAM            */
    uint32 MemSizeA;                            /*!< Size of serial flash A                                  */
    uint32 MemSizeB;                            /*!< Size of serial flash B                                  */
    uint8 CsHoldTime;                           /*!< CS hold time, expressed in serial clock cycles          */
    uint8 CsSetupTime;                          /*!< CS setup time, expressed in serial clock cycles         */
    uint8 ColumnAddr;                           /*!< Width of the column address, 0 if not used              */
    boolean WordAddresable;                     /*!< True if serial flash is word addressable                */
    boolean LoopbackEnable;                     /*!< Enable loopback mode                                    */
    boolean DataTestModeEnable;                 /*!< Enable data test mode, sampling data with data test.    */
    uint8 SampleDelay;                          /*!< Delay (in clock cycles) used for sampling Rx data       */
    Qspi_SampleEdgeType SampleEdge;          /*!< Clock phase used for sampling Rx data                   */
    uint8 CoarseDelay;                          /*!< Loopback or DQS coarse delay configuration              */
    boolean DqsLatencyEnable;                   /*!< Enable DQS latency for reads (Hyperflash)               */
    Qspi_FlashDataAlignType DataAlign;       /*!< Alignment of output data sent to serial flash           */
    boolean DdrHalfEnable;                      /*!< Half cycle sample offset in DDR mode.                   */
    Qspi_EndianessType Endianess;            /*!< Endianess configuration                                 */
    uint8 Io2IdleValueA;                        /*!< (0 / 1) Logic level of IO[2] signal when not used on side A      */
    uint8 Io3IdleValueA;                        /*!< (0 / 1) Logic level of IO[3] signal when not used on side A      */
    uint8 Io2IdleValueB;                        /*!< (0 / 1) Logic level of IO[2] signal when not used on side B      */
    uint8 Io3IdleValueB;                        /*!< (0 / 1) Logic level of IO[3] signal when not used on side B      */
    boolean ClockStop;                          /*!< Enable clock output stop when RxFIFO full               */
    Qspi_ControllerAhbConfigType AhbConfig;  /*!< AHB buffers configuration                     */
} Qspi_ControllerConfigType;

/*!
* @brief Status register configuration structure
*
* This structure contains information about the status registers of the external flash
*/
typedef struct
{
    uint16 StatusRegInitReadLut;     /*!< Command used to read the status register during initialization                   */
    uint16 StatusRegReadLut;         /*!< Command used to read the status register                                         */
    uint16 StatusRegWriteLut;        /*!< Command used to write the status register                                        */
    uint16 WriteEnableSRLut;         /*!< Write enable command used before writing to status register                      */
    uint16 WriteEnableLut;           /*!< Write enable command used before write or erase operations                       */
    uint8 RegSize;                   /*!< Size in bytes of status register                                                 */
    uint8 BusyOffset;                /*!< Position of "busy" bit inside status register                                    */
    uint8 BusyValue;                 /*!< Value of "busy" bit which indicates that the device is busy; can be 0 or 1       */
    uint8 WriteEnableOffset;         /*!< Position of "write enable" bit inside status register                            */
    uint8 BlockProtectionOffset;     /*!< Offset of block protection bits inside status register                           */
    uint8 BlockProtectionWidth;      /*!< Width of block protection bitfield                                               */
    uint8 BlockProtectionValue;      /*!< Value of block protection bitfield, indicate the protected area                  */
} Qspi_StatusConfigType;


/*!
* @brief Describes one type of erase
*
* This structure contains information about one type of erase supported by the external flash
*/
typedef struct
{
    uint16 EraseLut;              /*!< Lut index for erase command */
    uint8 Size;                   /*!< Size of the erased area: 2 ^ size; e.g. 0x0C means 4 Kbytes */
} Qspi_EraseVarConfigType;


/*!
* @brief Erase capabilities configuration structure
*
* This structure contains information about the erase capabilities of the external flash
*/
typedef struct
{
    Qspi_EraseVarConfigType EraseTypes[QSPI_ERASE_TYPES];      /*!< Erase types supported by the device  */
    uint16 ChipEraseLut;                                             /*!< Lut index for chip erase command */
} Qspi_EraseConfigType;


/*!
* @brief Read Id capabilities configuration structure
*
* This structure contains information about the read manufacturer/device ID command
*/
typedef struct
{
    uint16 ReadIdLut;                               /*!< Read Id command                                  */
    uint16 ReadIdSize;                              /*!< Size of data returned by Read Id command         */
    uint32 ReadIdExpected;                          /*!< Device ID configured value (Memory density | Memory type | Manufacturer ID) */
} Qspi_ReadIdConfigType;


/*!
* @brief Suspend capabilities configuration structure
*
* This structure contains information about the Program / Erase Suspend capabilities of the external flash
*/
typedef struct
{
    uint16 EraseSuspendLut;       /*!< Lut index for the erase suspend operation   */
    uint16 EraseResumeLut;        /*!< Lut index for the erase resume operation    */
    uint16 ProgramSuspendLut;     /*!< Lut index for the program suspend operation */
    uint16 ProgramResumeLut;      /*!< Lut index for the program resume operation  */
} Qspi_SuspendConfigType;


/*!
 * @brief Soft Reset capabilities configuration structure
 *
 * This structure contains information about the Soft Reset capabilities of the external flash
 */
typedef struct
{
    uint16 ResetCmdLut;                   /*!< First command in reset sequence            */
    uint8 ResetCmdCount;                  /*!< Number of commands in reset sequence       */
} Qspi_ResetConfigType;


/*!
 * @brief List of LUT sequences.
 *
 * List of LUT sequences. Each sequence describes a command to the external flash. Sequences are separated by a 0 operation
 */
typedef struct
{
    uint16 OpCount;                    /*!< Number of operations in the LUT table    */
    const Qspi_InstrOpType *LutOps;        /*!< List of operations                       */
} Qspi_LutConfigType;


/*!
 * @brief Initialization operation
 *
 * This structure describes one initialization operation.
 */
typedef struct
{
    Qspi_OperationType OpType;         /*!< Operation type                                 */
    uint16 Command1Lut;            /*!< Index of first command sequence in Lut; for RMW type this is the read command               */
    uint16 Command2Lut;            /*!< Index of second command sequence in Lut, only used for RMW type, this is the write command  */
    uint16 WeLut;                  /*!< Index of write enable sequence in Lut, only used for Write and RMW type                     */
    uint32 Addr;                   /*!< Address, if used in command.                                                                */
    uint8 Size;                    /*!< Size in bytes of configuration register                                                     */
    uint8 Shift;                   /*!< Position of configuration field inside the register                                         */
    uint8 Width;                   /*!< Width in bits of configuration field.                                                       */
    uint32 Value;                  /*!< Value to set in the field                                                                   */
    const Qspi_ControllerConfigType *CtrlCfgPtr;   /*!< New controller configuration, valid only for QSPI_OP_TYPE_QSPI_CFG type     */
} Qspi_InitOperationType;


/*!
 * @brief Initialization sequence
 *
 * Describe sequence that will be performed only once during initialization to put the flash in the desired state for operation.
 * This may include, for example, setting the QE bit, activating 4-byte addressing, activating XPI mode
 */
typedef struct
{
    uint8 OpCount;                                     /*!< Number of operations  */
    const Qspi_InitOperationType *Operations;             /*!< List of operations    */
} Qspi_InitConfigType;



/*!
 * @brief Driver configuration structure
 *
 * This structure is used to provide configuration parameters for the external flash driver
 * at initialization time.
 */
typedef struct
{
    uint32 MemSize;                                       /*!< Memory size (in bytes)                           */
    uint32 PageSize;                                      /*!< Page size (in bytes)                             */
    uint16 ReadLut;                                       /*!< Command used to read data from flash             */
    uint16 WriteLut;                                      /*!< Command used to write data to flash              */
    uint16 Read0xxLut;                                    /*!< 0-x-x mode read command                          */
    uint16 Read0xxLutAHB;                                 /*!< 0-x-x mode AHB read command                      */
    Qspi_ReadIdConfigType ReadIdSettings;              /*!< Erase settings of the external flash             */
    Qspi_EraseConfigType EraseSettings;                /*!< Erase settings of the external flash             */
    Qspi_StatusConfigType StatusConfig;                /*!< Status register information                      */
    Qspi_SuspendConfigType SuspendSettings;            /*!< Program / Erase Suspend settings                 */
    Qspi_ResetConfigType ResetSettings;                /*!< Soft Reset settings, used at runtime             */
    Qspi_ResetConfigType InitResetSettings;            /*!< Soft Reset settings, used for first time reset   */
    Qspi_InitConfigType InitConfiguration;             /*!< Operations for initial flash configuration       */
    Qspi_LutConfigType LutSequences;                   /*!< List of LUT sequences describing flash commands  */
    Qspi_InitCalloutPtrType InitCallout;               /*!< Pointer to init callout                          */
    Qspi_ResetCalloutPtrType ResetCallout;             /*!< Pointer to reset callout                         */
    Qspi_ErrorCheckCalloutPtrType ErrorCheckCallout;   /*!< Pointer to error check callout                   */
    Qspi_EccCheckCalloutPtrType EccCheckCallout;       /*!< Pointer to ecc check callout                     */
    const Qspi_ControllerConfigType *CtrlAutoCfgPtr;   /*!< Initial controller configuration, if needed      */
} Qspi_MemoryConfigType;


/*!
 * @brief Flash-controller conections configuration structure
 *
 * This structure specifies thte connecctions of each flash device to QSPI controllers
 * at initialization time.
 */
typedef struct
{
    uint32 QspiInstance;                              /*!< QSPI Instance where this device is connected     */
    Qspi_ConnectionType ConnectionType;            /*!< Device connection to QSPI module                 */
    uint8  MemAlignment;                              /*!< Memory alignment required by the external flash  */
} Qspi_MemoryConnectionType;

/*!
 * @brief Driver configuration structure
 *
 * This structure is used to provide configuration parameters for the external flash driver
 * at initialization time.c
 */
typedef struct
{
    const Qspi_MemoryConfigType *Configuration;      /*!< Serial flash device configuration                */
    const Qspi_MemoryConnectionType *Connection;     /*!< Connection to a QSPI device                      */
    uint32 BaseAddress;                                 /*!< Base address of serial flash device              */
    uint32 LastCommand;                                 /*!< Last command sent to the flash device            */
    uint16 ActiveReadLut;                               /*!< LUT number for currently active read mode        */
} Qspi_StateType;
/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/
#ifdef __cplusplus
}
#endif

#endif /* End of file Fls_Qspi_Types.h */
