/*
* @file    Fls_Lld_Qspi.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Fls_Lld_Qspi
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.3.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef FLS_LLD_QSPI_H
#define FLS_LLD_QSPI_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "Fls_Qspi_Types.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define FLS_LLD_QSPI_VENDOR_ID                    (180)
#define FLS_LLD_QSPI_AR_REL_MAJOR_VER              (4)
#define FLS_LLD_QSPI_AR_REL_MINOR_VER              (4)
#define FLS_LLD_QSPI_AR_REL_REVISION_VER           (0)
#define FLS_LLD_QSPI_SW_MAJOR_VER                  (2)
#define FLS_LLD_QSPI_SW_MINOR_VER                  (3)
#define FLS_LLD_QSPI_SW_PATCH_VER                  (0)
/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
 /* Phisical LUT seq to use for all flash commands */
#define QSPI_COMMAND_LUT 0U
 /* Phisical LUT seq to use for AHB reads */
#define QSPI_AHB_LUT 1U

/*! Maximum number of bytes then can be read in one operation */
#define     QSPI_MAX_READ_SIZE       (FEATURE_QSPI_RX_BUF_SIZE)
/*! Maximum number of bytes then can be written in one operation */
#define     QSPI_MAX_WRITE_SIZE      (FEATURE_QSPI_TX_BUF_SIZE)
/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                ENUMS
==================================================================================================*/
/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/

/*!
 * @brief Initializes the serial flash memory driver
 *
 * This function initializes the external flash driver and prepares it for operation.
 *
 * @param Instance     External flash Instance number
 * @param ConfigPtr      Pointer to the driver configuration structure.
 * @param ConnectPtr     Pointer to the flash device connection structure.
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_Init(uint32 Instance,
                                const Qspi_MemoryConfigType * ConfigPtr,
                                const Qspi_MemoryConnectionType * ConnectPtr
                               );


/*!
 * @brief De-initializes the serial flash memory driver
 *
 * This function de-initializes the qspi driver. The driver can't be used
 * again until reinitialized. The state structure is no longer needed by the driver and
 * may be freed after calling this function.
 *
 * @param Instance     External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_Deinit(uint32 Instance);


/*!
 * @brief Erase a sector in the serial flash.
 *
 * This function performs one erase sector (block) operation on the external flash. The erase Size must match one of
 * the device's erase types.
 *
 * @param Instance     External flash Instance number
 * @param Address      Address of sector to be erased
 * @param Size         Size of the sector to be erase. The sector Size must match one of the supported erase sizes of the device.
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_EraseBlock(uint32 Instance,
                                      uint32 Address,
                                      uint32 Size
                                     );

/*!
 * @brief Erase the entire serial flash
 *
 * @param Instance     External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_EraseChip(uint32 Instance);

/*!
 * @brief Check the status of the flash device
 *
 * @param Instance     External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_GetMemoryStatus(uint32 Instance);

/*!
 * @brief Resets the flash device
 *
 * @param Instance     External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_Reset(uint32 Instance);

#if 0
/*!
 * @brief Suspends a program operation.
 *
 * @param Instance     External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_ProgramSuspend(uint32 Instance);


/*!
 * @brief Resumes a program operation.
 *
 * @param Instance     External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_ProgramResume(uint32 Instance);


/*!
 * @brief Suspends an erase operation.
 *
 * @param Instance     External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_EraseSuspend(uint32 Instance);


/*!
 * @brief Resumes an erase operation.
 *
 * @param Instance     External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_EraseResume(uint32 Instance);
#endif

/*!
 * @brief Read Data from serial flash
 *
 * @param Instance     External flash Instance number
 * @param Address      Start Address for read operation
 * @param Data         Buffer where to store read Data
 * @param Size         Size of Data buffer
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_Read(uint32 Instance,
                                uint32 Address,
                                uint8 * Data,
                                uint32 Size
                               );


/*!
 * @brief Read manufacturer ID/device ID from serial flash
 *
 * @param Instance     External flash Instance number
 * @param Data         Buffer where to store read Data. Buffer Size must match ReadId initialization settings.
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_ReadId(uint32 Instance,
                                  uint8 * Data
                                 );


/*!
 * @brief Verifies the correctness of the programmed Data
 *
 * @param Instance     External flash Instance number
 * @param Address      Start Address of area to be verified
 * @param Data         Data to be verified
 * @param Size         Size of area to be verified
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_ProgramVerify(uint32 Instance,
                                         uint32 Address,
                                         const uint8 * Data,
                                         uint32 Size
                                        );


/*!
 * @brief Checks whether or not an area in the serial flash is erased
 *
 * @param Instance     External flash Instance number
 * @param Address      Start Address of area to be verified
 * @param Size         Size of area to be verified
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_EraseVerify(uint32 Instance,
                                       uint32 Address,
                                       uint32 Size
                                      );

/*!
 * @brief Writes Data in serial flash
 *
 * @param Instance     External flash Instance number
 * @param Address      Start Address of area to be programmed
 * @param Data         Data to be programmed in flash
 * @param Size         Size of Data buffer
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_Program(uint32 Instance,
                                   uint32 Address,
                                   const uint8 * Data,
                                   uint32 Size
                                  );
/*!
 * @brief Sets up AHB reads to the serial flash
 *
 * @param Instance   External flash Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_AhbReadEnable(uint32 Instance);


/*!
 * @brief Check the status of the QSPI controller
 *
 * @param Instance     QSPI peripheral Instance number
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_ControllerGetStatus(uint32 Instance);


/*!
 * @brief Initializes the qspi driver
 *
 * This function initializes the qspi driver and prepares it for operation.
 *
 * @param Instance         QSPI peripheral Instance number
 * @param UserConfigPtr    Pointer to the qspi configuration structure.
 * @return    Error or success status returned by API
 */
Qspi_StatusType Qspi_Lld_ControllerInit(uint32 Instance,
                                          const Qspi_ControllerConfigType * UserConfigPtr
                                         );

#ifdef __cplusplus
}
#endif

#endif /* End of file Fls_Lld_Qspi.h */
/** @} */
