/*
 * Copyright 2020-2022 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may
 * only be used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

#ifndef PCRC_HW_ACCESS_H
#define PCRC_HW_ACCESS_H

/*! @file pcrc_hw_access.h */

#include "pcrc_driver.h"

/*!

 * @brief Cyclic Redundancy Check Hardware Access.
 *
 * This section describes the programming interface of the PCRC HW ACCESS.
 * @{
 */

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @name PCRC HW ACCESS API
 * @{
 */

#if defined(__cplusplus)
extern "C" {
#endif

/*!
 * @brief Gets the current PCRC result
 *
 * This function gets the current PCRC result from the data register
 *
 * @param[in] base The PCRC peripheral base address
 * @return Returns the current PCRC result
 */
static inline uint32_t PCRC_GetResultReg(const PCRC_Type *const base)
{
    return base->RESULT;
}

/*!
 * @brief Gets the current PCRC seed
 *
 * This function gets the current PCRC seed from the init register
 *
 * @param[in] base The PCRC peripheral base address
 * @return Returns the current PCRC seed
 */
static inline uint32_t PCRC_GetSeedReg(const PCRC_Type *const base)
{
    return base->INIT;
}

/*!
 * @brief Sets the 32 bits of PCRC data register
 *
 * This function sets the 32 bits of PCRC data register
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] value New data for PCRC computation
 */
static inline void PCRC_SetDataReg(PCRC_Type *const base,
                                  uint32_t value)
{
    base->DATA.IN32 = value;
}

/*!
 * @brief Sets the 16 bits of PCRC data register
 *
 * This function sets the 16 bits of PCRC data register
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] value New data for PCRC computation
 */
static inline void PCRC_SetDataReg16(PCRC_Type *const base,
                                    uint16_t value)
{
    base->DATA.IN16 = value;
}

/*!
 * @brief Sets the 8 bits of PCRC data register
 *
 * This function sets the 8 bits of PCRC data register
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] value New data for PCRC computation
 */
static inline void PCRC_SetDataReg8(PCRC_Type *const base,
                                   uint8_t value)
{
    base->DATA.IN8 = value;
}

/*!
 * @brief Sets the init seed register
 *
 * This function sets the seed register
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] value Seed value
 */
static inline void PCRC_SetSeedReg(PCRC_Type *const base,
                                  uint32_t value)
{
    base->INIT = value;
}

/*!
 * @brief Sets the PCRC polynomials
 *
 * This function sets the polynomial register
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] value Polynomial value
 */
static inline void PCRC_SetPolyReg(PCRC_Type *const base,
                                  uint32_t poly)
{
    base->POLY = poly;
}

/*!
 * @brief Gets the PCRC polynomials
 *
 * This function Gets the polynomial register
 *
 * @param[in] base The PCRC peripheral base address
 * @return    Polynomial value
 */
static inline uint32_t PCRC_GetPolyReg(const PCRC_Type *const base)
{
    return base->POLY;
}

/*!
 * @brief Gets complement read of PCRC data register
 *
 * This function gets complement read of PCRC data register.
 * Some PCRC protocols require the final checksum to be XORed with 0xFFFFFFFF
 * or 0xFFFF. Complement mode enables "on the fly" complementing of read data
 *
 * @param[in] base The PCRC peripheral base address
 * @return Complement read
 *         -true: Invert or complement the read value of the PCRC Data register.
 *         -false: No XOR on reading.
 */
static inline bool PCRC_GetFXorMode(const PCRC_Type *const base)
{
    return ((base->CTRL & PCRC_CTRL_INV_OUT_MASK) >> PCRC_CTRL_INV_OUT_SHIFT) != 0U;
}

/*!
 * @brief Sets complement read of PCRC data register
 *
 * This function sets complement read of PCRC data register.
 * Some PCRC protocols require the final checksum to be XORed with 0xFFFFFFFF
 * or 0xFFFF. Complement mode enables "on the fly" complementing of read data
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] enable Enable or disable complementing of read data
 */
static inline void PCRC_SetFXorMode(PCRC_Type *const base,
                                   bool enable)
{
    uint32_t ctrlTemp = base->CTRL;

    ctrlTemp &= ~(PCRC_CTRL_INV_OUT_MASK);
    ctrlTemp |= PCRC_CTRL_INV_OUT(enable ? 1UL : 0UL);
    base->CTRL = ctrlTemp;
}

/*!
 * @brief Gets the PCRC protocol width
 *
 * This function gets the PCRC protocol width
 *
 * @param[in] base The PCRC peripheral base address
 * @return PCRC protocol width
 *         - PCRC_BITS_16: 16-bit PCRC protocol.
 *         - PCRC_BITS_32: 32-bit PCRC protocol.
 */
static inline uint32_t PCRC_GetProtocolWidth(const PCRC_Type *const base)
{
    uint32_t retVal = 0U;
    retVal = (base->CTRL & PCRC_CTRL_MODE_MASK) >> PCRC_CTRL_MODE_SHIFT;
    return (retVal + 1);
}

/*!
 * @brief Sets the PCRC protocol width
 *
 * This function sets the PCRC protocol width
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] width The PCRC protocol width
 *            - PCRC_BITS_16: 16-bit PCRC protocol.
 *            - PCRC_BITS_32: 32-bit PCRC protocol.
 */
static inline void PCRC_SetProtocolWidth(PCRC_Type *const base,
                                        uint32_t width)
{
    uint32_t ctrlTemp = base->CTRL;

    ctrlTemp &= ~(PCRC_CTRL_MODE_MASK);
    ctrlTemp |= PCRC_CTRL_MODE(width - 1);
    base->CTRL = ctrlTemp;
}

/*!
 * @brief Gets the PCRC transpose type for writes
 *
 * This function gets the PCRC transpose type for writes
 *
 * @param[in] base The PCRC peripheral base address
 * @return PCRC input transpose type for writes
 */
static inline pcrc_transpose_t PCRC_GetWriteTranspose(const PCRC_Type *const base)
{
    pcrc_transpose_t type;
    type = (pcrc_transpose_t) ((base->CTRL & PCRC_CTRL_SWAP_IN_MASK) >> PCRC_CTRL_SWAP_IN_SHIFT);
    return type;
}

/*!
 * @brief Sets the PCRC transpose type for writes
 *
 * This function sets the PCRC transpose type for writes
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] transp The PCRC input transpose type
 */
static inline void PCRC_SetWriteTranspose(PCRC_Type *const base,
                                         pcrc_transpose_t transp)
{
    uint32_t ctrlTemp = base->CTRL;

    ctrlTemp &= ~(PCRC_CTRL_SWAP_IN_MASK);
    ctrlTemp |= PCRC_CTRL_SWAP_IN(transp);
    base->CTRL = ctrlTemp;
}

/*!
 * @brief Gets the PCRC transpose type for reads
 *
 * This function gets the PCRC transpose type for reads
 *
 * @param[in] base The PCRC peripheral base address
 * @return PCRC output transpose type
 */
static inline pcrc_transpose_t PCRC_GetReadTranspose(const PCRC_Type *const base)
{
    pcrc_transpose_t type;
    type = (pcrc_transpose_t) ((base->CTRL & PCRC_CTRL_SWAP_OUT_MASK) >> PCRC_CTRL_SWAP_OUT_SHIFT);
    return type;
}

/*!
 * @brief Sets the PCRC transpose type for reads
 *
 * This function sets the PCRC transpose type for reads
 *
 * @param[in] base The PCRC peripheral base address
 * @param[in] transp The PCRC output transpose type
 */
static inline void PCRC_SetReadTranspose(PCRC_Type *const base,
                                        pcrc_transpose_t transp)
{
    uint32_t ctrlTemp = base->CTRL;

    ctrlTemp &= ~(PCRC_CTRL_SWAP_OUT_MASK);
    ctrlTemp |= PCRC_CTRL_SWAP_OUT(transp);
    base->CTRL = ctrlTemp;
}

/*! @} */

#if defined(__cplusplus)
}
#endif

/*! @} */

#endif /* PCRC_HW_ACCESS_H */
/*******************************************************************************
 * EOF
 ******************************************************************************/
